package io.dyte.core.controllers

import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_ALL
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_PEER
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.PIN_PARTICIPANT
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class HostController(controllerContainer: IControllerContainer) : IHostController,
  BaseController(controllerContainer) {
  override suspend fun kick(id: String) {
    controllerContainer.loggerController.traceLog("DyteParticipants.kickPeer")
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_kick")
      throw UnsupportedOperationException("not allowed to kick participant")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(id)
    controllerContainer.socketController.sendMessage(KICK_PEER, JsonObject(content))
  }

  // TODO : is there any differance between kick single and kick all ?
  override suspend fun kickAll() {
    controllerContainer.loggerController.traceLog("DyteParticipants.kickAll")
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_kick_all")
      throw UnsupportedOperationException("not allowed to kick all")
    }
    controllerContainer.socketController.sendMessage(KICK_ALL, null)
  }

  // TODO : is there any differance between mute single and mute all ?
  override suspend fun muteAllAudio(allowUnmute: Boolean) {
    controllerContainer.loggerController.traceLog("DyteParticipants.disableAllAudio")
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_disable_all_audio")
      throw UnsupportedOperationException("not allowed to mute all audio")
    }
    val content = HashMap<String, JsonElement>()
    content["allowUnMute"] = JsonPrimitive(allowUnmute)
    controllerContainer.socketController.sendMessage(MUTE_ALL_AUDIO, JsonObject(content))
  }

  override suspend fun muteAllVideo() {
    controllerContainer.loggerController.traceLog("DyteParticipants.disableAllVideo")
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      controllerContainer.loggerController.traceError("DyteParticipants::unauthorized_disable_all_video")
      throw UnsupportedOperationException("not allowed to mute all video")
    }
    controllerContainer.socketController.sendMessage(MUTE_ALL_VIDEO, null)
  }

  override suspend fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant) {
    controllerContainer.loggerController.traceLog("DyteParticipants.disablePeerAudio")
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_disable_audio")
      throw UnsupportedOperationException("cant disable participant audio")
    }
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessage(MUTE_PEER_AUDIO, JsonObject(content))
  }

  override suspend fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant) {
    controllerContainer.loggerController.traceLog("DyteParticipants.disablePeerVideo")
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_disable_video")
      throw UnsupportedOperationException("cant produce video")
    }
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessage(MUTE_PEER_VIDEO, JsonObject(content))
  }

  override suspend fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant) {
    controllerContainer.loggerController.traceLog("DyteParticipants.pinPeer")
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_pin_peer")
      throw UnsupportedOperationException("not allowed to pin participant")
    }

    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    controllerContainer.socketController.sendMessage(PIN_PARTICIPANT, JsonObject(content))
  }

  override suspend fun unpinParticipant(){
    controllerContainer.loggerController.traceLog("DyteParticipants.unpinPeer")
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      controllerContainer.loggerController.traceError("DyteParticipant::unauthorized_unpin_peer")
      throw UnsupportedOperationException("not allowed to un-pin participant")
    }
    val content = HashMap<String, JsonElement>()
    val nullString: String? = null
    content["id"] = JsonPrimitive(nullString)
    controllerContainer.socketController.sendMessage(PIN_PARTICIPANT, JsonObject(content))
  }
}

interface IHostController {
  suspend fun muteAllAudio(allowUnmute: Boolean)
  suspend fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant)
  suspend fun muteAllVideo()
  suspend fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant)
  suspend fun kickAll()
  suspend fun kick(id: String)
  suspend fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant)
  suspend fun unpinParticipant()
}
