package io.dyte.core.platform

import android.app.Activity
import android.app.Application
import android.os.Bundle
import io.dyte.core.cameramanager.DyteCameraManager
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.incallmanager.InCallManagerEventCallbacks
import io.dyte.core.incallmanager.InCallManagerModule
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.EARPIECE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.NONE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.SPEAKER_PHONE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.WIRED_HEADSET
import io.dyte.core.media.DyteCommonMediaUtils
import io.dyte.core.models.AudioDeviceType.BLUETOOTH
import io.dyte.core.models.AudioDeviceType.EAR_PIECE
import io.dyte.core.models.AudioDeviceType.SPEAKER
import io.dyte.core.models.AudioDeviceType.UNKNOWN
import io.dyte.core.models.AudioDeviceType.WIRED
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.models.VideoDeviceType.FRONT

internal class DyteAndroidMedia(
  private val utilsProvider: IDytePlatformUtilsProvider
) : DyteCommonMediaUtils(), IDyteMediaUtils, InCallManagerEventCallbacks {

  private lateinit var inCallManagerModule: InCallManagerModule
  private val audioDevices = arrayListOf<DyteAudioDevice>()

  private lateinit var dyteCameraManager: DyteCameraManager
  private val videoDevices = arrayListOf<DyteVideoDevice>()
  private var selectedVideoDevice: DyteVideoDevice? = null

  override fun init() {
    dyteCameraManager = DyteCameraManager(
      utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application
    )
    videoDevices.clear()
    videoDevices.addAll(dyteCameraManager.getCameraDevices())

    utilsProvider.getControllerContainer().loggerController.traceLog("LocalMediaHandler::video_devices::${videoDevices.map { it.type }}")
    selectedVideoDevice = videoDevices.find { it.type == FRONT }

    inCallManagerModule =
      InCallManagerModule(utilsProvider.getPlatformUtils().getActivity() as Activity, this)
    inCallManagerModule.start("audio", true, "")

    loadAudioDevices()
    utilsProvider.getControllerContainer().loggerController.traceLog("LocalMediaHandler::audio_devices::${audioDevices.map { it.type }}")
    inCallManagerModule.setDefaultAudioDevice(SPEAKER_PHONE)
  }

  override fun routeAudioFromSpeakerphone() {
    // This is needed only on iOS
  }

  private fun loadAudioDevices() {
    val devices = inCallManagerModule.getAudioDevices()
    audioDevices.clear()
    devices.forEach { audioDevice ->
      when (audioDevice) {
        SPEAKER_PHONE -> {
          audioDevices.add(DyteAudioDevice("speaker", SPEAKER))
        }
        WIRED_HEADSET -> {
          audioDevices.add(DyteAudioDevice("wired", WIRED))
        }
        EARPIECE -> {
          audioDevices.add(DyteAudioDevice("earpiece", EAR_PIECE))
        }
        AudioDevice.BLUETOOTH -> {
          audioDevices.add(DyteAudioDevice("bluetooth", BLUETOOTH))
        }
        NONE -> {
          audioDevices.add(DyteAudioDevice("unknown", UNKNOWN))
        }
      }
    }
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return audioDevices
  }

  @Suppress("DEPRECATION")
  override fun setAudioDevice(device: DyteAudioDevice) {
    utilsProvider.getControllerContainer().loggerController.traceLog("LocalMediaHandler::setAudioDevice::${device.type}")
    when (device.type) {
      WIRED -> {
        inCallManagerModule.selectAudioDevice(WIRED_HEADSET)
      }
      SPEAKER -> {
        inCallManagerModule.selectAudioDevice(SPEAKER_PHONE)
      }
      BLUETOOTH -> {
        inCallManagerModule.selectAudioDevice(AudioDevice.BLUETOOTH)
      }
      EAR_PIECE -> {
        inCallManagerModule.selectAudioDevice(EARPIECE)
      }
      UNKNOWN -> {
      }
    }
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    var selectedDevice = DyteAudioDevice("unknowm", UNKNOWN)
    val selectedAudioDevice = inCallManagerModule.selectedAudioDevice
    selectedAudioDevice?.let { device ->
      audioDevices.find { it.type.deviceType == device.name }?.let {
        selectedDevice = it
      }
    }
    return selectedDevice
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return videoDevices
  }

  override fun setVideoDevice(device: DyteVideoDevice) {
    utilsProvider.getControllerContainer().loggerController.traceLog("LocalMediaHandler::setVideoDevice::${device.type}")
    if (selectedVideoDevice!=device) {
      utilsProvider.getControllerContainer().sfuUtils.switchCamera()
      selectedVideoDevice = device
    }
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return selectedVideoDevice ?: videoDevices.first()
  }

  override fun onEvent(eventName: String?, args: Any?) {
    eventName?.let { event ->
      when (event) {
        "WiredHeadset" -> {
        }

        "NoisyAudio" -> {
        }

        "MediaButton" -> {
        }

        "Proximity" -> {
          val data = args as Bundle?
          var isNear = false
          data?.let {
            isNear = data.getBoolean("isNear")
          }
          /*utilsProvider.getControllerContainer().eventController.triggerEvent(
            DyteEventType.OnProximityChanged(
              isNear
            )
          )*/
        }

        "onAudioFocusChange" -> {
        }

        "onAudioDeviceChanged" -> {
          utilsProvider.getControllerContainer().loggerController.traceLog("LocalMediaHandler::onAudioDeviceChanged")
          loadAudioDevices()
          utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
        }
        else -> {}
      }
    }
  }
}