package io.dyte.core.network

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.info.MeetingSessionInfo
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.network.models.ActiveLiveStreamResponse
import io.dyte.core.network.models.GraphQlRequest
import io.dyte.core.network.models.GraphQlRequestVariables
import io.dyte.core.network.models.IceServersWrapper
import io.dyte.core.network.models.IpDetailsResponsePayload
import io.dyte.core.network.models.MeetingSessionDataResponseV1
import io.dyte.core.network.models.MeetingSessionDataWrapper
import io.dyte.core.network.models.MultiplePluginResponse
import io.dyte.core.network.models.PluginAuthResponse
import io.dyte.core.network.models.PluginAuthorizationModel
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.network.models.PluginResponse
import io.dyte.core.network.models.RecordingResponseWrapper
import io.dyte.core.network.models.RecordingResponseWrapperV2
import io.dyte.core.network.models.StartLiveStreamResponse
import io.dyte.core.network.models.StartRecordingModel
import io.dyte.core.network.models.StopLiveStreamResponse
import io.dyte.core.network.models.StopRecordingModel
import io.dyte.core.network.models.StopRecordingModelV2
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.UserPresetRequestModel
import io.dyte.core.observability.PostDyteLogs
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.logging.LogLevel.ALL
import io.ktor.client.plugins.logging.LogLevel.NONE
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.plugins.logging.SIMPLE
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.encodeToJsonElement

// TODO : get rid of auth headers from all requests
private const val VERSION = "0.5.0"
private const val CLIENT_TYPE = "CLIENT_APP"
private const val GRAPHQL_REQUEST_QUERY: String =
  "\nquery Session(" + "$" + "roomName: String!, " + "$" + "password: String) {\n\tsession(roomName: " + "$" + "roomName, password: " + "$" + "password) {\n\t  title,\n\t  roomNodeLink,\n\t  roomName,\n\t  password\n\t}\n  }\n"

internal class ApiClient(
  private val controllerContainer: IControllerContainer
) : IApiClient {

  internal val client: HttpClient = HttpClient {
    install(Logging) {
      logger = Logger.SIMPLE
      level =
        if (controllerContainer.platformUtilsProvider.getLogger().isLoggerEnabled()) ALL else NONE
    }
    install(ContentNegotiation) {
      json(Json {
        prettyPrint = true
        isLenient = true
        ignoreUnknownKeys = true
      })
    }
    install(HttpTimeout) {
      socketTimeoutMillis = 30000
      requestTimeoutMillis = 30000
      connectTimeoutMillis = 30000
    }
    defaultRequest {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      header("dyte-tracing-id", controllerContainer.metaController.getPeerId())
      contentType(ContentType.Application.Json)
    }
  }

  val json = Json {
    ignoreUnknownKeys = true
  }

  override suspend fun getRoomNodeData(): MeetingSessionInfo {
    val roomName = controllerContainer.metaController.getRoomName()
    val ipInformation = getIpDetails(controllerContainer.metaController.getPeerId())
    return if (controllerContainer.metaController.isV2AuthToken()) {
      val data =
        client.post("${controllerContainer.metaController.getBaseUrl()}/internals/rooms") {
          val ipInformationPayload = buildJsonObject {
            put("ip_information", BaseApiService.json.encodeToJsonElement(ipInformation))
          }
          setBody(ipInformationPayload)
          contentType(ContentType.Application.Json)
        }.body<MeetingSessionDataWrapper>()
      MeetingSessionInfo(requireNotNull(data.data.title), data.data.roomNodeLink, data.data.useHive)
    } else {
      val graphQlRequestVariables = GraphQlRequestVariables(
        roomName,
        "",
        BaseApiService.json.encodeToString(IpDetailsResponsePayload.serializer(), ipInformation)
      )
      // val requestData = GraphQlRequest(GRAPHQL_REQUEST_QUERY, graphQlRequestVariables)
      val requestData = GraphQlRequest(GRAPHQL_ROOM_NODE_DATA_QUERY, graphQlRequestVariables)
      val response = client.post("${controllerContainer.metaController.getBaseUrlWithoutVersion()}/graphql") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(requestData)
        contentType(ContentType.Application.Json)
      }.body<MeetingSessionDataResponseV1>()
      MeetingSessionInfo(
        requireNotNull(response.data.session.title),
        requireNotNull(response.data.session.roomNodeLink)
      )
    }
  }

  // TODO : fix this
  override suspend fun getICEServers(): IceServersWrapper {
    val response = client.get("${controllerContainer.metaController.getBaseUrlWithoutVersion()}/iceservers") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
    }
    return response.body()
  }

  override suspend fun getParticipantInfo(): ParticipantInfo {
    return if (controllerContainer.metaController.isV2AuthToken()) {
      val userData =
        client.get("${controllerContainer.metaController.getBaseUrl()}/internals/participant-details") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        }.body<UserDataWrapper>()
      ParticipantInfo.getParticipantInfoFromV2(userData)
    } else {
      val userDataWrapper =
        client.get("${controllerContainer.metaController.getBaseUrlWithoutVersion()}/auth/basicUserDetails?authToken=${controllerContainer.metaController.getAuthToken()}")
          .body<UserDataWrapperV1>()
      val userPresetDataWrapper = getUserPreset()
      ParticipantInfo.getParticipantInfoFromV1(userDataWrapper, userPresetDataWrapper)
    }
  }

  override suspend fun getUserPreset(): UserPresetDataWrapper {
    val roomName = controllerContainer.metaController.getRoomName()
    val response = client.post("${controllerContainer.metaController.getBaseUrl()}/userpreset") {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(
        UserPresetRequestModel(
          controllerContainer.metaController.getAuthToken(),
          CLIENT_TYPE,
          roomName,
          VERSION
        )
      )
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  override suspend fun startRecording(): RecordingInfo {
    val recordingInfo: RecordingInfo
    if (controllerContainer.metaController.isV2AuthToken()) {
      val response =
        client.post("${controllerContainer.metaController.getBaseUrl()}/recordings") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          setBody(
            StartRecordingModel(controllerContainer.metaController.getMeetingId())
          )
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapperV2>()
      recordingInfo = RecordingInfo(response.data.id)
    } else {
      val response =
        client.post("${controllerContainer.metaController.getBaseUrl()}/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recording") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapper>()
      recordingInfo = RecordingInfo(response.data.recording.id)
    }
    return recordingInfo
  }

  override suspend fun stopRecording(recordingInfo: RecordingInfo) {
    if (controllerContainer.metaController.isV2AuthToken()) {
      client.put("${controllerContainer.metaController.getBaseUrl()}/recordings/${recordingInfo.id}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(
          StopRecordingModelV2("stop")
        )
        contentType(ContentType.Application.Json)
      }
    } else {
      client.put("${controllerContainer.metaController.getBaseUrl()}/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/recordings/${recordingInfo.id}") {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        setBody(
          StopRecordingModel("stop")
        )
        contentType(ContentType.Application.Json)
      }
    }
  }

  override suspend fun getActiveRecording(): RecordingInfo {
    return if (controllerContainer.metaController.isV2AuthToken()) {
      val response =
        client.get("${controllerContainer.metaController.getBaseUrl()}/recordings/active-recording/${controllerContainer.metaController.getMeetingId()}") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapperV2>()
      RecordingInfo(response.data.id)
    } else {
      val response =
        client.get("${controllerContainer.metaController.getBaseUrl()}/organizations/${controllerContainer.metaController.getOrgId()}/rooms/${controllerContainer.metaController.getRoomName()}/active-recording/") {
          header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
          contentType(ContentType.Application.Json)
        }.body<RecordingResponseWrapper>()
      RecordingInfo(response.data.recording.id)
    }
  }

  override suspend fun getPlugins(): MultiplePluginResponse {
    val pluginsUrl =
      "${controllerContainer.metaController.getBaseUrlWithoutVersion()}/v2/plugins/user"
    val response = client.get(pluginsUrl) {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun getPluginDetails(pluginId: String): PluginResponse {
    val pluginDetailsUrl =
      "${controllerContainer.metaController.getBaseUrlWithoutVersion()}/v2/plugins/view/$pluginId"
    val response = client.get(pluginDetailsUrl) {
        header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
        contentType(ContentType.Application.Json)
      }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse {
    val response = client.get("${pluginBaseUrl}/dyte-config.json") {
      contentType(ContentType.Application.Json)
    }
    return response.body()
  }

  // TODO : fix this endpoint
  override suspend fun authorizePlugin(pluginId: String): PluginAuthResponse {
    val authorizePluginUrl =
      "${controllerContainer.metaController.getBaseUrlWithoutVersion()}/v2/plugins/authorize/$pluginId"
    val response = client.post(authorizePluginUrl) {
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      contentType(ContentType.Application.Json)
      setBody(
        PluginAuthorizationModel(
          controllerContainer.metaController.getRoomName(),
          controllerContainer.metaController.getPeerId()
        )
      )
    }
    return response.body()
  }

  override suspend fun uploadLogs(logs: PostDyteLogs) {
    client.post("https://api-silos.dyte.io/otel/logs") {
      contentType(ContentType.Application.Json)
      header("Authorization", "Bearer ${controllerContainer.metaController.getAuthToken()}")
      setBody(logs)
    }
  }

  override suspend fun getLiveStreamUrl(): String {
    val response = client.get("${controllerContainer.metaController.getBaseUrl()}/meetings/${controllerContainer.metaController.getMeetingId()}/active-livestream").body<ActiveLiveStreamResponse>()
    return response.data.url
  }

  override suspend fun startLiveStream() {
    client.post("${controllerContainer.metaController.getBaseUrl()}/meetings/${controllerContainer.metaController.getMeetingId()}/livestreams").body<StartLiveStreamResponse>()
  }

  override suspend fun stopLiveStream() {
    client.post("${controllerContainer.metaController.getBaseUrl()}/meetings/${controllerContainer.metaController.getMeetingId()}/active-livestream/stop").body<StopLiveStreamResponse>()
  }

  override fun getClient(): HttpClient {
    return client
  }

  override suspend fun getIpDetails(peerId: String): IpDetailsResponsePayload {
    val response = client.get("https://media-location.dyte.io")
    if (response.status.value in 200..299) {
      // Workaround for deserializing as the response doesn't have [application/json; charset=utf-8] content-type set
      val ipDetailsResponsePayload = BaseApiService.json.decodeFromString(
        deserializer = IpDetailsResponsePayload.serializer(),
        string = response.bodyAsText()
      )

      // Is this check really needed? And shouldn't we check if location has both longitude as well as latitude?
      if ((ipDetailsResponsePayload.location?.length ?: 6) > 5) {
        return ipDetailsResponsePayload
      }
      throw Exception("Insufficient data")
    }

    throw Exception("Failed to get IP Details")
  }

  companion object {
    private const val GRAPHQL_ROOM_NODE_DATA_QUERY = """
      query Session(${'$'}roomName: String!, ${'$'}password: String, ${'$'}ipInformation: String) {
        session(roomName: ${'$'}roomName, password: ${'$'}password) {
          title,
          roomNodeLink(ipInformation: ${'$'}ipInformation),
          useHiveMedia,
          roomName,
          password
        }
      }
    """
  }
}

interface IApiClient {
  suspend fun getRoomNodeData(): MeetingSessionInfo
  suspend fun getParticipantInfo(): ParticipantInfo
  suspend fun getUserPreset(): UserPresetDataWrapper
  suspend fun getICEServers(): IceServersWrapper
  suspend fun startRecording(): RecordingInfo
  suspend fun stopRecording(recordingInfo: RecordingInfo)
  suspend fun getActiveRecording(): RecordingInfo
  suspend fun getPlugins(): MultiplePluginResponse
  suspend fun getPluginDetails(pluginId: String): PluginResponse
  suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse
  suspend fun authorizePlugin(pluginId: String): PluginAuthResponse
  suspend fun uploadLogs(logs: PostDyteLogs)
  suspend fun getLiveStreamUrl(): String
  suspend fun startLiveStream()
  suspend fun stopLiveStream()
  fun getClient(): HttpClient

  suspend fun getIpDetails(peerId: String): IpDetailsResponsePayload
}