package io.dyte.core.plugins

import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.ISocketController
import io.dyte.core.controllers.SocketMessageEventListener
import io.dyte.core.events.JoinRoomEventReceiver
import io.dyte.core.feat.DytePlugin
import io.dyte.core.network.IApiClient
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.encodeToJsonElement
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal class RoomNodePluginsController(
  private val v1Plugins: Set<String>,
  private val apiClient: IApiClient,
  private val json: Json,
  private val roomNodeSocket: ISocketController,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
  private val joinRoomEventReceiver: JoinRoomEventReceiver,
  private val controllerContainer: IControllerContainer
) : PluginsController(controllerContainer) {

  /*
  * note(swapnil): Using MainScope i.e. Main Dispatcher since the class itself is not doing any heavy work
  * directly on the main thread. All the heavy work is off-loaded to background thread by the suspend
  * functions.
  * */
  private var coroutineScope = MainScope()
  
  private val pluginSocketMessageEventListener = object : SocketMessageEventListener {
    override suspend fun onMessageEvent(event: InboundMeetingEvent) {
      onPluginSocketEvent(event)
    }
  }

  override suspend fun init() {
    addPluginSocketMessageListeners()
    fetchAllPlugins()

    joinRoomEventReceiver.joinRoomEvent.onEach {
      loadActivePlugins()
    }.launchIn(coroutineScope)
  }

  override suspend fun launchPlugin(pluginId: String, staggered: Boolean) {
    val activatePluginPayload = buildJsonObject {
      put("id", pluginId)
      put("staggered", staggered)
    }
    roomNodeSocket.sendMessage(OutboundMeetingEventType.ADD_ROOM_PLUGIN, activatePluginPayload)
  }

  override suspend fun closePlugin(pluginId: String) {
    val deactivatePluginPayload = buildJsonObject {
      put("id", pluginId)
    }
    roomNodeSocket.sendMessage(OutboundMeetingEventType.REMOVE_ROOM_PLUGIN, deactivatePluginPayload)
  }

  override suspend fun handlePluginMessage(pluginId: String, message: JsonObject) {
    val type = message["type"]?.jsonPrimitive?.content
    when (type) {
      "pluginEvent",
      "storePluginData",
      "getPluginData",
      "enablePluginForAll",
      "enablePluginForUser",
      "disablePluginForUser",
      "chatMessage" -> {
        val response = roomNodeSocket.sendMessageSyncGeneric(message)
        val jsonResponse = json.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }

      "getJoinedPeers" -> {
        val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
        val roomState = (socketIoMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState!!
        val peers = roomState.peers ?: emptyList()
        val pluginEvent = buildJsonObject {
          put("type", "websocket/get-joined-peers")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peers", json.encodeToJsonElement(peers))
          }
        }

        submitEventToPlugin(pluginId, pluginEvent)
      }

      "getPeerInfo" -> {
        val peerId =
          message["payload"]?.jsonObject?.get("peerId")?.jsonPrimitive?.content
            ?: controllerContainer.selfController.getSelf().id
        val socketPayload = buildJsonObject {
          put("type", type)
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peerId", peerId)
          }
        }
        val response = roomNodeSocket.sendMessageSyncGeneric(socketPayload)
        val jsonResponse = json.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }

      "getRoomState" -> {
        val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
        (socketIoMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState?.let {
          val roomState = it
          val peers = roomState.peers ?: emptyList()
          roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
            val enabledBy = it
            peers.find { it.userId == enabledBy }?.id?.let {
              val enablerPeerId = it
              val payload = buildJsonObject {
                put("type", "websocket/get-room-state")
                putJsonObject("payload") {
                  message["payload"]?.let { payloadElement ->
                    payloadElement.jsonObject.entries.forEach {
                      put(it.key, it.value)
                    }
                  }
                  put("roomName", roomState.roomName)
                  put("pluginInitiatorPeerId", enablerPeerId)
                  put("pluginInitiatorUserId", enabledBy)
                  put(
                    "peers",
                    json.encodeToJsonElement(peers)
                  )
                  put("displayTitle", roomState.displayTitle)
                  put("roomUUID", roomState.roomUUID)
                  put(
                    "currentPeer",
                    controllerContainer.selfController.getSelf().id
                  )
                }
              }
              submitEventToPlugin(pluginId, payload)
            }

          }

        }
      }

      "getPluginInitiatorId" -> {
        val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
        (socketIoMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState?.let {
          val roomState = it
          val peers = roomState.peers ?: emptyList()
          roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
            val enabledBy = it
            peers.find { it.userId == enabledBy }?.id?.let {
              val enablerPeerId = it
              val payload = buildJsonObject {
                put("type", "websocket/get-plugin-initiator-peer-id")
                putJsonObject("payload") {
                  message["payload"]?.let { payloadElement ->
                    payloadElement.jsonObject.entries.forEach {
                      put(it.key, it.value)
                    }
                  }
                  put("pluginInitiatorPeerId", enablerPeerId)
                }
              }
              submitEventToPlugin(pluginId, payload)
            }

          }

        }
      }

      "getPluginInitiatorUserId" -> {
        val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
        (socketIoMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState?.let {
          val roomState = it
          roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
            val enabledBy = it
            val payload = buildJsonObject {
              put("type", "websocket/get-plugin-initiator-user-id")
              putJsonObject("payload") {
                message["payload"]?.let { payloadElement ->
                  payloadElement.jsonObject.entries.forEach {
                    put(it.key, it.value)
                  }
                }
                put("pluginInitiatorUserId", enabledBy)
              }
            }
            submitEventToPlugin(pluginId, payload)
          }
        }
      }

      "pluginEventToParent" -> {}
      else -> {
        // no-op
      }
    }
  }

  private suspend fun loadActivePlugins() {
    val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
    (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload as WebSocketRoomStateModel).roomState?.let {
      val roomState = it
      roomState.plugins?.forEach { roomPlugin ->
        val plugin = _allPlugins[roomPlugin.id]
        plugin?.let {
          enablePluginForLocalUser(plugin.id, roomPlugin.enabledBy)
        }
      }
    }
  }

  private suspend fun fetchAllPlugins() {
    val getPluginsResponse = apiClient.getPlugins()
    val roomNodePlugins = getPluginsResponse.data.plugins.filter {
      v1Plugins.contains(it.id)
    }
    roomNodePlugins.forEach {
      val plugin = DytePlugin(
        id = it.id,
        name = it.name,
        picture = it.picture,
        description = it.description,
        isPrivate = it.private,
        staggered = it.staggered,
        baseURL = it.baseURL,
        controller = controllerContainer
      )
      _allPlugins[plugin.id] = plugin
    }
  }

  private fun addPluginSocketMessageListeners() {
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT,
      pluginSocketMessageEventListener
    )
  }

  private suspend fun onPluginSocketEvent(event: InboundMeetingEvent) {
    when (event.eventType) {
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED -> {
        onEnablePlugin(event.payload as WebSocketPluginEnabled)
      }

      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED -> {
        onDisablePlugin(event.payload as WebSocketPluginDisabled)
      }

      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA, InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT -> {
        onPluginSocketMessage(event.eventType.type, event.payload as WebSocketPluginEvent)
      }

      else -> {
        // TODO: log invalid plugin socket event
      }
    }
  }

  private suspend fun onEnablePlugin(payload: WebSocketPluginEnabled) {
    if (payload.id != null && payload.enabledBy != null) {
      // Return if plugin is already active
      var pluginActive = false
      _activePlugins[payload.id]?.let { plugin ->
        if (plugin.isActive) {
          pluginActive = true
        }
      }

      if (!pluginActive) {
        enablePluginForLocalUser(payload.id!!, payload.enabledBy!!)
        _activePlugins[payload.id]?.let { plugin ->
          if (plugin.isActive) {
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnPluginEnabled(
                plugin
              )
            )
          }
        }
      }
    }
  }

  private fun onDisablePlugin(payload: WebSocketPluginDisabled) {
    _activePlugins[payload.id]?.let {
      it.disableLocal()
      _activePlugins.remove(it.id)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginDisabled(it))
    }
  }

  private fun onPluginSocketMessage(type: String, payload: WebSocketPluginEvent) {
    payload.payload["id"]?.jsonPrimitive?.content?.let {
      val pluginPayload = buildJsonObject {
        put("type", type)
        put("payload", payload.payload)
      }
      submitEventToPlugin(it, pluginPayload)
    }
  }
}