package io.dyte.core.controllers

import io.dyte.core.ControllerScopeProvider
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.socket.*
import io.dyte.core.socket.SocketEvent.*
import io.dyte.core.socket.SocketOptions.Transport.WEBSOCKET
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.InboundMeetingEventType.*
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.BasePayloadModel
import io.dyte.core.socket.events.payloadmodel.OutboundMeetingEvents
import io.dyte.core.socket.events.payloadmodel.inbound.*
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

private const val QUERY_PARAM_ROOM_URL = "roomURL"
private const val QUERY_PARAM_PEER_ID = "peerId"
private const val QUERY_PARAM_AUTH_TOKEN = "authToken"
private const val QUERY_PARAM_VERSION = "version"
private const val VERSION = "0.5.0"

internal class SocketController(val controllerContainer: IControllerContainer) :
  ControllerScopeProvider(), ISocketController {
  private lateinit var socket: Socket

  // https://dyte.slack.com/archives/C043X7B6RHC/p1681805925170109
  private var isKicked = false

  private var isSocketInReconnectState = false
  private var socketSerialScope = CoroutineScope(newSingleThreadContext("SocketSerialScope"))

  private val socketConnectionEventManager = SocketConnectionEventManager()

  suspend fun init() {
    withContext(serialScope.coroutineContext) {
      val meetingSessionData = controllerContainer.apiClient.getRoomNodeData()
      controllerContainer.metaController.setMeetingTitle(meetingSessionData.title)
      val roomNodeLink = getRoomSocketLinkQueryParams(
        requireNotNull(meetingSessionData.roomNodeLink),
        controllerContainer.metaController.getPeerId(),
        controllerContainer.metaController.getMeetingId(),
        controllerContainer.metaController.getAuthToken()
      )

      val params = HashMap<String, String>().apply {
        put(QUERY_PARAM_ROOM_URL, controllerContainer.metaController.getMeetingId())
        put(QUERY_PARAM_PEER_ID, controllerContainer.metaController.getPeerId())
        put(QUERY_PARAM_AUTH_TOKEN, controllerContainer.metaController.getAuthToken())
        put(QUERY_PARAM_VERSION, VERSION)
        put("EIO", "4")
      }

      socket = Socket(
        endpoint = roomNodeLink,
        config = SocketOptions(
          transport = WEBSOCKET,
          queryParams = params
        )
      ) {
        on(Connect) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_CONNECT,
              isSocketInReconnectState
            )
          )

          if (isSocketInReconnectState) {
            controllerContainer.loggerController.traceLog("SocketController | Socket Reconnected")
            isSocketInReconnectState = false
            socketConnectionEventManager.submitConnectionEvent(SocketConnectionEvent.Reconnected)
          } else {
            controllerContainer.loggerController.traceLog("SocketController | Socket Connected")
            socketConnectionEventManager.submitConnectionEvent(SocketConnectionEvent.Connected)
          }

          continuation?.let { cont ->
            cont.resume(Unit)
            continuation = null
          }
        }

        on(Connecting) {
          controllerContainer.loggerController.traceLog("SocketController | Socket Connecting")
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_CONNECTING
            )
          )
        }

        on(Disconnect) {
          controllerContainer.loggerController.traceLog("SocketController | Socket Disconnected")
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_DISCONNECT
            )
          )

          socketConnectionEventManager.submitConnectionEvent(SocketConnectionEvent.Disconnected)
        }

        on(Error) {
          controllerContainer.loggerController.traceLog(
            "SocketController | Socket Connection error -> ${(it as? Exception)?.message ?: "Socket error"}"
          )
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_ERROR
            )
          )

          /*
          * Todo: Identify 'it' param as an IOException or Other.
          * Todo: Add subtypes to ConnectionError for Network Disconnect and Other
          * */
          socketConnectionEventManager.submitConnectionEvent(
            SocketConnectionEvent.ConnectionError(
              (it as? Exception)?.message ?: "Socket error"
            )
          )

          continuation?.let { cont ->
            cont.resume(Unit)
            continuation = null
          }
        }

        on(Reconnect) {
          controllerContainer.loggerController.traceLog("SocketController | Socket Reconnecting")

          isSocketInReconnectState = true
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_RECONNECT
            )
          )

          socketConnectionEventManager.submitConnectionEvent(SocketConnectionEvent.Reconnecting)
        }

        on(ReconnectAttempt) {
          controllerContainer.loggerController.traceLog(
            "SocketController | Socket Reconnection attempt -> $it"
          )

          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_RECONNECT_ATTEMPT
            )
          )

          socketConnectionEventManager.submitConnectionEvent(
            SocketConnectionEvent.ReconnectAttempt(it)
          )
        }

        on(Ping) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_PING
            )
          )
        }

        on(Pong) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_PONG
            )
          )
        }

        on(Message) {
          controllerContainer.eventController.triggerEvent(
            OnMeetingStateChanged(
              EVENT_MESSAGE
            )
          )
        }

        on("event://server-simple-message") { response, onDone ->
          if (isKicked) {
            return@on
          }
          socketSerialScope.launch {
            try {
              val parsedResponse =
                controllerContainer.socketMessageResponseParser.parseResponse(
                  response
                )
              controllerContainer.loggerController.traceLog(parsedResponse.eventType.name)
              handleEvent(parsedResponse.eventType, parsedResponse.payload, onDone)
            } catch (ignored: Exception) {

            }
          }
        }
      }
    }
  }

  private var continuation: Continuation<Unit>? = null
  suspend fun connect() {
    withContext(serialScope.coroutineContext) {
      if (!socket.isConnected()) {
        suspendCoroutine { cont ->
          continuation = cont
          socket.connect()
        }
      }
    }
  }

  override suspend fun sendMessage(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  ): String {
    val result = withContext(serialScope.coroutineContext) {
      val eventData = createEventData(outboundMeetingEventType, payload).toString()

      val response = socket.emitAck(
        OutboundMeetingEvents.SEND_MESSAGE.eventName,
        eventData
      )
      return@withContext requireNotNull(response)
    }
    return result
  }

  override suspend fun sendMessageAsync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  ) {
    serialScope.launch {
      val eventData = createEventData(outboundMeetingEventType, payload).toString()
      socket.emitAckAsync(
        OutboundMeetingEvents.SEND_MESSAGE.eventName,
        eventData
      )
    }
  }

  override suspend fun sendMessageSyncGeneric(
    payload: JsonObject
  ): String {
    return withContext(serialScope.coroutineContext) {
      val response = socket.emitAck(
        OutboundMeetingEvents.SEND_MESSAGE.eventName,
        payload.toString()
      )
      requireNotNull(response)
    }
  }

  override fun addConnectionListener(listener: SocketConnectionListener) {
    serialScope.launch {
      socketConnectionEventManager.addConnectionListener(listener)
    }
  }

  override fun removeConnectionListener(listener: SocketConnectionListener) {
    serialScope.launch {
      socketConnectionEventManager.removeConnectionListener(listener)
    }
  }

  override fun disconnect() {
    socket.disconnect()
  }

  private suspend fun handleEvent(
    inboundMeetingEventType: InboundMeetingEventType,
    payloadModel: BasePayloadModel,
    onDone: () -> Unit
  ) {
    when (inboundMeetingEventType) {
      WEB_SOCKET_PEER_JOINED -> handlePeerJoined(payloadModel as WebSocketMeetingPeerUser)
      WEB_SOCKET_PEER_LEFT -> handlePeerLeft(payloadModel as WebSocketPeerLeftModel)

      WEB_SOCKET_WAITLIST_PEER_ADDED -> handleWaitlistPeerAdded(
        payloadModel as WebSocketWaitlistPeerAdded
      )
      WEB_SOCKET_WAITLIST_PEER_ACCEPTED -> handleWaitlistPeerAccepted(
        payloadModel as WebSocketWaitlistPeerAccepted
      )
      WEB_SOCKET_WAITLIST_PEER_REJECTED -> handleWaitlistPeerRejected(
        payloadModel as WebSocketWaitlistPeerRejected
      )
      WEB_SOCKET_WAITLIST_PEER_CLOSED -> handleWaitlistPeerClosed(
        payloadModel as WebSocketWaitlistPeerClosed
      )

      WEB_SOCKET_SELECTED_PEERS -> handleSelectedPeers(payloadModel as WebSocketSelectedPeersModel)
      WEB_SOCKET_NEW_CONSUMER -> handleNewConsumer(
        payloadModel as WebSocketConsumerModel,
        onDone
      )
      WEB_SOCKET_RESUME_CONSUMER -> handleResumeConsumer(payloadModel as WebSocketConsumerResumedModel)
      WEB_SOCKET_CLOSE_CONSUMER -> handleCloseConsumer(payloadModel as WebSocketConsumerClosedModel)
      WEB_SOCKET_PAUSE_CONSUMER -> handlePauseConsumer(payloadModel as WebSocketConsumerClosedModel)
      WEB_SOCKET_PRODUCER_CONNECT -> handleProducerConnect(
        payloadModel as WebSocketProducerConnectModel
      )
      WEB_SOCKET_CONNECT_TRANSPORT -> handleConnectTransport(
        payloadModel as WebSocketConnectTransportModel
      )
      WEB_SOCKET_PEER_MUTED -> handlePeerMuted(payloadModel as WebSocketPeerMuteModel)
      WEB_SOCKET_PEER_UNMUTED -> handlePeerUnMuted(payloadModel as WebSocketPeerMuteModel)
      WEB_SOCKET_PRODUCER_CLOSED -> handleProducerClosed(payloadModel as WebSocketProducerClosedModel)
      WEB_SOCKET_ON_CHAT_MESSAGE -> handleChatMessage(payloadModel as WebSocketChatMessage)
      WEB_SOCKET_ON_CHAT_MESSAGES -> handleChatMessages(payloadModel as WebSocketChatMessagesModel)
      WEB_SOCKET_ON_POLLS -> handlePolls(payloadModel as WebSocketPollsModel)
      WEB_SOCKET_ON_POLL -> handlePoll(payloadModel as WebSocketPollModel)
      WEB_SOCKET_MUTE_ALL_VIDEO -> handleMuteAllVideo()
      WEB_SOCKET_MUTE_ALL_AUDIO -> handleMuteAllAudio()
      WEB_SOCKET_ACTIVE_SPEAKER -> {
        handleActiveSpeaker(payloadModel as WebSocketActiveSpeakerModel)
      }
      WEB_SOCKET_NO_ACTIVE_SPEAKER -> {
        handleNoActiveSpeaker()
      }
      WEB_SOCKET_PEER_PINNED -> {
        handlePeerPinned(payloadModel as WebSocketPeerPinnedModel)
      }
      WEB_SOCKET_RECORDING_STARTED -> {
        handleRecordingStarted()
      }
      WEB_SOCKET_RECORDING_STOPPED -> {
        handleRecordingStopped()
      }
      WEB_SOCKET_KICKED -> {
        handleKicked()
      }
      WEB_SOCKET_DISABLE_AUDIO -> {
        handleDisableAudio()
      }
      WEB_SOCKET_DISABLE_VIDEO -> {
        handleDisableVideo()
      }
      WEB_SOCKET_PLUGIN_ENABLED -> {
        handlePluginEnabled(payloadModel as WebSocketPluginEnabled)
      }
      WEB_SOCKET_PLUGIN_DISABLED -> {
        handlePluginDisabled(payloadModel as WebSocketPluginDisabled)
      }
      WEB_SOCKET_PLUGIN_DATA, WEB_SOCKET_PLUGIN_EVENT -> {
        handlePluginEvent(
          inboundMeetingEventType.type,
          payloadModel as WebSocketPluginEvent
        )
      }
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED -> {
        handlePresentOnStageRequestReceived(payloadModel as WebSocketWebinarPresentRequestAccepted)
      }
      WEB_SOCKET_PEER_ADDED_TO_STAGE -> {
        handlePeerAddedToStage(payloadModel as WebSocketWebinarStagePeer)
      }
      WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE -> {
        handlePeerRejectedToStage(payloadModel as WebSocketWebinarStagePeer)
      }
      WEB_SOCKET_STOPPED_PRESENTING -> {
        handleStoppedPresenting()
      }
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED -> {
        requestToJoinPeerAdded(payloadModel as WebSocketWebinarRequestToJoinPeerAdded)
      }
      else -> {
        // no-op
      }
    }
  }

  private fun requestToJoinPeerAdded(webSocketRequestPeerAddedToStage: WebSocketWebinarRequestToJoinPeerAdded) {
    controllerContainer.webinarController.onRequestToPresentPeerAdded(
      RequestToPresentParticipant(
        webSocketRequestPeerAddedToStage.id,
        webSocketRequestPeerAddedToStage.name,
        webSocketRequestPeerAddedToStage.requestToJoinType
      )
    )
  }

  private fun handleStoppedPresenting() {
    controllerContainer.webinarController.onStoppedPresenting()
  }

  private fun handlePeerAddedToStage(webSocketWebinarStagePeer: WebSocketWebinarStagePeer) {
    controllerContainer.webinarController.onPeerAddedToStage(webSocketWebinarStagePeer.id)
  }

  private fun handlePeerRejectedToStage(webSocketWebinarStagePeer: WebSocketWebinarStagePeer) {
    controllerContainer.webinarController.onPeerRejectedToStage(webSocketWebinarStagePeer.id)
  }

  private fun handlePresentOnStageRequestReceived(payload: WebSocketWebinarPresentRequestAccepted) {
    controllerContainer.webinarController.onRequestedToPresent(payload.requestToJoinType)
  }

  private suspend fun handleDisableVideo() {
    controllerContainer.selfController.disableVideo()
  }

  private suspend fun handleDisableAudio() {
    controllerContainer.selfController.disableAudio()
  }

  private fun handleKicked() {
    isKicked = true
    controllerContainer.selfController.onRemovedFromMeeting()
  }

  private fun handleRecordingStopped() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
  }

  private fun handleRecordingStarted() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
  }

  private fun handlePeerPinned(websocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    if (websocketPeerPinnedModel.peerId != null) {
      controllerContainer.participantController.onPeerPinned(websocketPeerPinnedModel)
    } else {
      controllerContainer.participantController.onPeerUnpinned()
    }
  }

  private fun handleNoActiveSpeaker() {
    controllerContainer.participantController.onNoActiveSpeaker()
  }

  private fun handleActiveSpeaker(webSocketActiveSpeakerModel: WebSocketActiveSpeakerModel) {
    controllerContainer.participantController.onActiveSpeaker(webSocketActiveSpeakerModel.peerId)
  }

  private fun createEventData(
    type: OutboundMeetingEventType?,
    payload: JsonElement?
  ): JsonObject {
    val mapper = hashMapOf<String, JsonElement>()
    type?.type?.let { mapper.put("type", JsonPrimitive(it)) }
    payload?.let { mapper.put("payload", payload) }
    return JsonObject(mapper)
  }

  /**
   * Get room socket link query params
   *
   * adds roomName, authToken, peerId to the room node link, we receive in the meeting session graphql api
   *
   * @param roomNodeLink
   * @param peerId
   * @param meetingName
   * @param authToken
   * @return
   */
  private fun getRoomSocketLinkQueryParams(
    roomNodeLink: String,
    peerId: String,
    meetingName: String,
    authToken: String
  ): String {
    val params = HashMap<String, String>().apply {
      put(
        QUERY_PARAM_ROOM_URL,
        controllerContainer.platformUtilsProvider.getPlatformUtils()
          .getUrlEncodedString(meetingName)
      )
      put(QUERY_PARAM_PEER_ID, peerId)
      put(QUERY_PARAM_AUTH_TOKEN, authToken)
      put(QUERY_PARAM_VERSION, VERSION)
      put("EIO", "4")
    }

    val map = HashMap<String, String>()
    params.forEach {
      map[it.key] = it.value
    }

    val urlBuilder = StringBuilder(roomNodeLink)

    if (!map.isEmpty()) {
      urlBuilder.append("?")

      map.keys.forEachIndexed { index, key ->
        urlBuilder.append(key).append("=").append(map[key])

        if (index < map.keys.size - 1) {
          urlBuilder.append("&")
        }
      }
    }

    return urlBuilder.toString()
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed) {
    controllerContainer.waitlistController.onWaitlistPeerClosed(webSocketWaitlistPeerClosed)
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected) {
    controllerContainer.waitlistController.onWaitlistPeerRejected(
      webSocketWaitlistPeerRejected
    )
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted) {
    controllerContainer.waitlistController.onWaitlistPeerAccepted(
      webSocketWaitlistPeerAccepted
    )
  }

  private fun handleWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded) {
    controllerContainer.waitlistController.onWaitlistPeerAdded(webSocketWaitlistPeerAdded)
  }

  private fun handlePoll(webSocketPollModel: WebSocketPollModel) {
    handleSocketPoll(webSocketPollModel.poll)
  }

  private suspend fun handleMuteAllVideo() {
    controllerContainer.selfController.disableVideo()
  }

  private suspend fun handleMuteAllAudio() {
    controllerContainer.selfController.disableAudio()
  }

  private fun handleSocketPoll(webSocketPoll: WebSocketPoll) {
    controllerContainer.pollsController.onNewPoll(webSocketPoll)
  }

  private fun handleChatMessages(webSocketChatMessagesModel: WebSocketChatMessagesModel) {
    controllerContainer.chatController.handleChatMessages(webSocketChatMessagesModel)
  }

  private fun handleChatMessage(webSocketChatMessage: WebSocketChatMessage) {
    controllerContainer.chatController.handleNewChatMessage(webSocketChatMessage)
  }

  private fun handlePolls(webSocketPollsModel: WebSocketPollsModel) {
    webSocketPollsModel.polls?.values?.forEach {
      handleSocketPoll(it)
    }
  }

  private fun handlePeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    controllerContainer.participantController.onPeerLeft(webSocketPeerLeftModel)
  }

  private fun handleCloseConsumer(payloadModel: WebSocketConsumerClosedModel) {
    controllerContainer.participantController.onPeerVideoMuted(payloadModel)
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handleCloseConsumer(
      payloadModel
    )
  }

  private fun handlePauseConsumer(payloadModel: WebSocketConsumerClosedModel) {
    controllerContainer.participantController.onPeerVideoMuted(payloadModel)
  }

  private fun handleResumeConsumer(payloadModel: WebSocketConsumerResumedModel) {
    controllerContainer.participantController.onPeerVideoUnmuted(payloadModel)
  }

  private fun handlePeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    controllerContainer.participantController.onPeerJoined(meetingPeerUser)
    controllerContainer.waitlistController.onPeerJoined(meetingPeerUser)
  }

  private fun handleSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    controllerContainer.participantController.onSelectedPeers(webSocketSelectedPeersModel)
  }

  private fun handleConnectTransport(
    payloadModel: WebSocketConnectTransportModel,
  ) {
    /*controllerContainer.platformUtilsProvider.getMediaSoupUtils().connectTransport(
      payloadModel.id ?: return,
      payloadModel.producing ?: false,
    )*/
  }

  private fun handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit
  ) {
    controllerContainer.mediaSoupController.onNewConsumer(webSocketConsumerModel)
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
      ?.handleNewConsumer(webSocketConsumerModel, onDone)
  }

  private fun handlePeerMuted(payloadModel: WebSocketPeerMuteModel) {
    if (payloadModel.peerId == controllerContainer.metaController.getPeerId()) {
      controllerContainer.selfController.onAudioDisabled()
    } else {
      controllerContainer.participantController.onPeerAudioMuted(payloadModel)
    }
  }

  private fun handlePeerUnMuted(payloadModel: WebSocketPeerMuteModel) {
    if (payloadModel.peerId == controllerContainer.metaController.getPeerId()) {
      controllerContainer.selfController.onAudioEnabled()
    } else {
      controllerContainer.participantController.onPeerAudioUnmuted(payloadModel)
    }
  }

  private fun handleProducerConnect(payloadModel: WebSocketProducerConnectModel) {
    controllerContainer.selfController.onVideoEnabled(requireNotNull(payloadModel.id))
  }

  @Suppress("UNUSED_PARAMETER")
  private fun handleProducerClosed(payloadModel: WebSocketProducerClosedModel) {
    controllerContainer.selfController.onVideoDisabled()
  }

  /* Plugin message handlers */
  private suspend fun handlePluginEnabled(payloadModel: WebSocketPluginEnabled) {
    controllerContainer.pluginsController.onEnablePlugin(payloadModel)
  }

  private fun handlePluginDisabled(payloadModel: WebSocketPluginDisabled) {
    controllerContainer.pluginsController.onDisablePlugin(payloadModel)
  }

  private fun handlePluginEvent(eventType: String, payloadModel: WebSocketPluginEvent) {
    controllerContainer.pluginsController.onPluginSocketEvent(eventType, payloadModel)
  }
}

interface ISocketController {
  fun disconnect()

  suspend fun sendMessage(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  ): String

  suspend fun sendMessageAsync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?
  )

  suspend fun sendMessageSyncGeneric(
    payload: JsonObject
  ): String

  fun addConnectionListener(listener: SocketConnectionListener)

  fun removeConnectionListener(listener: SocketConnectionListener)
}

interface SocketConnectionListener {
  fun onConnectionEvent(event: SocketConnectionEvent)
}

sealed class SocketConnectionEvent {
  object Connected : SocketConnectionEvent()

  object Disconnected : SocketConnectionEvent()

  data class ConnectionError(val message: String) : SocketConnectionEvent()

  object Reconnecting : SocketConnectionEvent()

  data class ReconnectAttempt(val attemptNumber: Int) : SocketConnectionEvent()

  data class ReconnectAttemptFailed(val attemptNumber: Int) : SocketConnectionEvent()

  object Reconnected : SocketConnectionEvent()

  object ReconnectFailed : SocketConnectionEvent()
}

internal class SocketConnectionEventManager {
  private var eventSerialScope = CoroutineScope(newSingleThreadContext("SocketEventSerialScope"))

  private val connectionListeners: MutableSet<SocketConnectionListener> = mutableSetOf()

  fun addConnectionListener(listener: SocketConnectionListener) {
    eventSerialScope.launch {
      connectionListeners.add(listener)
    }
  }

  fun removeConnectionListener(listener: SocketConnectionListener) {
    eventSerialScope.launch {
      connectionListeners.remove(listener)
    }
  }

  fun submitConnectionEvent(connectionEvent: SocketConnectionEvent) {
    eventSerialScope.launch {
      connectionListeners.forEach {
        it.onConnectionEvent(connectionEvent)
      }
    }
  }
}