package io.dyte.core.controllers

import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import kotlinx.coroutines.launch

internal class SelfController(
  private val participantInfo: ParticipantInfo,
  controllerContainer: IControllerContainer,
) : ISelfController, BaseController(controllerContainer) {
  private lateinit var selfParticipant: DyteSelfParticipant

  private var _roomJoined = false
  override val roomJoined: Boolean
    get() = _roomJoined

  override fun init() {
    this.selfParticipant =
      DyteSelfParticipant(
        controllerContainer.metaController.getPeerId(),
        requireNotNull(participantInfo.id),
        controllerContainer.metaController.getDisplayName(),
        participantInfo.picture,
        false,
        participantInfo.clientSpecificId,
        ParticipantFlags(controllerContainer.presetController.canRecord(), false),
        controllerContainer
      )
  }

  override fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  override suspend fun disableAudio() {
    controllerContainer.loggerController.traceLog("DyteSelf.disableAudio")
    if (!selfParticipant.audioEnabled) {
      return
    }

    controllerContainer.sfuUtils.muteSelfAudio()
    if (_roomJoined) {
      controllerContainer.socketController.sendMessage(MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = false
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    }
  }

  override fun onAudioDisabled() {
    selfParticipant._audioEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override suspend fun enableAudio() {
    controllerContainer.loggerController.traceLog("DyteSelf.enableAudio")
    if (selfParticipant.audioEnabled) {
      return
    }

    if (!controllerContainer.presetController.canPublishAudio()) {
      return
    }

    val permissionGranted =
      controllerContainer.permissionController.askPermissions(true, videoEnabled = false)

    if (!permissionGranted) {
      return
    }
    controllerContainer.metaController.getMeetingConfig().invalidateDefautlAudioSetting = true

    if (_roomJoined) {
      controllerContainer.sfuUtils.unmuteSelfAudio()
      controllerContainer.socketController.sendMessage(UN_MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    }
  }

  override fun onAudioEnabled() {
    selfParticipant._audioEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override suspend fun disableVideo() {
    controllerContainer.loggerController.traceLog("DyteSelf.disableVideo")
    if (!selfParticipant.videoEnabled) {
      // disableVideo should only be called for video Which is already enable
      return
    }
    controllerContainer.sfuUtils.muteSelfVideo()
    selfParticipant._videoEnabled = false
    selfParticipant._videoTrack = null
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoDisabled() {
    controllerContainer.mediaSoupController.destroyCurrentVideo()
    selfParticipant._videoEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoEnabled(videoProducerId: String) {
    selfParticipant._videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override suspend fun enableVideo() {
    controllerContainer.loggerController.traceLog("DyteSelf.enableVideo")
    if (selfParticipant.videoEnabled) {
      // enableVideo should only be called for video Which is already disabled
      return
    }
    if (!controllerContainer.presetController.canPublishVideo()) {
      return
    }

    val permissionGranted =
      controllerContainer.permissionController.askPermissions(false, videoEnabled = true)

    if (!permissionGranted) {
      return
    }

    selfParticipant._videoEnabled = true
    controllerContainer.metaController.getMeetingConfig().invalidateDefaultVideoSetting = true
    controllerContainer.sfuUtils.unmuteSelfVideo()
    controllerContainer.platformUtilsProvider.getMediaUtils().resumeVideo()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun setDevice(dyteAudioDevice: DyteAudioDevice) {
    controllerContainer.loggerController.traceLog("DyteSelf.setDevice.audio.${dyteAudioDevice.type.displayName}")
    controllerContainer.platformUtilsProvider.getMediaUtils().setAudioDevice(dyteAudioDevice)
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getAudioDevices()
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedAudioDevice()
  }

  override fun onAudioDevicesChanged() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getVideoDevices()
  }

  override fun setDevice(dyteVideoDevice: DyteVideoDevice) {
    controllerContainer.loggerController.traceLog("DyteSelf.setDevice.video.${dyteVideoDevice.type.displayName}")
    controllerContainer.platformUtilsProvider.getMediaUtils().setVideoDevice(dyteVideoDevice)
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedVideoDevice()
  }

  override fun onVideoDevicesChanged() {
    TODO("Not yet implemented")
  }

  override fun onEnteredInRoom() {
    _roomJoined = true
  }

  override fun onUpdate() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfUpdate)
  }

  override fun onRemovedFromMeeting() {
    serialScope.launch {
      controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
      controllerContainer.sfuUtils.leaveCall()
      controllerContainer.socketController.disconnect()
      controllerContainer.socketService.disconnect()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnHostKicked)
    }
  }
}

interface ISelfController {
  val roomJoined: Boolean
  fun getSelf(): DyteSelfParticipant
  suspend fun disableAudio()
  fun onAudioDisabled()
  suspend fun enableAudio()
  fun onAudioEnabled()
  suspend fun disableVideo()
  fun onVideoDisabled()
  suspend fun enableVideo()
  fun onVideoEnabled(videoProducerId: String)

  fun getAudioDevices(): List<DyteAudioDevice>
  fun setDevice(dyteAudioDevice: DyteAudioDevice)
  fun getSelectedAudioDevice(): DyteAudioDevice
  fun onAudioDevicesChanged()
  fun getVideoDevices(): List<DyteVideoDevice>
  fun setDevice(dyteVideoDevice: DyteVideoDevice)
  fun getSelectedVideoDevice(): DyteVideoDevice
  fun onVideoDevicesChanged()
  fun onEnteredInRoom()
  fun onUpdate()

  fun onRemovedFromMeeting()
}