package io.dyte.core.controllers

import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.network.info.RecordingInfo
import kotlinx.coroutines.launch

internal class RecordingController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IRecordingController {
  private lateinit var recordingInfo: RecordingInfo
  private var recordingState: DyteRecordingState = IDLE

  override fun init() {
    serialScope.launch {
      try {
        recordingInfo = controllerContainer.apiClient.getActiveRecording()
        updateState(RECORDING)
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
      } catch (e: Exception) {
        // e.printStackTrace()
      }
    }
  }

  override fun start() {
    if (controllerContainer.presetController.canRecord().not()) {
      return
    }

    serialScope.launch {
      try {
        updateState(STARTING)
        recordingInfo = controllerContainer.apiClient.startRecording()
      } catch (e: Exception) {
        e.printStackTrace()
        controllerContainer.loggerController.traceError("error in starting recording ${e.message}")
        updateState(IDLE)
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
      }
    }
  }

  override fun stop() {
    if (controllerContainer.presetController.canRecord().not()) {
      return
    }
    serialScope.launch {
      if (recordingState != RECORDING) {
        return@launch
      }
      try {
        if (::recordingInfo.isInitialized.not()) {
          recordingInfo = controllerContainer.apiClient.getActiveRecording()
        }
        if (recordingState == RECORDING) {
          updateState(STOPPING)
          controllerContainer.apiClient.stopRecording(recordingInfo)
          updateState(IDLE)
        }
      } catch (e: Exception) {
        e.printStackTrace()
        controllerContainer.loggerController.traceError("error in stopping recording ${e.message}")
        updateState(IDLE)
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
      }
    }
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  private fun updateState(newState: DyteRecordingState) {
    recordingState = newState
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnMeetingRecordingStateUpdate(
        recordingState
      )
    )
  }

  override fun onRecordingPeerJoined() {
    updateState(RECORDING)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
  }

  override fun onRecordingPeerLeft() {
    updateState(IDLE)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
  }
}

interface IRecordingController {
  fun start()
  fun stop()
  fun getState(): DyteRecordingState

  fun onRecordingPeerJoined()
  fun onRecordingPeerLeft()
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  STOPPING;
}