package io.dyte.core.controllers

import defaults.BuildKonfig
import io.dyte.callstats.CallStats
import io.dyte.callstats.events.DeviceInfo
import io.dyte.callstats.events.PeerMetaData
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.WaitListStatus.WAITING
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement

internal class RoomNodeController(
  controllerContainer: IControllerContainer,
  private val callStatsClient: CallStats,
) : IRoomNodeController, BaseController(controllerContainer) {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  private var waitingToReconnect: Boolean = false

  private val socketConnectionListener = object : SocketConnectionListener {
    override fun onConnectionEvent(event: SocketConnectionEvent) {
      onSocketConnectionEvent(event)
    }
  }

  override fun init() {
    jsonParser = BaseApiService.json
    controllerContainer.socketController.addConnectionListener(socketConnectionListener)
    serialScope.launch {
      try {
        iceServers = getICEServers()
      } catch (e: Exception) {
        controllerContainer.loggerController.traceError("error in getting ice servers ${e.message}")
      }
    }
  }

  override suspend fun joinRoom() {
    withContext(serialScope.coroutineContext) {
      controllerContainer.loggerController.traceLog("SelfController.joinRoom")
      if (controllerContainer.selfController.getSelf().shouldShowSetupScreen()) {
        try {
          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          roomState =
            requireNotNull(
              controllerContainer.socketMessageResponseParser.parseResponse(
                roomStateResponse
              ).payload as WebSocketRoomStateModel
            )
          controllerContainer.loggerController.traceLog("room state received")

          val routerCapabilitiesResponse =
            controllerContainer.socketController.sendMessage(GET_ROUTER_RTP_CAPABILITIES, null)
          routerCapabilities =
            (controllerContainer.socketMessageResponseParser.parseResponse(
              routerCapabilitiesResponse
            ).payload as RouterCapabilitiesModel)

          controllerContainer.loggerController.traceLog(
            "received and processed router capabilities"
          )

          // Created Media Soup devices and load RTP Capability
          withContext(Dispatchers.Main) {
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
              ?.loadRouterRtpCapabilities(routerCapabilities)
          }

          val joinRoomPayload = JoinRoomPayloadRequestModel()
          joinRoomPayload.device = Device(
            osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(),
            osVersionName = controllerContainer.platformUtilsProvider.getPlatformUtils()
              .getOsVersion()
          )
          joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
          joinRoomPayload.audioMuted = !controllerContainer.selfController.getSelf().audioEnabled
          joinRoomPayload.rtpCapabilities = routerCapabilities
          val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
          val joinRoomResponse = controllerContainer.socketController.sendMessage(
            OutboundMeetingEventType.JOIN_ROOM,
            payload
          )

          joinRoom =
            controllerContainer.socketMessageResponseParser.parseResponse(
              joinRoomResponse
            ).payload as WebSocketJoinRoomModel

          controllerContainer.loggerController.traceLog("join room done!")

          if (joinRoom.waitlisted == true) {
            controllerContainer.loggerController.traceLog("aww snap, I'm in waiting room!")
            controllerContainer.selfController.getSelf()._waitListStatus = WAITING
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnSelfWaitListStatusUpdate(
                WAITING
              )
            )
            return@withContext
          }

          controllerContainer.loggerController.traceLog(
            "loaded router rtp capabilities into media soup device"
          )

          val consumerResponse = controllerContainer.socketController.sendMessage(
            CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
            CreateWebRTCTransportPayloadRequestModel(
              forceTcp = false,
              producing = false,
              consuming = true
            )
          )
          )


          webRtcCreateTransportModelConsumer =
              (controllerContainer.socketMessageResponseParser.parseResponse(
                consumerResponse
              ).payload as WebRtcCreateTransportModel)
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
              ?.createWebRtcTransportRecv(webRtcCreateTransportModelConsumer, iceServers)


          controllerContainer.loggerController.traceLog("created receive transport")


          val producerResponse = controllerContainer.socketController.sendMessage(
            CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
            CreateWebRTCTransportPayloadRequestModel(
              forceTcp = false,
              producing = true,
              consuming = false
            )
          )
          )


            webRtcCreateTransportModelProducer =
              controllerContainer.socketMessageResponseParser.parseResponse(
                producerResponse
              ).payload as WebRtcCreateTransportModel
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
              ?.createWebRtcTransportProd(webRtcCreateTransportModelProducer, iceServers)
            controllerContainer.loggerController.traceLog("created produce transport")



          // check for remote peers joined before local user
          controllerContainer.participantController.handleRoomState(roomState)
          // Pick out active plugins, if any
          controllerContainer.pluginsController.handleRoomState(roomState)

          controllerContainer.loggerController.traceLog("room state processed")

          controllerContainer.participantController.handleRoomJoined(joinRoom)
          controllerContainer.waitlistController.handleRoomJoined(joinRoom)

          controllerContainer.metaController.setMeetingStartedTimestamp(
            requireNotNull(joinRoom.startedAt)
          )

          controllerContainer.loggerController.traceLog(
            "can produce audio ? ${controllerContainer.presetController.canPublishAudio()} } video ? ${controllerContainer.presetController.canPublishVideo()}"
          )

          val selfTracks =
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.getSelfTrack()
              ?: Pair(null, null)
          val selfParticipant = controllerContainer.selfController.getSelf()
          // selfParticipant.audioTrack = selfTracks.first
          selfParticipant._videoTrack = selfTracks.second
          controllerContainer.selfController.onEnteredInRoom()

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomJoined(
              controllerContainer.selfController.getSelf()
            )
          )

          controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
          controllerContainer.loggerController.traceLog("joined!")

          if (controllerContainer.permissionController.isPermissionGrated(CAMERA).not()) {
            controllerContainer.loggerController.traceLog("joined without camera permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
            )
          }
          if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE).not()) {
            controllerContainer.loggerController.traceLog("joined without microphone permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
            )
          }
          controllerContainer.pollsController.loadPolls()
          controllerContainer.chatController.loadChatMessages()

          controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()

          controllerContainer.participantController.setPage(0)

          sendCallJoinBeginCallStatEvent()
        } catch (e: Exception) {
          e.printStackTrace()
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
        }
      } else {
        controllerContainer.chatController.loadChatMessages()
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoined(
            controllerContainer.selfController.getSelf()
          )
        )
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
      }
    }
  }

  private suspend fun getICEServers(): List<IceServerData> {
    return requireNotNull(controllerContainer.apiClient.getICEServers().iceServers)
  }

  private fun onSocketConnectionEvent(event: SocketConnectionEvent) {

    when (event) {
      SocketConnectionEvent.Disconnected -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Disconnected from meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomDisconnected)
      }

      SocketConnectionEvent.Reconnecting -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Trying to rejoin meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnReconnectingToMeetingRoom)
      }

      SocketConnectionEvent.Reconnected -> {
        serialScope.launch {
          (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handlePreReconnection()
          joinRoom()
          waitingToReconnect = false
          controllerContainer.loggerController.traceLog(
            "RoomNodeController | Rejoined meeting room on network reconnection"
          )
          controllerContainer.eventController.triggerEvent(DyteEventType.OnReconnectedToMeetingRoom)
        }
      }

      is SocketConnectionEvent.ConnectionError -> {
        serialScope.launch {
          if (!waitingToReconnect) {
            controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
            controllerContainer.sfuUtils.leaveCall()
            waitingToReconnect = true
            controllerContainer.loggerController.traceLog(
              "RoomNodeController | Socket connection error. Waiting to rejoin meeting room"
            )
          }

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomConnectionError(event.message)
          )
        }
      }
      else -> {
        // no op
      }
    }
  }

  private fun sendCallJoinBeginCallStatEvent() {
    // cpus and memory field is populated by Callstats lib internally
    val deviceInfo = DeviceInfo(
      isMobile = true,
      osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(),
      osVersionName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsVersion(),
      cpus = 0,
      memory = null
    )

    val roomViewType = when (controllerContainer.metaController.getRoomType()) {
      "GROUP_CALL" -> {
        "groupCall"
      }

      else -> controllerContainer.metaController.getRoomType()
    }

    val peerMetaData = PeerMetaData(
      metaData = emptyMap(),
      deviceInfo = deviceInfo,
      displayName = controllerContainer.metaController.getDisplayName(),
      meetingEnv = null, // populated by Callstats lib internally
      peerId = controllerContainer.selfController.getSelf().id,
      userId = controllerContainer.selfController.getSelf().userId,
      clientSpecificId = controllerContainer.selfController.getSelf().clientSpecificId,
      roomName = controllerContainer.metaController.getRoomName(),
      roomUUID = controllerContainer.metaController.getRoomUuid(),
      permissions = emptyMap(),
      participantRole = null,
      roomViewType = roomViewType,
      sdkName = "mobile-core",
      sdkVersion = BuildKonfig.sdkVersion
    )
    callStatsClient.sendCallJoinBeginEvent(peerMetaData)
  }
}

interface IRoomNodeController {
  suspend fun joinRoom()
}
