package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveParticipantsChanged
import io.dyte.core.controllers.DyteEventType.OnParticipantsUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareEnded
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareStarted
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteScreenShareMeetingParticipant
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.platform.VideoView
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.webrtc.MediaStreamTrack
import io.dyte.webrtc.VideoStreamTrack
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlin.math.ceil

internal class ParticipantController(
  controllerContainer: IControllerContainer
) : IParticipantController, BaseController(controllerContainer) {

  private val joinedParticipants = arrayListOf<DyteJoinedMeetingParticipant>()
  private val activeParticipants = arrayListOf<DyteJoinedMeetingParticipant>()
  private val screenshareParticipants = arrayListOf<DyteScreenShareMeetingParticipant>()

  private val hiddenParticipants = arrayListOf<DyteJoinedMeetingParticipant>()

  override val meetingRoomParticipants = DyteRoomParticipants(
    controllerContainer.waitlistController.waitlistedParticipants,
    joinedParticipants,
    activeParticipants,
    screenshareParticipants,
    controllerContainer
  )

  private var _pinnedParticipant: DyteJoinedMeetingParticipant? = null
  override val pinnedParticipant: DyteJoinedMeetingParticipant?
    get() = _pinnedParticipant

  private var _activeSpeaker: DyteJoinedMeetingParticipant? = null
  override val activeSpeaker: DyteJoinedMeetingParticipant?
    get() = _activeSpeaker

  private lateinit var gridPagesInfo: GridInfo

  private var requestedPageNumber = 0

  override fun init() {
    gridPagesInfo = GridInfo(
      0,
      0,
      isNextPagePossible = false,
      isPreviousPagePossible = false
    )
  }

  override fun getVideoView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView? {
    return controllerContainer.platformUtilsProvider.getVideoUtils()
      .getVideoView(dyteMeetingParticipant)
  }

  override fun getScreenShareView(dyteMeetingParticipant: DyteScreenShareMeetingParticipant): VideoView {
    return controllerContainer.platformUtilsProvider.getVideoUtils()
      .getScreenShareView(dyteMeetingParticipant)
  }

  /**
   * On peer joined
   *
   * when a remote peer joins, this does not contain audio track or video track
   *
   * @param meetingPeerUser
   */
  override fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    val participant = getDyteParticipant(meetingPeerUser)
    if (participant.flags.hiddenParticipant || participant.flags.recorder) {
      hiddenParticipants.add(participant)
      if (participant.flags.recorder) {
        controllerContainer.recordingController.onRecordingPeerJoined()
      }
    } else {
      val existing = joinedParticipants.find { it.userId == participant.userId }
      if (existing == null) {
        joinedParticipants.add(participant)
      }
      updatePageCount()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerJoin(participant))
      if (joinedParticipants.size < controllerContainer.presetController.getMaxVideoCount()) {
        val existingActive = activeParticipants.find { it.userId == participant.userId }
        if (existingActive == null) {
          activeParticipants.add(participant)
        }
        controllerContainer.eventController.triggerEvent(
          OnActiveParticipantsChanged(
            activeParticipants
          )
        )
      }
    }
  }

  override fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    var isHiddentParticipantLeft = false
    var peerToRemove = joinedParticipants.find { it.id == webSocketPeerLeftModel.id }
    if (peerToRemove == null) {
      peerToRemove = hiddenParticipants.find { it.id == webSocketPeerLeftModel.id }
      if (peerToRemove != null) {
        if (peerToRemove.flags.recorder) {
          controllerContainer.recordingController.onRecordingPeerLeft()
        }
        hiddenParticipants.remove(peerToRemove)
        isHiddentParticipantLeft = true
      }
    }

    if (!isHiddentParticipantLeft) {
      peerToRemove?.let {
        val screenSharePeer = screenshareParticipants.find { it.id == peerToRemove.id }
        if (screenSharePeer != null) {
          screenshareParticipants.remove(screenSharePeer)
          controllerContainer.eventController.triggerEvent(OnPeerScreenShareEnded(screenSharePeer))
        }

        joinedParticipants.remove(peerToRemove)
        val removedFromActive = activeParticipants.removeAll { it.id == peerToRemove.id }
        if (removedFromActive) {
          if (activeParticipants.none { it.id == controllerContainer.selfController.getSelf().id }) {
            // add self to active now
            activeParticipants.add(controllerContainer.selfController.getSelf())
          }
          controllerContainer.eventController.triggerEvent(
            OnActiveParticipantsChanged(
              activeParticipants
            )
          )
        }
        updatePageCount()
        if (peerToRemove.id == pinnedParticipant?.id) {
          controllerContainer.eventController.triggerEvent(
            OnPeerUnpinned(
              requireNotNull(
                pinnedParticipant
              )
            )
          )
          _pinnedParticipant = null
        }
        controllerContainer.eventController.triggerEvent(OnPeerLeft(peerToRemove))
      }
    }
  }

  private fun updatePageCount(currentPageNumber: Int = gridPagesInfo.currentPageNumber) {
    val remotePeersCount = joinedParticipants.size.minus(1) // removing self
    val nextPages =
      if (remotePeersCount >= controllerContainer.presetController.getMaxVideoCount()) {
        ceil(
          joinedParticipants.size / controllerContainer.presetController.getMaxVideoCount()
            .toFloat()
        ).toInt().plus(1)
      } else {
        1
      }
    val newGridInfo = gridPagesInfo.copy(
      pageCount = nextPages,
      currentPageNumber = currentPageNumber,
      isNextPagePossible = currentPageNumber < nextPages.minus(1),
      isPreviousPagePossible = currentPageNumber > 0
    )
    if (newGridInfo != gridPagesInfo) {
      gridPagesInfo = newGridInfo
      controllerContainer.eventController.triggerEvent(OnParticipantsUpdate(meetingRoomParticipants))
    }
  }

  override fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = false
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel) {
    val consumerType =
      controllerContainer.mediaSoupController.getConsumerType(
        requireNotNull(webSocketConsumerClosedModel.id)
      )
    if ("audio" == consumerType) {
      return
    }
    webSocketConsumerClosedModel.id.let { id ->
      val appdata =
        controllerContainer.mediaSoupController.getAppDataFromConsumerId(id)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          if (appdata.screenShare == false) {
            participant._videoEnabled = false
            controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
          }
        }
      }
    }
  }

  override fun onPeerVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel) {
    webSocketConsumerClosedModel.id?.let {
      (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.resumeConsumer(it)
    }
  }

  override fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerPinnedModel.peerId }
    participant?.let {
      _pinnedParticipant?.let {
        controllerContainer.eventController.triggerEvent(OnPeerUnpinned(it))
      }
      _pinnedParticipant = participant
      controllerContainer.eventController.triggerEvent(OnPeerPinned(participant))
    }
  }

  override fun onPeerUnpinned() {
    _pinnedParticipant?.let {
      controllerContainer.eventController.triggerEvent(OnPeerUnpinned(it))
    }
    _pinnedParticipant = null
  }

  override fun onActiveSpeaker(id: String) {
    val participant = joinedParticipants.find { it.id == id }
    _activeSpeaker = participant
    _activeSpeaker?.let {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnActiveSpeakerChanged(
          requireNotNull(_activeSpeaker)
        )
      )
    }
  }

  override fun onNoActiveSpeaker() {
    _activeSpeaker = null
    controllerContainer.eventController.triggerEvent(DyteEventType.OnNoActiveSpeaker)
  }

  override fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel) {
    webSocketRoomStateModel.roomState?.peers?.forEach {
      onPeerJoined(it)
    }
  }

  override fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    if (joinedParticipants.find { it.userId == controllerContainer.selfController.getSelf().userId } == null) {
      joinedParticipants.add(controllerContainer.selfController.getSelf())
    }
    if (webSocketJoinRoomModel.pinnedPeerId?.isNotEmpty() == true) {
      val pinnedParticipant =
        joinedParticipants.find { it.id == webSocketJoinRoomModel.pinnedPeerId }
      pinnedParticipant?.let {
        _pinnedParticipant = pinnedParticipant
      }
    }
  }

  override fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    if (requestedPageNumber == 0) {
      processNewPage(
        webSocketSelectedPeersModel.peerIds?.toList() ?: emptyList(),
        0,
        gridPagesInfo.currentPageNumber
      )
    }
  }

  override suspend fun setPage(newPageNumber: Int) {
    if (newPageNumber < 0) {
      throw IllegalArgumentException("page number cant be less than 0")
    }
    requestedPageNumber = newPageNumber
    val content = HashMap<String, JsonElement>()
    content["pageNum"] = JsonPrimitive(newPageNumber)
    val prevPageResponse =
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.GET_PAGE,
        JsonObject(content)
      )
    val pageModel =
      controllerContainer.socketMessageResponseParser.parseResponse(prevPageResponse).payload as WebSocketGetPageModel
    if (pageModel.peerIds?.isEmpty() == false) {
      processNewPage(pageModel.peerIds, newPageNumber, gridPagesInfo.currentPageNumber)
    }
  }

  private fun processNewPage(peerIds: List<String>, pageNumber: Int, previousPageNumber: Int) {
    val newParticipant = getParticipantsArrayFrom(peerIds).toMutableList()

    if (pageNumber == 0) {
      if (controllerContainer.metaController.getRoomType() == "WEBINAR") {
        if (controllerContainer.webinarController.isPresenting()) {
          newParticipant.add(controllerContainer.selfController.getSelf())
        }
      } else {
        newParticipant.add(controllerContainer.selfController.getSelf())
      }
    }

    var activeParticipantsChanged = false

    if (pageNumber > 0) {
      activeParticipants.removeAll { it.id == controllerContainer.selfController.getSelf().id }
    }

    if (previousPageNumber != pageNumber) {
      // For new page we have to recreate active Participant and respect the one which is send by server
      activeParticipants.clear()
      activeParticipants.addAll(newParticipant.toList())
      activeParticipantsChanged = true
    } else {
      // Find new elements to add
      val participantsToAdd = getDiff(newParticipant, activeParticipants).toMutableList()
      // Find old elements to Remove
      val participantsToRemove = getDiff(activeParticipants, newParticipant)

      if (participantsToAdd.isNotEmpty() || participantsToRemove.isNotEmpty()) {
        // Means Either Add or Remove , And both Too, then we have to just trigger a callback
        activeParticipantsChanged = true
      }

      // search for element to remove
      participantsToRemove.forEach { toRemove ->
        var index = 0
        activeParticipants.forEach {
          // Find index of element in active array to remove
          if (it.id == toRemove.id) return
          index += 1
        }

        activeParticipants.removeAt(index)
        // Insert an element at the remove index if there is new participant is there to add.
        val elementToAdd = participantsToAdd.firstOrNull()
        elementToAdd?.let {
          // Replace this element inside the array
          activeParticipants.add(index, it)
          participantsToAdd.remove(it)
        }
      }
      // Check if we have elements still left to add
      participantsToAdd.forEach {
        activeParticipants.add(it)
      }
    }

    updatePageCount(pageNumber)
    if (activeParticipantsChanged) {
      controllerContainer.eventController.triggerEvent(
        OnActiveParticipantsChanged(
          activeParticipants
        )
      )
    }
  }

  private fun getParticipantsArrayFrom(peerIds: List<String>): List<DyteJoinedMeetingParticipant> {
    val result = arrayListOf<DyteJoinedMeetingParticipant>()
    peerIds.forEach { peerId ->
      val peer = joinedParticipants.find { it.id == peerId }
      peer?.let {
        result.add(peer)
      }
    }
    return result
  }

  private fun getDiff(
    a: List<DyteJoinedMeetingParticipant>,
    b: List<DyteJoinedMeetingParticipant>
  ): List<DyteJoinedMeetingParticipant> {
    val result = arrayListOf<DyteJoinedMeetingParticipant>()
    a.forEach { participant ->
      //check if this exist in Active participant array.
      val peer = b.find { it.id == participant.id }
      if (peer == null) {
        result.add(participant)
      }
    }
    return result
  }

  override fun onPeerScreenShareStarted(
    participant: DyteScreenShareMeetingParticipant,
    screenshareTrack: VideoStreamTrack
  ) {
    participant._screenShareTrack = screenshareTrack
    screenshareParticipants.add(participant)
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareStarted(participant))
    updatePageCount()
  }

  override fun onPeerScreenSharedEnded(participant: DyteScreenShareMeetingParticipant) {
    controllerContainer.platformUtilsProvider.getVideoUtils()
      .destroyView(screenshareParticipants.find { it.id == participant.id }!!)
    screenshareParticipants.removeAll { it.id == participant.id }
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareEnded(participant))
    updatePageCount()
  }

  private fun getDyteParticipant(meetingPeerUser: WebSocketMeetingPeerUser): DyteJoinedMeetingParticipant {
    val flags = ParticipantFlags(
      meetingPeerUser.flags?.recordere ?: false,
      meetingPeerUser.flags?.hiddenParticipant ?: false
    )
    return DyteJoinedMeetingParticipant(
      requireNotNull(meetingPeerUser.id),
      requireNotNull(meetingPeerUser.userId),
      meetingPeerUser.name ?: "",   // null in case of recorder
      meetingPeerUser.picture,
      meetingPeerUser.isHost ?: false,
      meetingPeerUser.clientSpecificId
        ?: controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid(),
      flags,
      controllerContainer
    ).apply {
      _audioEnabled = meetingPeerUser.audioMuted?.not() ?: false
    }
  }

  override fun getGridInfo(): GridInfo {
    return gridPagesInfo
  }
}

interface IParticipantController {
  val meetingRoomParticipants: DyteRoomParticipants
  val pinnedParticipant: DyteJoinedMeetingParticipant?
  val activeSpeaker: DyteJoinedMeetingParticipant?

  fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel)
  fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel)
  fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser)
  fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel)
  fun onPeerVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel)
  fun onPeerVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel)
  fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel)
  fun onPeerUnpinned()
  fun onActiveSpeaker(id: String)
  fun onNoActiveSpeaker()
  fun onPeerScreenShareStarted(
    participant: DyteScreenShareMeetingParticipant,
    screenshareTrack: VideoStreamTrack
  )

  fun onPeerScreenSharedEnded(participant: DyteScreenShareMeetingParticipant)
  fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel)
  suspend fun setPage(newPageNumber: Int)
  fun getVideoView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView?
  fun getScreenShareView(dyteMeetingParticipant: DyteScreenShareMeetingParticipant): VideoView

  fun getGridInfo(): GridInfo
}

data class GridInfo(
  val pageCount: Int,
  val currentPageNumber: Int,
  val isNextPagePossible: Boolean,
  val isPreviousPagePossible: Boolean
) {
  fun toMap(): Map<String, Any?> {
    val args = HashMap<String, Any?>()
    args["pageCount"] = pageCount
    args["currentPageNumber"] = currentPageNumber
    args["isNextPagePossible"] = isNextPagePossible
    args["isPreviousPagePossible"] = isPreviousPagePossible
    return args
  }
}