package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.LiveStreamingEvent
import socket.livestreaming.RoomPeerCountResponse

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService
) : BaseController(controllerContainer), ILiveStreamController {
  private var _liveStreamUrl: String? = null
  override val liveStreamUrl: String?
    get() = _liveStreamUrl

  private var _state: LiveStreamState = NONE
  override val state: LiveStreamState
    get() = _state

  private var _viewerCount: Int = 0
  override val viewerCount: Int
    get() = _viewerCount

  private var _roomName: String = ""
  override val roomName: String
    get() = _roomName

  private val socketEventListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, payload: ByteArray?) {
      when (event) {
        RoomEvent.STARTED.id -> {
          _state = STARTED
          val decodedPayload = LiveStreamingEvent.ADAPTER.decode(requireNotNull(payload))
          _roomName = decodedPayload.name
          _liveStreamUrl = decodedPayload.playback_url
          println("DyteMobileClient | LiveStreamController onEvent roomName $_roomName")
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        }

        RoomEvent.STOPPED.id -> {
          _state = STOPPED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
        }

        RoomEvent.ERRORED.id -> {
          _state = ERRORED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
        }

        RoomEvent.ROOM_PEER_COUNT.id -> {
          println("DyteMobileClient | LiveStreamController onEvent room peer count")
          val payload = RoomPeerCountResponse.ADAPTER.decode(requireNotNull(payload))
          println("DyteMobileClient | LiveStreamController onEvent ${payload.count}")
          _viewerCount = payload.count.toInt()
          controllerContainer.eventController.triggerEvent(OnViewerCountUpdated(_viewerCount))
        }
      }
    }
  }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ROOM_PEER_COUNT.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    try {
      serialScope.launch {
        try {
          _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
          _state = STARTED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        } catch (e: Exception) {
          // ignore exception
        }
      }
    } catch (e: Exception) {
      _state = NONE
      e.printStackTrace()
    }
  }

  override fun start() {
    serialScope.launch {
      controllerContainer.apiClient.startLiveStream()
    }
  }

  override fun stop() {
    serialScope.launch {
      controllerContainer.apiClient.stopLiveStream()
    }
  }
}

interface ILiveStreamController {
  val liveStreamUrl: String?
  val viewerCount: Int
  val state: LiveStreamState
  val roomName: String
  fun start()
  fun stop()
}

enum class LiveStreamState {
  NONE,
  STARTED,
  STOPPED,
  ERRORED;
}