package io.dyte.core.controllers

import io.dyte.core.chat.ChatSocketHandler
import io.dyte.core.chat.FileSocketChatMessage
import io.dyte.core.chat.ImageSocketChatMessage
import io.dyte.core.chat.SocketChatMessage
import io.dyte.core.chat.TextSocketChatMessage
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.platform.PlatformFile
import io.dyte.core.platform.Uri
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.logging.LogLevel.ALL
import io.ktor.client.plugins.logging.LogLevel.NONE
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.plugins.logging.SIMPLE
import io.ktor.client.request.header
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.request.url
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpStatusCode
import io.ktor.serialization.kotlinx.json.json
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json

/*
* TODO: Need to setup removeOnMessageListener somewhere.
* Ideal way would be to listen to `roomJoined` event to set listeners and
* `roomLeft` event to remove listeners.
* */
internal class ChatController(
  controllerContainer: IControllerContainer,
  private val chatSocketHandler: ChatSocketHandler,
  private val presignedUrlApiService: PresignedUrlApiService
) : IChatController, BaseController(controllerContainer), ChatSocketHandler.OnMessageListener {
  /*
  * Note(swapnil): We are using a separate instance of HttpClient in ChatController to upload files.
  * Need to investigate more to see if we can eliminate this extra instance.
  *
  * Reason: We have added a default Authorization header to the shared HttpClient in the ApiClient class. This
  * adds the Authorization header to every request. But the pre-signed upload Urls already have authorization
  * mechanism through some query params and don't allow additional Authorization header. In case, it is
  * added to the request, we get a 400 Bad request response with "Invalid Argument" message.
  * */
  private val httpClient: HttpClient by lazy { createPrivateHttpClient() }

  override val messages = arrayListOf<DyteChatMessage>()

  override fun init() {
    setOnMessageListener()
  }

  /**
   * Sets a message listener on the ChatSocketHandler for messages coming from the
   * socket service.
   * TODO: Need to figure out a way to remove listener from ChatSocketHandler.
   * Ideal way would be to listen to `roomJoined` event to set listeners and
   * `roomLeft` event to remove listeners.
   */
  private fun setOnMessageListener() {
    chatSocketHandler.setOnMessageListener(this)
  }

  override suspend fun loadChatMessages() {
    try {
      val socketChatMessages = chatSocketHandler.getChatMessages()
      // Need to offload this on a worker Dispatchers.Default
      val dyteChatMessages = arrayListOf<DyteChatMessage>()
      for (message in socketChatMessages) {
        try {
          val dyteChatMessage = convertToDyteChatMessage(message)
          dyteChatMessages.add(dyteChatMessage)
        } catch (e: Exception) {
          controllerContainer.loggerController.traceError("ChatController parse message error: $e, $message")
        }
      }
      messages.clear()
      messages.addAll(dyteChatMessages)
      if (dyteChatMessages.isNotEmpty()) {
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingMessagesReceived(
            dyteChatMessages
          )
        )
      }
    } catch (e: Exception) {
      controllerContainer.loggerController.traceError("ChatController loadChatMessages error: $e")
    }
  }

  override fun handleChatMessages(dyteChatMessages: WebSocketChatMessagesModel) {
    dyteChatMessages.messages?.forEach {
      val dyteChatMessage = addMessage(it)
      if (dyteChatMessage != null) {
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnNewMeetingMessageReceived(
            dyteChatMessage
          )
        )
      }
    }
  }

  override fun handleNewChatMessage(dyteChatMessage: WebSocketChatMessage) {
    val message = addMessage(dyteChatMessage)
    if (message != null) {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnNewMeetingMessageReceived(
          message
        )
      )
    }
  }

  override fun onMessageToRoom(message: SocketChatMessage) {
    try {
      val dyteChatMessage = convertToDyteChatMessage(message)
      messages.add(dyteChatMessage)
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnNewMeetingMessageReceived(
          dyteChatMessage
        )
      )
    } catch (e: Exception) {
      controllerContainer.loggerController.traceError("ChatController parse message error: $e, $message")
    }
  }

  private fun addMessage(chatMessage: WebSocketChatMessage): DyteChatMessage? {
    var dyteChatMessage: DyteChatMessage? = null
    when (chatMessage.type) {
      // text message
      0 -> {
        dyteChatMessage = DyteTextMessage(
          userId = chatMessage.userId,
          displayName = chatMessage.displayName,
          read = false,
          pluginId = null,
          message = requireNotNull(chatMessage.message),
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(chatMessage.time!!)
        )
        messages.add(dyteChatMessage)
      }

      // image message
      1 -> {
        dyteChatMessage = DyteImageMessage(
          userId = chatMessage.userId,
          displayName = chatMessage.displayName,
          read = false,
          pluginId = null,
          link = chatMessage.link ?: "",
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(chatMessage.time!!)
        )
        messages.add(dyteChatMessage)
      }

      // file message
      2 -> {
        dyteChatMessage = DyteFileMessage(
          userId = chatMessage.userId,
          displayName = chatMessage.displayName,
          read = false,
          pluginId = null,
          name = chatMessage.name ?: "",
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(chatMessage.time!!),
          link = chatMessage.link ?: "",
          size = chatMessage.size ?: 0L
        )
        messages.add(dyteChatMessage)
      }

      else -> {
        controllerContainer.loggerController.traceError("message type not supported $chatMessage")
      }
    }

    return dyteChatMessage
  }

  override fun sendMessage(message: String) {
    controllerContainer.loggerController.traceLog("DyteChat.sendTextMessage")
    if (!controllerContainer.presetController.canSendChatText()) {
      controllerContainer.loggerController.traceError("sendTextMessage::public_chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send text message")
    }

    if (message.isEmpty()) {
      controllerContainer.loggerController.traceError("sendTextMessage::message_can_not_be_empty")
      throw IllegalArgumentException("message can not be empty")
    }

    serialScope.launch {
      chatSocketHandler.sendTextMessage(message)
    }
  }

  override fun sendFileMessage(fileUri: Uri) {
    controllerContainer.loggerController.traceLog("DyteChat.sendFileMessage")
    if (!controllerContainer.presetController.canSendChatFile()) {
      controllerContainer.loggerController.traceLog("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      controllerContainer.platformUtilsProvider.getPlatformUtils().getPlatformFile(fileUri)
        ?: run {
          controllerContainer.loggerController.traceLog("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override fun sendFileMessage(filePath: String) {
    controllerContainer.loggerController.traceLog("DyteChat.sendFileMessage")
    if (!controllerContainer.presetController.canSendChatFile()) {
      controllerContainer.loggerController.traceLog("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      controllerContainer.platformUtilsProvider.getPlatformUtils().getPlatformFile(filePath)
        ?: run {
          controllerContainer.loggerController.traceLog("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override fun sendImageMessage(imageUri: Uri) {
    controllerContainer.loggerController.traceLog("DyteChat.sendImageMessage")

    if (!controllerContainer.presetController.canSendChatFile()) {
      controllerContainer.loggerController.traceError("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      controllerContainer.platformUtilsProvider.getPlatformUtils().getPlatformFile(imageUri)
        ?: run {
          controllerContainer.loggerController.traceError("sendImageMessage::required_argument_image_can_not_be_empty")
          throw Exception("Requested image doesn't exist or is missing read permission")
        }

    return sendImageMessageInternal(file)
  }

  override fun sendImageMessage(imagePath: String) {
    controllerContainer.loggerController.traceLog("DyteChat.sendImageMessage")
    if (!controllerContainer.presetController.canSendChatFile()) {
      controllerContainer.loggerController.traceError("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send image message")
    }

    val file =
      controllerContainer.platformUtilsProvider.getPlatformUtils().getPlatformFile(imagePath)
        ?: throw Exception("Requested image doesn't exist or is missing read permission")

    return sendImageMessageInternal(file)
  }

  private fun sendImageMessageInternal(file: PlatformFile) {
    if (file.mimeType !in ACCEPTED_IMAGE_TYPES) {
      controllerContainer.loggerController.traceError("sendImageMessage::image_type_not_supported")
      throw UnsupportedOperationException("Unsupported image file found -> ${file.mimeType}")
    }

    serialScope.launch {
      val presignedUrl = presignedUrlApiService.getPresignedUrl2(
        filename = file.name,
        roomUuid = controllerContainer.metaController.getRoomUuid()
      )

      val putLocation = presignedUrl?.data?.putLocation
      val getLocation = presignedUrl?.data?.getLocation

      if (putLocation.isNullOrBlank() || getLocation.isNullOrBlank()) {
        controllerContainer.loggerController.traceError("sendImageMessage::failure_get_presigned_url")
        throw Exception("Failed to upload image")
      }

      val uploadImageResponse = httpClient.put {
        url(putLocation)
        setBody(file.content)
      }

      if (uploadImageResponse.status.value != HttpStatusCode.OK.value) {
        controllerContainer.loggerController.traceError("sendImageMessage::failure_upload_image_file")
        throw Exception("Failed to upload image")
      }

      chatSocketHandler.sendImageMessage(getLocation)
    }
  }

  private fun sendFileMessageInternal(file: PlatformFile) {
    serialScope.launch {
      val presignedUrl = presignedUrlApiService.getPresignedUrl2(
        file.name,
        controllerContainer.metaController.getRoomUuid()
      )

      val putLocation = presignedUrl?.data?.putLocation
      val getLocation = presignedUrl?.data?.getLocation

      if (putLocation.isNullOrBlank() || getLocation.isNullOrBlank()) {
        controllerContainer.loggerController.traceError("sendFileMessage::failure_get_presigned_url")
        throw Exception("Failed to upload file")
      }

      val uploadFileResponse = httpClient.put(putLocation) {
        header(HttpHeaders.ContentType, file.mimeType)
        setBody(file.content)
      }

      if (uploadFileResponse.status.value != HttpStatusCode.OK.value) {
        controllerContainer.loggerController.traceError("sendFileMessage::failure_upload_file")
        throw Exception("Failed to upload file")
      }

      chatSocketHandler.sendFileMessage(getLocation, file.name, file.size)
    }
  }

  private fun createPrivateHttpClient(): HttpClient {
    return HttpClient {
      install(Logging) {
        logger = Logger.SIMPLE
        level =
          if (controllerContainer.platformUtilsProvider.getLogger()
              .isLoggerEnabled()
          ) ALL else NONE
      }
      install(ContentNegotiation) {
        json(Json {
          prettyPrint = true
          isLenient = true
          ignoreUnknownKeys = true
        })
      }
      install(HttpTimeout) {
        socketTimeoutMillis = 30000
        requestTimeoutMillis = 30000
        connectTimeoutMillis = 30000
      }
    }
  }

  private fun convertToDyteChatMessage(socketChatMessage: SocketChatMessage): DyteChatMessage {
    return when (socketChatMessage) {
      is TextSocketChatMessage -> {
        DyteTextMessage(
          userId = socketChatMessage.userId,
          displayName = socketChatMessage.displayName,
          read = socketChatMessage.read,
          pluginId = socketChatMessage.pluginId,
          message = socketChatMessage.message,
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(socketChatMessage.time)
        )
      }
      is ImageSocketChatMessage -> {
        DyteImageMessage(
          userId = socketChatMessage.userId,
          displayName = socketChatMessage.displayName,
          read = socketChatMessage.read,
          pluginId = socketChatMessage.pluginId,
          link = socketChatMessage.link,
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(socketChatMessage.time)
        )
      }
      is FileSocketChatMessage -> {
        DyteFileMessage(
          userId = socketChatMessage.userId,
          displayName = socketChatMessage.displayName,
          read = socketChatMessage.read,
          pluginId = socketChatMessage.pluginId,
          name = socketChatMessage.name,
          time = controllerContainer.platformUtilsProvider.getPlatformUtils()
            .getUserDisplayableTime(socketChatMessage.time),
          link = socketChatMessage.link,
          size = socketChatMessage.size
        )
      }
    }
  }

  companion object {
    private val ACCEPTED_IMAGE_TYPES = listOf("image/jpeg", "image/png")
  }
}

interface IChatController {
  val messages: List<DyteChatMessage>
  suspend fun loadChatMessages()
  fun handleChatMessages(dyteChatMessages: WebSocketChatMessagesModel)
  fun handleNewChatMessage(dyteChatMessage: WebSocketChatMessage)
  fun sendMessage(message: String)
  fun sendImageMessage(imageUri: Uri)
  fun sendImageMessage(imagePath: String)
  fun sendFileMessage(fileUri: Uri)
  fun sendFileMessage(filePath: String)
}