package io.dyte.core

import io.dyte.core.controllers.Controller
import io.dyte.core.feat.DyteChat
import io.dyte.core.feat.DyteLiveStream
import io.dyte.core.feat.DyteMeta
import io.dyte.core.feat.DytePlugins
import io.dyte.core.feat.DytePoll
import io.dyte.core.feat.DyteRecording
import io.dyte.core.feat.DyteWebinar
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteLiveStreamEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DytePluginEventsListener

import io.dyte.core.listeners.DyteDataUpdateListener
import io.dyte.core.listeners.DyteMicrophoneEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.platform.IDytePlatformUtilsProvider
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.withContext

/**
 * Dyte mobile client
 *
 * todo : rename -> DyteClient
 *
 * @constructor Create empty Dyte mobile client
 */
open class DyteMobileClient internal constructor(dytePlatformUtilsProvider: IDytePlatformUtilsProvider) :
  IDyteClient {
  private var controller: Controller = Controller(dytePlatformUtilsProvider)
  private var serialScope = CoroutineScope(newSingleThreadContext("DyteMobileClient"))

  override val localUser: DyteSelfParticipant
    get() = controller.getSelf()

  override val participants: DyteRoomParticipants
    get() = controller.getMeetingParticipant()

  override val chat: DyteChat
    get() = controller.getChat()

  override val recording: DyteRecording
    get() = controller.getRecording()

  override val polls: DytePoll
    get() = controller.getPolls()

  override val meta: DyteMeta
    get() = controller.getMeta()

  override val plugins: DytePlugins
    get() = controller.getPlugins()

  override val webinar: DyteWebinar
    get() = controller.getWebinar()

  override val liveStream: DyteLiveStream
    get() = controller.getLiveStream()

  override fun init(dyteMeetingInfo: DyteMeetingInfo) {
    serialScope.launch {
      controller.init(dyteMeetingInfo)
    }
  }

  override fun init(
    dyteMeetingInfo: DyteMeetingInfo,
    onSuccess: () -> Unit,
    onFailed: () -> Unit
  ) {
    serialScope.launch {
      try {
        controller.init(dyteMeetingInfo)
        onSuccess.invoke()
      } catch (e:Exception) {
        onFailed.invoke()
      }
    }
  }

  override fun init(dyteMeetingInfo: DyteMeetingInfoV2) {
    serialScope.launch {
      controller.init(dyteMeetingInfo)
    }
  }

  override fun init(
    dyteMeetingInfo: DyteMeetingInfoV2,
    onInitCompleted: () -> Unit,
    onInitFailed: () -> Unit
  ) {
    serialScope.launch {
      try {
        controller.init(dyteMeetingInfo)
        onInitCompleted.invoke()
      } catch (e:Exception) {
        onInitFailed.invoke()
      }
    }
  }

  override fun joinRoom() {
    serialScope.launch {
      controller.joinRoom()
    }
  }

  override fun joinRoom(onSuccess: () -> Unit, onFailed: () -> Unit) {
    serialScope.launch {
      try {
        controller.joinRoom()
        withContext(Dispatchers.Main) {
          onSuccess.invoke()
        }
      } catch (e: Exception) {
        withContext(Dispatchers.Main) {
          onFailed.invoke()
        }
        e.printStackTrace()
      }
    }
  }

  override fun leaveRoom() {
    serialScope.launch {
      controller.leaveRoom()
    }
  }

  override fun leaveRoom(onSuccess: () -> Unit, onFailed: () -> Unit) {
    serialScope.launch {
      try {
        controller.leaveRoom()
        withContext(Dispatchers.Main) {
          onSuccess.invoke()
        }
      } catch (e:Exception) {
        withContext(Dispatchers.Main) {
          onFailed.invoke()
        }
      }
    }
  }

  override fun release() {
    serialScope.launch {
      controller.release()
    }
  }

  override fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    controller.eventController.addRoomEventListener(meetingRoomEventsListener)
  }

  override fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    controller.eventController.removeRoomEventListener(meetingRoomEventsListener)
  }

  override fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.eventController.addSelfEventListener(selfEventsListener)
  }

  override fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.eventController.removeSelfEventListener(selfEventsListener)
  }

  override fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    controller.eventController.addParticipantEventListener(participantEventsListener)
  }

  override fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    controller.eventController.removeParticipantEventListener(participantEventsListener)
  }

  override fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    controller.eventController.addCameraEventListener(cameraEventsListener)
  }

  override fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    controller.eventController.removeCameraEventListener(cameraEventsListener)
  }

  override fun addMicrophoneEventsListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    controller.eventController.addMicrophoneEventListener(microphoneEventsListener)
  }

  override fun removeMicrophoneEventsListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    controller.eventController.removeMicrophoneEventListener(microphoneEventsListener)
  }

  override fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.eventController.addPluginEventListener(pluginEventsListener)
  }

  override fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.eventController.removePluginEventListener(pluginEventsListener)
  }

  override fun addLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    controller.eventController.addLiveStreamEventListener(liveStreamEventsListener)
  }

  override fun removeLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    controller.eventController.removeLiveStreamEventListener(liveStreamEventsListener)
  }

  override fun addWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    controller.eventController.addWaitlistEventListener(waitlistEventsListener)
  }

  override fun removeWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    controller.eventController.removeWaitlistEventListener(waitlistEventsListener)
  }

  override fun addChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    controller.eventController.addChatEventsListener(chatEventsListener)
  }

  override fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    controller.eventController.removeChatEventsListener(chatEventsListener)
  }

  override fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    controller.eventController.addRecordingEventsListener(recordingEventsListener)
  }

  override fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    controller.eventController.removeRecordingEventsListener(recordingEventsListener)
  }

  override fun addPollEventsListener(pollEventsListener: DytePollEventsListener) {
    controller.eventController.addPollEventsListener(pollEventsListener)
  }

  override fun removePollEventsListener(pollEventsListener: DytePollEventsListener) {
    controller.eventController.removePollEventsListener(pollEventsListener)
  }

  override fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    controller.eventController.addDataUpdateListener(dataUpdateListener)
  }

  override fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    controller.eventController.removeDataUpdateListener(dataUpdateListener)
  }
}

/**
 * Dyte client
 *
 * @constructor Create empty Dyte client
 */
internal interface IDyteClient {
  /**
   * This method makes the DyteClient ready to join in a given meeting.
   * Should be called from a background thread.
   *
   * @param dyteMeetingInfo
   */
  fun init(dyteMeetingInfo: DyteMeetingInfo)
  fun init(dyteMeetingInfo: DyteMeetingInfo, onInitCompleted: () -> Unit, onInitFailed: () -> Unit)

  /**
   *
   */
  fun init(dyteMeetingInfo: DyteMeetingInfoV2)
  fun init(dyteMeetingInfo: DyteMeetingInfoV2, onInitCompleted: () -> Unit, onInitFailed: () -> Unit)

  /**
   * Used to join in the room
   */
  fun joinRoom()
  fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit)

  /**
   * The leaveRoom() method can be used to leave a meeting.
   */
  fun leaveRoom()
  fun leaveRoom(onRoomLeft: () -> Unit, onRoomLeaveFailed: () -> Unit)

  fun release()

  /**
   * The self object can be used to manipulate audio and video settings,
   * and other configurations for the local participant.
   * This exposes methods to enable and disable media tracks, share the user's screen, etc.
   */
  val localUser: DyteSelfParticipant

  /**
   * The participants object consists of 4 lists of participants.
   * waitlisted, joined, active, pinned.
   * @return [DyteRoomParticipants]
   */
  val participants: DyteRoomParticipants

  /**
   * The chat object stores the chat messages that were sent in the meeting.
   * This includes text messages, images, and files.
   */
  val chat: DyteChat

  /**
   * This class helps you start and stop recording of a given meeting.
   */
  val recording: DyteRecording

  /**
   * The polls object stores the polls that were initiated in the meeting.
   * It exposes methods to create and vote on polls.
   */
  val polls: DytePoll

  /**
   * Meta object contains information about meetingTitle, meetingStartedAt, meetingType.
   */
  val meta: DyteMeta

  val plugins: DytePlugins

  val webinar: DyteWebinar

  val liveStream: DyteLiveStream

  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)
  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener)
  fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)

  fun addMicrophoneEventsListener(microphoneEventsListener: DyteMicrophoneEventsListener)
  fun removeMicrophoneEventsListener(microphoneEventsListener: DyteMicrophoneEventsListener)

  fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener)
  fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener)

  fun addLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener)
  fun removeLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener)

  fun addWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener)
  fun removeWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener)

  fun addChatEventsListener(chatEventsListener: DyteChatEventsListener)
  fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener)

  fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)
  fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)

  fun addPollEventsListener(pollEventsListener: DytePollEventsListener)
  fun removePollEventsListener(pollEventsListener: DytePollEventsListener)

  fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)
  fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)
}