package io.dyte.core.socket

import io.socket.client.Ack
import io.socket.client.IO
import io.socket.engineio.client.transports.Polling
import io.socket.engineio.client.transports.WebSocket
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject
import org.json.JSONArray
import org.json.JSONObject
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

actual class Socket actual constructor(
  endpoint: String,
  config: SocketOptions?,
  build: SocketBuilder.() -> Unit
) {
  private val socketIo: io.socket.client.Socket = IO.socket(endpoint, IO.Options().apply {
    transports = config?.transport?.let {
      when (it) {
        SocketOptions.Transport.DEFAULT -> return@let null
        SocketOptions.Transport.WEBSOCKET -> return@let arrayOf(WebSocket.NAME)
        SocketOptions.Transport.POLLING -> return@let arrayOf(Polling.NAME)
      }
    }
  })

  init {
    object : SocketBuilder {
      override fun on(event: String, action: Socket.(message: String, onDone: () -> Unit) -> Unit) {
        socketIo.on(event) {
          this@Socket.action(rawDataToString(it)) {
            if (it.size > 1 && it[1] is Ack) {
              (it[1] as Ack).call("Done")
            } else {
            }
          }
        }

        socketIo.io().on(event) {
          this@Socket.action(rawDataToString(it)) {
            // no-op
          }
        }
      }

      override fun <T> on(socketEvent: SocketEvent<T>, action: Socket.(data: T) -> Unit) {
        socketEvent.socketIoEvents.forEach { event ->
          socketIo.on(event) { data ->
            this@Socket.action(socketEvent.mapper(data))
          }

          socketIo.io().on(event) { data ->
            this@Socket.action(socketEvent.mapper(data))
          }
        }
      }
    }.build()
  }

  actual fun emit(event: String, data: JsonObject) {
    socketIo.emit(event, JSONObject(data.toString()))
  }

  actual fun emit(event: String, data: JsonArray) {
    socketIo.emit(event, JSONArray(data.toString()))
  }

  actual fun emit(event: String, data: String) {
    socketIo.emit(event, data)
  }

  actual suspend fun emitAck(event: String, data: String): String? {
    return suspendCoroutine { cont ->
      socketIo.emit(event, data, object : Ack {
        override fun call(vararg args: Any?) {
          val resp = rawDataToString(args)
          cont.resume(resp)
        }
      })
    }
  }

  actual fun emitAckAsync(event: String, data: String) {
    socketIo.emit(event, data)
  }

  actual fun connect() {
    socketIo.connect()
  }

  actual fun disconnect() {
    socketIo.disconnect()
  }

  actual fun isConnected(): Boolean {
    return socketIo.connected()
  }

  private companion object {
    fun rawDataToString(data: Array<out Any?>): String {
      return data.first().toString()
    }
  }
}