package io.dyte.core.controllers

import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteParticipant
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_WAITLIST_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_WAITLIST_REQUEST
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAccepted
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerRejected
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class WaitlistController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IWaitlistController {
  override val waitlistedParticipants = arrayListOf<DyteMeetingParticipant>()

  override fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    if (controllerContainer.presetController.permissions.waitingRoom.canAcceptRequests) {
      webSocketJoinRoomModel.waitlistedPeers?.forEach {
        val participant = DyteParticipant(
          requireNotNull(it.id),
          requireNotNull(it.id),
          requireNotNull(it.name),
          null,
          false,
          null,
          ParticipantFlags(false, false),
          controllerContainer
        )
        handleWaitlistedPeer(participant)
      }
    }
  }

  override fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {

  }

  override fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded) {
    val participant = DyteParticipant(
      webSocketWaitlistPeerAdded.id,
      webSocketWaitlistPeerAdded.id,
      webSocketWaitlistPeerAdded.name,
      null,
      false,
      null,
      ParticipantFlags(false, false),
      controllerContainer
    )
    handleWaitlistedPeer(participant)
  }

  private fun handleWaitlistedPeer(participant: DyteMeetingParticipant) {
    waitlistedParticipants.add(participant)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerJoined(participant))
  }

  override fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted) {
    controllerContainer.selfController.getSelf()._waitListStatus = ACCEPTED
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnWaitListPeerAccepted(
        controllerContainer.selfController.getSelf()
      )
    )
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnSelfWaitListStatusUpdate(
        ACCEPTED
      )
    )
   serialScope.launch {
     controllerContainer.roomNodeController.joinRoom()
   }
  }

  override fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected) {
    controllerContainer.selfController.getSelf()._waitListStatus = REJECTED
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnWaitListPeerRejected(
        controllerContainer.selfController.getSelf()
      )
    )
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnSelfWaitListStatusUpdate(
        REJECTED
      )
    )
  }

  override fun onWaitlistPeerRejected(dyteMeetingParticipant: DyteMeetingParticipant) {
    waitlistedParticipants.remove(dyteMeetingParticipant)
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnWaitListPeerRejected(
        dyteMeetingParticipant
      )
    )
  }

  override fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed) {
    val closedPeer = waitlistedParticipants.find { it.id == webSocketWaitlistPeerClosed.id }
    closedPeer?.let {
      waitlistedParticipants.remove(closedPeer)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerClosed(closedPeer))
    }
  }

  override suspend fun acceptAllWaitingRequests() {
    if (controllerContainer.presetController.permissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to accept all waitlisted request")
    }
    controllerContainer.participantController.meetingRoomParticipants.waitlisted.forEach {
      acceptWaitlistedRequest(it)
    }
  }

  override suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant) {
    if (controllerContainer.presetController.permissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to accept waitlisted request")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    controllerContainer.socketController.sendMessage(
      ACCEPT_WAITLIST_REQUEST,
      JsonObject(content)
    )
    waitlistedParticipants.remove(dyteMeetingParticipant)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerAccepted(dyteMeetingParticipant))
  }

  override suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant) {
    if (controllerContainer.presetController.permissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to reject waitlisted request")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    controllerContainer.socketController.sendMessage(
      REJECT_WAITLIST_REQUEST,
      JsonObject(content)
    )
    waitlistedParticipants.remove(dyteMeetingParticipant)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerRejected(dyteMeetingParticipant))
  }
}

interface IWaitlistController {
  val waitlistedParticipants: List<DyteMeetingParticipant>

  fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel)
  fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser)

  suspend fun acceptAllWaitingRequests()
  suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant)
  suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteMeetingParticipant)

  fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded)
  fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted)
  fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected)
  fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed)
  fun onWaitlistPeerRejected(dyteMeetingParticipant: DyteMeetingParticipant)
}