package io.dyte.core.controllers

import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.LiveStreamingEvent

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService
) : BaseController(controllerContainer), ILiveStreamController {
  private var _liveStreamUrl: String? = null
  override val liveStreamUrl: String?
    get() = _liveStreamUrl

  private var _state: LiveStreamState = NONE
  override val state: LiveStreamState
    get() = _state

  private val socketEventListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, payload: ByteArray?) {
      println("DyteMobileClient | LiveStreamController onEvent $event")
      when (event) {
        RoomEvent.STARTED.id -> {
          println("DyteMobileClient | LiveStreamController onEvent started")
          _state = STARTED
          val payload = LiveStreamingEvent.ADAPTER.decode(requireNotNull(payload))
          println("DyteMobileClient | LiveStreamController onEvent ${payload.playback_url}")
          _liveStreamUrl = payload.playback_url
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        }

        RoomEvent.STOPPED.id -> {
          println("DyteMobileClient | LiveStreamController onEvent stopped")
          _state = STOPPED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
        }

        RoomEvent.ERRORED.id -> {
          println("DyteMobileClient | LiveStreamController onEvent errored")
          _state = ERRORED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
        }
      }
    }
  }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    try {
      serialScope.launch {
        try {
          _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
          _state = STARTED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        } catch (e: Exception) {
          // ignore exception
        }
      }
    } catch (e: Exception) {
      _state = NONE
      e.printStackTrace()
    }
  }

  override fun start() {
    serialScope.launch {
      controllerContainer.apiClient.startLiveStream()
    }
  }

  override fun stop() {
    serialScope.launch {
      controllerContainer.apiClient.stopLiveStream()
    }
  }
}

interface ILiveStreamController {
  val liveStreamUrl: String?
  val state: LiveStreamState
  fun start()
  fun stop()
}

enum class LiveStreamState {
  NONE,
  STARTED,
  STOPPED,
  ERRORED;
}