package io.dyte.core.platform

import android.app.Application
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.mediasoup.Consumer
import io.dyte.mediasoup.Device
import io.dyte.mediasoup.LogHandler
import io.dyte.mediasoup.MediasoupClient
import io.dyte.mediasoup.MediasoupException
import io.dyte.mediasoup.Producer
import io.dyte.mediasoup.RecvTransport
import io.dyte.mediasoup.SendTransport
import io.dyte.mediasoup.Transport
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.serialization.encodeToString
import org.webrtc.AudioTrack
import org.webrtc.Logging.Severity.LS_VERBOSE
import org.webrtc.PeerConnection
import org.webrtc.PeerConnection.RTCConfiguration
import org.webrtc.VideoTrack

internal class DyteAndroidMediaSoup : IDyteMediaSoupUtils {
  private lateinit var controllerContainer: IControllerContainer
  private lateinit var peerConnectionUtils: IDytePeerConnectionUtils

  private var sendTransport: SendTransport? = null
  private var receiveTransport: RecvTransport? = null

  private var onCameraStreamKilled = false
  private val jsonParser = BaseApiService.json
  private val consumersToParticipants = HashMap<String, DyteMeetingParticipant>()
  private val consumers = arrayListOf<Consumer>()

  private var micProducer: Producer? = null
  private var cameraProducer: Producer? = null

  private lateinit var mediaSoupDevice: Device

  private var localVideoTrack: VideoTrack? = null
  private var localAudioTrack: AudioTrack? = null

  private var recvTransportListener: RecvTransport.Listener? = object : RecvTransport.Listener {
    val serialScope = CoroutineScope(newSingleThreadContext("recvTransportListener"))
    override fun onConnect(transport: Transport, dtlsParameters: String) {
      serialScope.launch {
        controllerContainer.mediaSoupController.onReceiveTransportConnected(
          transport.id,
          dtlsParameters
        )
      }
    }

    override fun onConnectionStateChange(transport: Transport, newState: String) {
      serialScope.launch {
        controllerContainer.connectionController.onReceiveTransportStateChanged(newState)
      }
    }
  }

  private var sendTransportListener: SendTransport.Listener? = object : SendTransport.Listener {
    val serialScope = CoroutineScope(newSingleThreadContext("sendTransportListener"))
    override fun onConnect(transport: Transport, dtlsParameters: String) {
      serialScope.launch {
        controllerContainer.mediaSoupController.onSendTransportConnected(
          transport.id,
          dtlsParameters
        )
      }
    }

    override fun onConnectionStateChange(transport: Transport, newState: String) {
      serialScope.launch {
        controllerContainer.connectionController.onSendTransportStateChanged(newState)
      }
    }

    override fun onProduce(
      transport: Transport,
      kind: String,
      rtpParameters: String,
      appData: String?
    ): String {
      serialScope.launch {
        controllerContainer.mediaSoupController.onProduce(
          transport.id,
          kind,
          rtpParameters,
          appData
        )
      }
      return ""
    }

    override fun onProduceData(
      transport: Transport,
      sctpStreamParameters: String,
      label: String,
      protocol: String,
      appData: String?
    ): String {
      return ""
    }
  }

  private var consumerListener: Consumer.Listener? = object : Consumer.Listener {
    override fun onTransportClose(consumer: Consumer) {
    }
  }

  private var produceListener: Producer.Listener? = object : Producer.Listener {
    override fun onTransportClose(producer: Producer) {
    }
  }

  override fun init(controllerContainer: IControllerContainer) {
    controllerContainer.platformUtilsProvider.getPlatformUtils()
      .printThread("DyteMobileClient | DyteAndroidMediaSoup init ")
    println("DyteMobileClient | DyteAndroidMediaSoup init ")
    this.controllerContainer = controllerContainer
    peerConnectionUtils = controllerContainer.platformUtilsProvider.getPeerConnectionUtils()

    if (controllerContainer.permissionController.isPermissionGrated(CAMERA) && controllerContainer.presetController.canPublishVideo()) {
      println("DyteMobileClient | DyteAndroidMediaSoup init creating video track")
      localVideoTrack =
        peerConnectionUtils.createVideoTrack("cam") as VideoTrack?
      controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      controllerContainer.selfController.getSelf()._videoEnabled = true
    } else {
      println("DyteMobileClient | DyteAndroidMediaSoup init no camera")
    }

    if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE) && controllerContainer.presetController.canPublishAudio()) {
      println("DyteMobileClient | DyteAndroidMediaSoup init creating audio track")
      localAudioTrack =
        peerConnectionUtils.createAudioTrack("mic") as AudioTrack
      controllerContainer.selfController.getSelf()._audioEnabled = true
    } else {
      println("DyteMobileClient | DyteAndroidMediaSoup init no microphone")
    }

    MediasoupClient.initialize(
      controllerContainer.platformUtilsProvider.getPlatformUtils()
        .getAndroidApplicationContext() as Application,
      object :
        LogHandler {
        override fun log(
          priority: Int,
          tag: String?,
          t: Throwable?,
          message: String?,
          vararg args: Any?,
        ) {
          // Log.v("DC-MediaSoup", message ?: "")
          // message?.printMediaSoupLog()
        }
      },
      false,
      null,
      LS_VERBOSE,
      "mediasoupclient_so"
    )
  }

  override fun loadRouterRtpCapabilities(routerRtpCapabilitiesString: String) {
    mediaSoupDevice = Device(null)
    // To fix orientation on mobile.
    val newRtp =
      routerRtpCapabilitiesString.replace(
        "urn:3gpp:video-orientation",
        "do not respect !urn:3gpp:video-orientation!"
      )
    mediaSoupDevice.load(newRtp)
  }

  @Suppress("DEPRECATION")
  override fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>
  ) {
    val id = requireNotNull(model.id)
    val iceParameters = jsonParser.encodeToString(model.iceParameters)
    val iceCandidates = jsonParser.encodeToString(model.iceCandidates)
    val dtlsParameters = jsonParser.encodeToString(model.dtlsParameters)

    val iceServersList = arrayListOf<PeerConnection.IceServer>()
    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      val iceServer = PeerConnection.IceServer(it.url, it.username, it.credential)
      iceServersList.add(iceServer)
    }

    receiveTransport = mediaSoupDevice.createRecvTransport(
      requireNotNull(recvTransportListener),
      id,
      iceParameters,
      iceCandidates,
      dtlsParameters,
      null,
      RTCConfiguration(iceServersList)
    )
  }

  @Suppress("DEPRECATION")
  override fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>
  ) {
    val id = requireNotNull(model.id)
    val iceParameters = jsonParser.encodeToString(model.iceParameters)
    val iceCandidates = jsonParser.encodeToString(model.iceCandidates)
    val dtlsParameters = jsonParser.encodeToString(model.dtlsParameters)

    val iceServersList = arrayListOf<PeerConnection.IceServer>()
    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      val iceServer = PeerConnection.IceServer(it.url, it.username, it.credential)
      iceServersList.add(iceServer)
    }

    sendTransport = mediaSoupDevice.createSendTransport(
      requireNotNull(sendTransportListener),
      id,
      iceParameters,
      iceCandidates,
      dtlsParameters,
      rtcConfig = RTCConfiguration(iceServersList)
    )
  }

  override fun handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit
  ) {
    val participant =
      controllerContainer.participantController.meetingRoomParticipants.joined.find { it.id == webSocketConsumerModel.peerId }
    participant?.let {
      val id = requireNotNull(webSocketConsumerModel.id)
      val producerId = requireNotNull(webSocketConsumerModel.producerId)
      val kind = requireNotNull(webSocketConsumerModel.kind)
      val rtpParameters = jsonParser.encodeToString(webSocketConsumerModel.rtpParameters)
      val appData: String = jsonParser.encodeToString(webSocketConsumerModel.appData)

      controllerContainer.platformUtilsProvider.getPlatformUtils()
        .printThread("DyteMobileClient | DyteAndroidMediaSoup handleNewConsumer ")

      try {
        val consumer = requireNotNull(receiveTransport).consume(
          requireNotNull(consumerListener),
          id,
          producerId,
          kind,
          rtpParameters,
          appData
        )
        consumers.add(consumer)
        consumersToParticipants[id] = participant
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    onDone.invoke()
  }

  override fun handleCloseConsumer(webSocketConsumerModel: WebSocketConsumerClosedModel) {
    webSocketConsumerModel.id?.let { consumerId ->
      val consumer = consumers.find { it.id == consumerId }
      consumer?.let {
        val appData =
          jsonParser.decodeFromString(ConsumerAppData.serializer(), consumer.appData)
        if (appData.screenShare == true) {
          val participant =
            requireNotNull(consumersToParticipants[webSocketConsumerModel.id])
          participant._screenShareTrack = null
          controllerContainer.participantController.onPeerScreenSharedEnded(participant)
        }
        consumers.remove(consumer)
        consumersToParticipants.remove(consumerId)
      }
    }
  }

  override fun resumeConsumer(id: String) {
    val consumer = consumers.find { it.id == id }
    consumer?.let {
      val participant = requireNotNull(consumersToParticipants[id])

      val isAudioTrack = consumer.track is AudioTrack

      consumer.resume()

      if (isAudioTrack) {
        // participant._audioEnabled = (consumer.track as AudioTrack).enabled()
        // controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerAudioUpdate(participant))
      } else {
        val appData =
          jsonParser.decodeFromString(ConsumerAppData.serializer(), consumer.appData)
        if (appData.screenShare == true) {
          controllerContainer.participantController.onPeerScreenShareStarted(
            participant,
            consumer.track
          )
        } else {
          participant._videoTrack = consumer.track as VideoTrack
          participant._videoEnabled = (consumer.track as VideoTrack).enabled()
          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnPeerVideoUpdate(
              participant
            )
          )
        }
      }
    } ?: run {
      // throw IllegalArgumentException("no consumer found for $id")
      controllerContainer.loggerController.traceError("no consumer found for $id")
    }
  }

  override fun connectAudioTransport(id: String) {
    controllerContainer.loggerController.traceLog("Android mediasoup connect Audio transports")
    if (controllerContainer.presetController.canPublishAudio()
      && controllerContainer.permissionController.isPermissionGrated(MICROPHONE)
    ) {
      enableMicImpl()
    }
  }

  override fun connectCameraTransport() {
    controllerContainer.loggerController.traceLog("Android mediasoup connect Camera transports")
    if (controllerContainer.presetController.canPublishVideo()
      && controllerContainer.permissionController.isPermissionGrated(CAMERA)
    ) {
      enableCamImpl()
    }
  }

  private fun enableCamImpl() {
    try {
      if (cameraProducer != null) {
        return
      }
      if (!mediaSoupDevice.loaded) {
        return
      }
      if (!mediaSoupDevice.canProduce("video")) {
        return
      }
      /*if (controllerContainer.selfController.getSelf()._videoEnabled.not()) {
        return
      }*/
      localVideoTrack?.setEnabled(controllerContainer.selfController.getSelf()._videoEnabled)
      createCameraProducer()
    } catch (e: MediasoupException) {
      e.printStackTrace()
    }
  }

  private fun enableMicImpl() {
    try {
      if (micProducer != null) {
        return
      }
      if (!mediaSoupDevice.loaded) {
        return
      }
      if (!mediaSoupDevice.canProduce("audio")) {
        return
      }
      if (localAudioTrack == null) {
        localAudioTrack =
          peerConnectionUtils.createAudioTrack("mic") as AudioTrack
        localAudioTrack?.setEnabled(true)
      }
      if (controllerContainer.selfController.getSelf().audioEnabled) {
        createMicProducer()
      }
    } catch (e: MediasoupException) {
      e.printStackTrace()
      if (localAudioTrack != null) {
        localAudioTrack?.setEnabled(false)
      }
    }
  }

  private fun createMicProducer() {
    if (sendTransport == null || localAudioTrack == null) {
      return
    }
    if (controllerContainer.presetController.canPublishAudio().not()) {
      return
    }
    if (micProducer == null) {
      val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())
      micProducer = requireNotNull(sendTransport).produce(
        requireNotNull(produceListener), localAudioTrack!!, emptyList(),
        null,
        BaseApiService.json.encodeToString(data)
      )
    }
  }

  private fun createCameraProducer() {
    if (cameraProducer != null) {
      return
    }
    if (sendTransport == null || localVideoTrack == null) {
      return
    }
    if (controllerContainer.presetController.canPublishVideo().not()) {
      return
    }
    val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())
    cameraProducer = requireNotNull(sendTransport).produce(
      requireNotNull(produceListener),
      localVideoTrack!!,
      emptyList(),
      null,
      BaseApiService.json.encodeToString(data)
    )
  }

  override fun muteSelfAudio(): Boolean {
    return localAudioTrack?.setEnabled(false) ?: false
  }

  override fun unmuteSelfAudio(): Boolean {
    createMicProducer()
    return localAudioTrack?.setEnabled(true) ?: false
  }

  override fun muteSelfVideo(): Boolean {
    cameraProducer?.pause()
    localVideoTrack?.setEnabled(false)
    return true
  }

  override fun unmuteSelfVideo(): Boolean {
    if (onCameraStreamKilled || localVideoTrack == null) {
      localVideoTrack =
        peerConnectionUtils.createVideoTrack("cam") as VideoTrack?
      createCameraProducer()
    }
    cameraProducer?.resume()
    localVideoTrack?.setEnabled(true)
    controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
    return true
  }

  override fun leaveCall() {
    try {
      consumers.forEach { consumer ->
        consumer.close()
      }

      // localVideoTrack?.dispose()
      // localAudioTrack?.dispose()

      // cameraProducer?.close()
      // micProducer?.close()

      sendTransport?.dispose()
      receiveTransport?.dispose()
      peerConnectionUtils.dispose()
      mediaSoupDevice.dispose()

      receiveTransport = null
      sendTransport = null
      produceListener = null
      consumerListener = null
      receiveTransport = null
      sendTransport = null
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  override fun getSelfTrack(): Pair<Any?, Any?> {
    return Pair(localAudioTrack, localVideoTrack)
  }

  override fun switchCamera() {
    peerConnectionUtils.switchCamera()
  }

  override fun onCameraStreamKilled() {
    onCameraStreamKilled = true
  }
}