package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.WaitListStatus.NONE
import io.dyte.core.platform.VideoView
import kotlinx.coroutines.*

// TODO : add enabledPreview, disablePreview
// TODO : getAudioDevices, getVideoDevices, getSpeakerDevices, getDeviceBydId, getAllDevices, setDevice
// TODO : setIsPinned, pin, unpin,
// TODO : add roomJoined, remove userId
// TODO : wrap audioTrack and videoTrack inside preview
// TODO : we can get rid of audioTrack, its not been used anywhere.
class DyteSelfParticipant internal constructor(
  override val id: String,
  override val userId: String,
  override var name: String,
  override val picture: String?,
  override val isHost: Boolean,
  override val clientSpecificId: String?,
  override val flags: ParticipantFlags,
  controllerContainer: IControllerContainer
) : DyteMeetingParticipant(
  id, userId, name, picture,
  isHost, clientSpecificId, flags, controllerContainer
) {
  val permissions = controllerContainer.presetController.permissions

  internal var _waitListStatus: WaitListStatus = NONE
  val waitListStatus: WaitListStatus
    get() = _waitListStatus

  internal var _cameraPermissionGranted: Boolean = controllerContainer.permissionController.isPermissionGrated(CAMERA)
  val isCameraPermissionGranted: Boolean
    get() = _cameraPermissionGranted

  internal var _microphonePermissionGranted: Boolean = controllerContainer.permissionController.isPermissionGrated(MICROPHONE)
  val isMicrophonePermissionGranted: Boolean
    get() = _microphonePermissionGranted

  val roomJoined: Boolean
    get() = controllerContainer.selfController.roomJoined

  /**
   * This method is used to unmute the local participant's audio.
   */

  override fun fetchVideoEnabled(): Boolean {
       if (!controllerContainer.metaController.isVideoEnabled()) {
          return  false
       }
      // Means user is already have videoEnable == true or enableVideo is called by user once.
      return super.fetchVideoEnabled()
  }
    override fun fetchAudioEnabled(): Boolean {
        if (!controllerContainer.metaController.isAudioEnabled()) {
            return  false
        }
        // Means user is already have videoEnable == true or enableVideo is called by user once.
        return super.fetchAudioEnabled()
    }

   fun enableAudio() {
    serialScope.launch {
      controllerContainer.selfController.enableAudio()
    }
  }

  /**
   * This method is used to mute the local participant's audio.
   */
  override fun disableAudio() {
    serialScope.launch {
      controllerContainer.selfController.disableAudio()
    }
  }

  /**
   * This method is used to start streaming the local participant's video to the meeting.
   */
   fun enableVideo() {
      serialScope.launch {
          controllerContainer.selfController.enableVideo()
      }
  }

  /**
   * This participant is used to disable the local participant's video.
   */
 override  fun disableVideo() {
    serialScope.launch {
      controllerContainer.selfController.disableVideo()
    }
  }
    override  fun getVideoView(): VideoView? {
        var videoEnableByDeveloper = controllerContainer.metaController.isVideoEnabled()
        if (!controllerContainer.presetController.canPublishVideo()
            || !videoEnableByDeveloper) {
            return null
        }
        return super.getVideoView()
    }
  fun setDisplayName(name: String) {
    serialScope.launch {
      this@DyteSelfParticipant.name = name
    }
  }

  fun getAudioDevices(): List<DyteAudioDevice> {
   return runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.selfController.getAudioDevices()
      }
    }
  }

  fun getVideoDevices(): List<DyteVideoDevice> {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.selfController.getVideoDevices()
      }
    }
  }

  fun setAudioDevice(dyteAndroidDevice: DyteAudioDevice) {
    serialScope.launch {
      controllerContainer.selfController.setDevice(dyteAndroidDevice)
    }
  }

  fun setVideoDevice(dyteVideoDevice: DyteVideoDevice) {
    serialScope.launch {
      controllerContainer.selfController.setDevice(dyteVideoDevice)
    }
  }

  fun getSelectedVideoDevice(): DyteVideoDevice {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.selfController.getSelectedVideoDevice()
      }
    }
  }

  fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.selfController.getSelectedAudioDevice()
  }

  fun canDoParticipantHostControls(): Boolean  {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.presetController.canDoParticipantHostControls()
      }
    }
  }

  fun shouldShowSetupScreen(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        permissions.media.canPublishAudio || permissions.media.canPublishVideo
      }
    }
  }

  override fun toMap(): Map<String, Any?> {
    val map = HashMap(super.toMap())
    map["selfPermissions"] = permissions.toMap()
    return map
  }
}