package io.dyte.core.controllers

import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import kotlinx.coroutines.launch

internal class SelfController(
  private val participantInfo: ParticipantInfo,
  controllerContainer: IControllerContainer,
) : ISelfController, BaseController(controllerContainer) {
  private lateinit var selfParticipant: DyteSelfParticipant

  private var _roomJoined = false
  override val roomJoined: Boolean
    get() = _roomJoined

  override fun init() {
    this.selfParticipant =
      DyteSelfParticipant(
        controllerContainer.metaController.getPeerId(),
        requireNotNull(participantInfo.id),
        controllerContainer.metaController.getDisplayName(),
        participantInfo.picture,
        false,
        participantInfo.clientSpecificId,
        ParticipantFlags(controllerContainer.presetController.canRecord(), false),
        controllerContainer
      )
  }

  override fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  override suspend fun disableAudio() {
    if (!selfParticipant.audioEnabled) {
      println("DyteMobileClient | audio is already disabled")
      return
    }

    controllerContainer.platformUtilsProvider.getSFUUtils().muteSelfAudio()
    if (_roomJoined) {
      controllerContainer.socketController.sendMessage(MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = false
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    }
  }

  override fun onAudioDisabled() {
    selfParticipant._audioEnabled = false
    println("DyteMobileClient | onAudioDisabled event received")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override suspend fun enableAudio() {

    if (selfParticipant.audioEnabled) {
      println("DyteMobileClient | audio is already enabled")
      return
    }

    if (!controllerContainer.presetController.canPublishAudio()) {
      return
    }

    val permissionGranted =
      controllerContainer.permissionController.askPermissions(true, videoEnabled = false)

    if (!permissionGranted) {
      return
    }
    controllerContainer.metaController.getMeetingConfig().invalidateDefautlAudioSetting = true

    if (_roomJoined) {
      controllerContainer.platformUtilsProvider.getSFUUtils().unmuteSelfAudio()
      controllerContainer.socketController.sendMessage(UN_MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    }
  }

  override fun onAudioEnabled() {
    selfParticipant._audioEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override suspend fun disableVideo() {
    println("DyteMobileClient | calling disableVideo")

    if (!selfParticipant.videoEnabled) {
      // disableVideo should only be called for video Which is already enable
      println("DyteMobileClient | video is already disabled")

      return
    }
    controllerContainer.platformUtilsProvider.getSFUUtils().muteSelfVideo()
    controllerContainer.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
    if (_roomJoined) {
      controllerContainer.mediaSoupController.sendDestroyVideoMessage()
    } else {
      selfParticipant._videoEnabled = false
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
    }
  }

  override fun onVideoDisabled() {

    controllerContainer.mediaSoupController.destroyCurrentVideo()
    selfParticipant._videoEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoEnabled(videoProducerId: String) {
    selfParticipant._videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override suspend fun enableVideo() {
    println("DyteMobileClient | calling enableVideo")
    if (selfParticipant.videoEnabled) {
      // enableVideo should only be called for video Which is already disabled
      println("DyteMobileClient | video is already enabled")
      return
    }
    if (!controllerContainer.presetController.canPublishVideo()) {
      println("DyteMobileClient | user is not having permission in Preset to publishVideo")
      return
    }

    val permissionGranted =
      controllerContainer.permissionController.askPermissions(false, videoEnabled = true)

    if (!permissionGranted) {
      println("DyteMobileClient | user does given permission to access camera")
      return
    }

    controllerContainer.metaController.getMeetingConfig().invalidateDefaultVideoSetting = true
    controllerContainer.platformUtilsProvider.getSFUUtils().unmuteSelfVideo()
    controllerContainer.platformUtilsProvider.getPeerConnectionUtils().resumeVideo()
    if (roomJoined) {
      controllerContainer.mediaSoupController.produceVideoIfNeeded()
    }
    selfParticipant._videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun setDevice(dyteAudioDevice: DyteAudioDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setAudioDevice(dyteAudioDevice)
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getAudioDevices()
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedAudioDevice()
  }

  override fun onAudioDevicesChanged() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getVideoDevices()
  }

  override fun setDevice(dyteVideoDevice: DyteVideoDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setVideoDevice(dyteVideoDevice)
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedVideoDevice()
  }

  override fun onVideoDevicesChanged() {
    TODO("Not yet implemented")
  }

  override fun onEnteredInRoom() {
    _roomJoined = true
  }

  override fun onUpdate() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfUpdate)
  }

  override fun onRemovedFromMeeting() {
    serialScope.launch {
      controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
      controllerContainer.platformUtilsProvider.getSFUUtils().leaveCall()
      controllerContainer.socketController.disconnect()
      controllerContainer.socketService.disconnect()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnHostKicked)
    }
  }
}

interface ISelfController {
  val roomJoined: Boolean
  fun getSelf(): DyteSelfParticipant
  suspend fun disableAudio()
  fun onAudioDisabled()
  suspend fun enableAudio()
  fun onAudioEnabled()
  suspend fun disableVideo()
  fun onVideoDisabled()
  suspend fun enableVideo()
  fun onVideoEnabled(videoProducerId: String)

  fun getAudioDevices(): List<DyteAudioDevice>
  fun setDevice(dyteAudioDevice: DyteAudioDevice)
  fun getSelectedAudioDevice(): DyteAudioDevice
  fun onAudioDevicesChanged()
  fun getVideoDevices(): List<DyteVideoDevice>
  fun setDevice(dyteVideoDevice: DyteVideoDevice)
  fun getSelectedVideoDevice(): DyteVideoDevice
  fun onVideoDevicesChanged()
  fun onEnteredInRoom()
  fun onUpdate()

  fun onRemovedFromMeeting()
}