package io.dyte.core.controllers

import io.dyte.core.feat.DytePlugin
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.ktor.http.URLBuilder
import io.ktor.http.set
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.encodeToJsonElement
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal class PluginsController(val controllerContainer: IControllerContainer):  IPluginsController {

  private val _allPlugins = mutableMapOf<String, DytePlugin>()
  private val _activePlugins = mutableMapOf<String, DytePlugin>()

  override val allPlugins: List<DytePlugin>
    get() =  _allPlugins.values.toList()

  override val activePlugins: List<DytePlugin>
    get() =  _activePlugins.values.toList()

  override suspend fun onEnablePlugin(payload: WebSocketPluginEnabled) {
      if (payload.id != null && payload.enabledBy != null) {
        // Return if plugin is already active
        var pluginAcive = false
        _activePlugins[payload.id]?.let { plugin ->
          if (plugin.isActive) {
            pluginAcive = true
          }
        }
        if (pluginAcive == false) {

          enablePluginForLocalUser(payload.id!!, payload.enabledBy!!)
          _activePlugins[payload.id]?.let { plugin ->
            if (plugin.isActive) {
              controllerContainer.eventController.triggerEvent(
                DyteEventType.OnPluginEnabled(
                  plugin
                )
              )
            }
          }
        }
      }
  }

  override fun onDisablePlugin(payload: WebSocketPluginDisabled) {
      _activePlugins[payload.id]?.let {
        it.disableLocal()
        _activePlugins.remove(it.id)
        controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginDisabled(it))
      }
  }

  override fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent) {
      payload.payload["id"]?.jsonPrimitive?.content?.let {
        val pluginPayload = buildJsonObject {
          put("type", type)
          put("payload", payload.payload)
        }
        submitEventToPlugin(it, pluginPayload)
    }
  }

  override suspend fun onPluginWebViewMessage(pluginId: String, message: JsonObject) {
      val type = message["type"]?.jsonPrimitive?.content
      when (type) {
        "pluginEvent",
        "storePluginData",
        "getPluginData",
        "enablePluginForAll",
        "enablePluginForUser",
        "disablePluginForUser",
        "chatMessage" -> {
          val response =
            controllerContainer.socketController.sendMessageSyncGeneric(message)
          val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
          submitEventToPlugin(pluginId, jsonResponse.jsonObject)
        }
        "getJoinedPeers" -> {
          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel).roomState!!
          val peers = roomState.peers ?: emptyList()
          val pluginEvent = buildJsonObject {
            put("type", "websocket/get-joined-peers")
            putJsonObject("payload") {
              message["payload"]?.let { payloadElement ->
                payloadElement.jsonObject.entries.forEach {
                  put(it.key, it.value)
                }
              }
              put("peers", Json { encodeDefaults = true }.encodeToJsonElement(peers))
            }
          }

          submitEventToPlugin(pluginId, pluginEvent)
        }
        "getPeerInfo" -> {
          val peerId =
            message["payload"]?.jsonObject?.get("peerId")?.jsonPrimitive?.content
              ?: controllerContainer.selfController.getSelf().id
          val socketPayload = buildJsonObject {
            put("type", type)
            putJsonObject("payload") {
              message["payload"]?.let { payloadElement ->
                payloadElement.jsonObject.entries.forEach {
                  put(it.key, it.value)
                }
              }
              put("peerId", peerId)
            }
          }
          val response =
            controllerContainer.socketController.sendMessageSyncGeneric(socketPayload)
          val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
          submitEventToPlugin(pluginId, jsonResponse.jsonObject)
        }
        "getRoomState" -> {
          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          (controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel).roomState?.let {
            val roomState = it
            val peers = roomState.peers ?: emptyList()
            roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
              val enabledBy = it
              peers.find { it.userId == enabledBy }?.id?.let {
                val enablerPeerId = it
                val payload = buildJsonObject {
                  put("type", "websocket/get-room-state")
                  putJsonObject("payload") {
                    message["payload"]?.let { payloadElement ->
                      payloadElement.jsonObject.entries.forEach {
                        put(it.key, it.value)
                      }
                    }
                    put("roomName", roomState.roomName)
                    put("pluginInitiatorPeerId", enablerPeerId)
                    put("pluginInitiatorUserId", enabledBy)
                    put(
                      "peers",
                      Json {
                        encodeDefaults = true
                      }.encodeToJsonElement(peers)
                    )
                    put("displayTitle", roomState.displayTitle)
                    put("roomUUID", roomState.roomUUID)
                    put(
                      "currentPeer",
                      controllerContainer.selfController.getSelf().id
                    )
                  }
                }
                submitEventToPlugin(pluginId, payload)
              }

            }

          }
        }
        "getPluginInitiatorId" -> {
          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          (controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel).roomState?.let {
            val roomState = it
            val peers = roomState.peers ?: emptyList()
            roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
              val enabledBy = it
              peers.find { it.userId == enabledBy }?.id?.let {
                val enablerPeerId = it
                val payload = buildJsonObject {
                  put("type", "websocket/get-plugin-initiator-peer-id")
                  putJsonObject("payload") {
                    message["payload"]?.let { payloadElement ->
                      payloadElement.jsonObject.entries.forEach {
                        put(it.key, it.value)
                      }
                    }
                    put("pluginInitiatorPeerId", enablerPeerId)
                  }
                }
                submitEventToPlugin(pluginId, payload)
              }

            }

          }
        }
        "getPluginInitiatorUserId" -> {
          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          (controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel).roomState?.let {
            val roomState = it
            roomState.plugins?.find { it.id == pluginId }?.enabledBy?.let {
              val enabledBy = it
              val payload = buildJsonObject {
                put("type", "websocket/get-plugin-initiator-user-id")
                putJsonObject("payload") {
                  message["payload"]?.let { payloadElement ->
                    payloadElement.jsonObject.entries.forEach {
                      put(it.key, it.value)
                    }
                  }
                  put("pluginInitiatorUserId", enabledBy)
                }
              }
              submitEventToPlugin(pluginId, payload)
            }
          }
        }
        "pluginEventToParent" -> {}
        else -> {
          // no-op
        }
    }
  }

  override suspend fun activatePlugin(plugin: DytePlugin) {
    if (controllerContainer.presetController.permissions.plugins.canLaunch.not()) {
      throw UnsupportedOperationException("can not start plugin")
    } else {
      val activatePluginPayload = buildJsonObject {
        put("id", plugin.id)
        put("staggered", plugin.staggered)
      }
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.ADD_ROOM_PLUGIN,
        activatePluginPayload
      )
    }
  }

  @Throws(Exception::class)
  override suspend fun deactivatePlugin(plugin: DytePlugin) {
    if (controllerContainer.presetController.permissions.plugins.canClose.not()) {
      throw UnsupportedOperationException("can not close plugin")
    } else {
      val deactivatePluginPayload = buildJsonObject {
        put("id", plugin.id)
      }
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.REMOVE_ROOM_PLUGIN,
        deactivatePluginPayload
      )
    }
  }

  override suspend fun handleRoomState(roomState: WebSocketRoomStateModel) {
      roomState.roomState?.plugins?.forEach { roomPlugin ->
        val plugin = _allPlugins[roomPlugin.id]
        plugin?.let {
          enablePluginForLocalUser(plugin.id, roomPlugin.enabledBy)
        }
    }
  }

  override fun onPluginFileRequest(plugin: DytePlugin) {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnPluginFileRequest(
          plugin
        )
      )
  }

  private suspend fun enablePluginForLocalUser(pluginId: String, enabledBy: String) {
    // for safe-guard
    if (_activePlugins.containsKey(pluginId)) return

    val plugin = _allPlugins[pluginId] ?: return

    // just for safe-guard
    if (plugin.isActive) return


    try {
      val pluginAuthToken =
        controllerContainer.apiClient.authorizePlugin(plugin.id).data.token
      val pluginUrl = URLBuilder(plugin.baseURL)
      pluginUrl.set {
        parameters["auth"] = pluginAuthToken
        parameters["pluginId"] = plugin.id
        parameters["backend"] = controllerContainer.metaController.getBaseUrl()
        encodedParameters["parent"] = "*"
      }

      plugin.enableLocal(
        pluginAuthToken,
        pluginUrl.buildString(),
        enabledBy
      )
      _activePlugins[plugin.id] = plugin
    } catch (e: Exception) {
      // no-op
      println("DyteMobileClient | Exception raise on enabling plugin ${e.message}")
    }
  }

  suspend fun init() {
      val response = controllerContainer.apiClient.getPlugins()
      response.data.plugins
        .filter {
          it.name.contains("v2", true).not()
        }  // filtering out v2 plugins from list
        .forEach {
          val plugin = DytePlugin(
            id = it.id,
            name = it.name,
            picture = it.picture,
            description = it.description,
            isPrivate = it.private,
            staggered = it.staggered,
            baseURL = it.baseURL,
            controller = controllerContainer
          )
          _allPlugins[plugin.id] = plugin
        }
  }

  private fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    _activePlugins[pluginId]?.submitEvent(payload)
  }
}

interface IPluginsController {
  val allPlugins: List<DytePlugin>

  val activePlugins: List<DytePlugin>

  suspend fun  onEnablePlugin(payload: WebSocketPluginEnabled)

  fun onDisablePlugin(payload: WebSocketPluginDisabled)

  fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent)

  suspend fun  onPluginWebViewMessage(pluginId: String, message: JsonObject)

  @Throws(Exception::class)
  suspend fun activatePlugin(plugin: DytePlugin)

  @Throws(Exception::class)
  suspend fun deactivatePlugin(plugin: DytePlugin)

  suspend fun  handleRoomState(roomState: WebSocketRoomStateModel)

  fun onPluginFileRequest(plugin: DytePlugin)
}
