package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.network.models.VideoConfig
import io.ktor.http.websocket.websocketServerAccept
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

internal class PermissionController(controller: IControllerContainer) : IPermissionController,
  BaseController(controller) {

  override val permissions: List<Permission>
    get() = _permissions.toList()

  private val _permissions = arrayListOf<Permission>()

  private var isNotified = false

  override fun init() {
    super.init()

    /*askPermissions(true, true, {
      //no-op
    }, {
      //no-op
    })*/
  }

  override fun isPermissionGrated(permissionType: PermissionType): Boolean {
    return controllerContainer.platformUtilsProvider.getPermissionUtils()
      .isPermissionGranted(Permission(permissionType, false))
  }

  override suspend fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  ) {
    return suspendCoroutine { continuation ->
      _askPermissions(audioEnabled, videoEnabled, onPermissionGranted = {
        onPermissionGranted.invoke()
        continuation.resume(Unit)
      }, onPermissionDenied = {
        onPermissionDenied.invoke()
        continuation.resume(Unit)
      })
    }
  }

  override suspend fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean
  ): Boolean {
    return suspendCoroutine { continuation ->
      _askPermissions(audioEnabled, videoEnabled, onPermissionGranted = {
        continuation.resume(true)
      }, onPermissionDenied = {
        continuation.resume(false)
      })
    }
  }

  private fun _askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  ) {
    isNotified = false
    val permissionsToAsk = arrayListOf<PermissionType>()
    if (videoEnabled) {
      permissionsToAsk.add(CAMERA)
    }
    if (audioEnabled) {
      permissionsToAsk.add(MICROPHONE)
    }

    _permissions.clear()
    permissionsToAsk.forEach {
      _permissions.add(Permission(it, false))
    }


    val isHandled = controllerContainer.platformUtilsProvider.getPermissionUtils()
      .askPermissions(_permissions, {
        if (isNotified.not()) {
          onPermissionGranted.invoke()
          isNotified = true
        }
      }, {
        if (isNotified.not()) {
          onPermissionDenied.invoke()
          isNotified = true
        }
      })
    if (!isHandled) {
      // THis should only be applicable for iOS
      var getAudioPermission: Boolean? = null
      var getVideoPermission: Boolean? = null

      fun call() {
        if (audioEnabled) {
          if (videoEnabled) {
            if (getAudioPermission != null && getVideoPermission != null) {
              if (getAudioPermission!! && getVideoPermission!!) {
                onPermissionGranted.invoke()
              } else {
                onPermissionDenied.invoke()
              }
            }
          } else {
            if (getAudioPermission != null) {
              if (getAudioPermission!!) {
                onPermissionGranted.invoke()
              } else {
                onPermissionDenied.invoke()
              }
            }
          }
        } else {
          if (videoEnabled) {
            if (getVideoPermission != null) {
              if (getVideoPermission!!) {
                onPermissionGranted.invoke()
              } else {
                onPermissionGranted.invoke()
              }
            }
          } else {
            onPermissionDenied.invoke()
          }
        }
      }
      if (audioEnabled || videoEnabled) {
        if (audioEnabled) {
          controllerContainer.platformUtilsProvider.getPermissionUtils()
            .askAudioPermissions({
              // granted
              getAudioPermission = true
              call()
            },
              {
                //  denied
                getAudioPermission = false
                call()
              })
        }
        if (videoEnabled) {
          controllerContainer.platformUtilsProvider.getPermissionUtils()
            .askCameraPermissions({
              // granted
              getVideoPermission = true
              call()
            }, {
              //  denied
              getVideoPermission = false
              call()
            })
        }
      } else {
        call()
      }
    }
  }
}

interface IPermissionController {
  val permissions: List<Permission>
  fun isPermissionGrated(permissionType: PermissionType): Boolean
  suspend fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  )

  suspend fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean
  ): Boolean
}

data class Permission(
  val type: PermissionType,
  var isGranted: Boolean
)

enum class PermissionType(val displayName: String) {
  CAMERA("Camera"),
  MICROPHONE("Mic"),
  STORAGE("Storage");
}