package io.dyte.core.controllers

import io.dyte.core.models.ProduceData
import io.dyte.core.network.BaseApiService
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.PRODUCE
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class MediaSoupController(controllerContainer: IControllerContainer) :
  IMediaSoupController, BaseController(controllerContainer) {
  private lateinit var _produceDataAudio: ProduceData
  private lateinit var _produceDataVideo: ProduceData

//    override val produceData: ProduceData
//        get() = _produceData

  private var _videoProduceId: String? = null
  override val videoProduceId: String?
    get() = _videoProduceId

  private val consumers = arrayListOf<WebSocketConsumerModel>()

  override suspend fun onReceiveTransportConnected(transportId: String, dtlsParameters: String) {
    withContext(serialScope.coroutineContext) {
      println("DyteMobileClient | MediaSoupController onReceiveTransportConnected ")
      controllerContainer.loggerController.traceLog("onReceiveTransportConnected")
      val content = HashMap<String, JsonElement>()
      content["transportId"] = JsonPrimitive(transportId)
      content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
      val payload = JsonObject(content)
      val resp = controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
        payload
      )
      println("DyteMobileClient | MediaSoupController onReceiveTransportConnected recv transport $resp")
    }
  }

  override suspend fun onSendTransportConnected(transportId: String, dtlsParameters: String) {
    println("DyteMobileClient | MediaSoupController onSendTransportConnected calling HP")
    withContext(serialScope.coroutineContext) {
      println("DyteMobileClient | MediaSoupController onSendTransportConnected HP")
      controllerContainer.loggerController.traceLog("onSendTransportConnected")
      val content = HashMap<String, JsonElement>()
      content["transportId"] = JsonPrimitive(transportId)
      content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
      val payload = JsonObject(content)
      println("DyteMobileClient | MediaSoupController before calling sendMessage HP")
      val resp = controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
        payload
      )
      println("DyteMobileClient | MediaSoupController onReceiveTransportConnected send transport $resp")

    }
  }

  override suspend fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String {
    return withContext(serialScope.coroutineContext) {
      if (kind == "video") {
        _produceDataVideo = ProduceData(transportId, kind, rtpParameters, appData)

        if (controllerContainer.selfController.getSelf()._videoEnabled) {
          _videoProduceId = sendUnMuteVideoMessage(_produceDataVideo)
        }

        return@withContext _videoProduceId ?: ""
      } else {
        _produceDataAudio = ProduceData(transportId, kind, rtpParameters, appData)
        sendUnMuteVideoMessage(_produceDataAudio)
      }
      return@withContext ""
    }
  }


  override suspend fun produceVideoIfNeeded() {
    withContext(serialScope.coroutineContext) {
      if (_videoProduceId == null) {
        controllerContainer.platformUtilsProvider.getPlatformUtils().printThread()
        _videoProduceId = sendUnMuteVideoMessage(_produceDataVideo)
      }
    }
  }

  override fun destroyCurrentVideo() {
    serialScope.launch {
      _videoProduceId = null
    }
  }

  override suspend fun sendDestroyVideoMessage(): Boolean {
    return withContext(serialScope.coroutineContext) {
      val producerId = videoProduceId
      producerId?.let {
        val content = HashMap<String, JsonElement>()
        content["producerId"] = JsonPrimitive(producerId)
        controllerContainer.socketController.sendMessage(
          OutboundMeetingEventType.MUTE_VIDEO,
          JsonObject(content)
        )
        return@withContext true
      } ?: run {
        println("DyteMobileClient | SelfController disableVideo not producer id")
      }
      return@withContext false
    }
  }


  private suspend fun sendUnMuteVideoMessage(produceData: ProduceData): String {
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(produceData.transportId)
    content["rtpParameters"] =
      BaseApiService.json.parseToJsonElement(produceData.rtpParameters!!)
    content["kind"] = JsonPrimitive(produceData.kind)
    content["appData"] = BaseApiService.json.parseToJsonElement(produceData.appData!!)
    try {
      val resp =
        controllerContainer.socketController.sendMessage(PRODUCE, JsonObject(content))
      val parsedResp =
        controllerContainer.socketMessageResponseParser.parseResponse(resp).payload as WebSocketProducerConnectModel
      return requireNotNull(parsedResp.id)
    } catch (e: Exception) {
      // TODO : handle this, this gives error
      println("DyteMobileClient | Exception ${e.message}")
      e.printStackTrace()
    }
    return ""
  }

  override fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel) {
    serialScope.launch {
      consumers.add(webSocketConsumerModel)
    }
  }

  override fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData {
    return requireNotNull(consumers.find { it.id == consumerId }!!.appData)
  }

  override fun getConsumerType(consumerId: String): String {
    val consumer = consumers.find { it.id == consumerId }
    return consumer?.kind
      ?: throw IllegalArgumentException("consumer with $consumerId not found")
  }
}

internal interface IMediaSoupController {
  //    val produceData: ProduceData
  val videoProduceId: String?
  suspend fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String

  suspend fun onReceiveTransportConnected(transportId: String, dtlsParameters: String)
  suspend fun onSendTransportConnected(transportId: String, dtlsParameters: String)
  suspend fun produceVideoIfNeeded()
  suspend fun sendDestroyVideoMessage(): Boolean

  //Below method will actually setting current video producer id to null, so that we can generate new videoProducer Id every time when unmute
  fun destroyCurrentVideo()
  fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel)
  fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData?
  fun getConsumerType(consumerId: String): String
}