package io.dyte.core.controllers

import defaults.BuildKonfig
import io.dyte.core.ControllerScopeProvider
import io.dyte.core.chat.DefaultChatSocketHandler
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeaveStarted
import io.dyte.core.feat.DyteChat
import io.dyte.core.feat.DyteLiveStream
import io.dyte.core.feat.DyteMeta
import io.dyte.core.feat.DytePlugins
import io.dyte.core.feat.DytePoll
import io.dyte.core.feat.DyteRecording
import io.dyte.core.feat.DyteWebinar
import io.dyte.core.featureflag.DyteFeatureFlagEntity
import io.dyte.core.featureflag.DyteFeatureFlags
import io.dyte.core.featureflag.FeatureFlagService
import io.dyte.core.featureflag.FeatureFlagService.UserAttributeKeys
import io.dyte.core.featureflag.FlagsmithService
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.ApiClient
import io.dyte.core.network.IApiClient
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.observability.ILoggerController
import io.dyte.core.observability.LoggerController
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.SocketMessageResponseParser
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.DefaultSocketService
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import socket.room.JoinRoomRequest
import socket.room.Peer
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

internal class Controller(private val _platformUtilsProvider: IDytePlatformUtilsProvider) :
  IController, IControllerContainer, ControllerScopeProvider() {

  override val platformUtilsProvider: IDytePlatformUtilsProvider
    get() = _platformUtilsProvider

  private lateinit var _chatController: ChatController
  override val chatController: IChatController
    get() = _chatController

  private lateinit var _participantController: ParticipantController
  override val participantController: IParticipantController
    get() = this._participantController

  private lateinit var _pollController: PollController
  override val pollsController: IPollController
    get() = _pollController

  private lateinit var _metaController: MetaController
  override val metaController: IMetaController
    get() = _metaController

  private lateinit var _selfController: SelfController
  override val selfController: ISelfController
    get() = this._selfController

  private lateinit var _roomNodeController: RoomNodeController
  override val roomNodeController: IRoomNodeController
    get() = this._roomNodeController

  private lateinit var _presetController: PresetController
  override val presetController: IPresetController
    get() = this._presetController

  private lateinit var _hostController: HostController
  override val hostController: IHostController
    get() = this._hostController

  private lateinit var _apiClient: ApiClient
  override val apiClient: IApiClient
    get() = this._apiClient

  private lateinit var _socketController: SocketController
  override val socketController: ISocketController
    get() = this._socketController

  private var _eventController = EventController(this)
  override val eventController: IEventController
    get() = this._eventController

  private var _connectionController = ConnectionController(this)
  override val connectionController: IConnectionController
    get() = this._connectionController

  private lateinit var _socketMessageResponseParser: SocketMessageResponseParser
  override val socketMessageResponseParser: ISocketMessageResponseParser
    get() = this._socketMessageResponseParser

  private lateinit var _recordingController: RecordingController
  override val recordingController: IRecordingController
    get() = this._recordingController

  private lateinit var _permissionController: PermissionController
  override val permissionController: IPermissionController
    get() = this._permissionController

  private lateinit var _pluginsController: PluginsController
  override val pluginsController: IPluginsController
    get() = this._pluginsController

  private lateinit var _loggerController: LoggerController
  override val loggerController: ILoggerController
    get() = this._loggerController

  private lateinit var _mediaSoupController: MediaSoupController
  override val mediaSoupController: IMediaSoupController
    get() = this._mediaSoupController

  private lateinit var _webinarController: WebinarController
  override val webinarController: IWebinarController
    get() = this._webinarController

  private lateinit var _liveStreamController: LiveStreamController
  override val liveStreamController: ILiveStreamController
    get() = this._liveStreamController

  private lateinit var _waitlistController: WaitlistController
  override val waitlistController: IWaitlistController
    get() = this._waitlistController

  private lateinit var recording: DyteRecording
  private lateinit var poll: DytePoll
  private lateinit var meta: DyteMeta
  private lateinit var plugins: DytePlugins
  private lateinit var webinar: DyteWebinar
  private lateinit var liveStream: DyteLiveStream
  private lateinit var chat: DyteChat

  private lateinit var _socketService: SocketService
  override val socketService: SocketService
    get() = this._socketService

  private lateinit var featureFlagService: FeatureFlagService

  override suspend fun init(dyteMeetingInfo: DyteMeetingInfo) {

    var invalidateAudio = false
    var invalidateVideo = false
    if (dyteMeetingInfo.enableAudio) {
      invalidateAudio = true
    }
    if (dyteMeetingInfo.enableVideo) {
      invalidateVideo = true
    }
    _init(
      MeetingInfo(
        dyteMeetingInfo.authToken,
        MeetingConfig(
          dyteMeetingInfo.enableAudio,
          dyteMeetingInfo.enableVideo,
          invalidateVideo,
          invalidateAudio
        ),
        dyteMeetingInfo.baseUrl
      ).apply {
        roomName = dyteMeetingInfo.roomName
      })
  }

  override suspend fun init(dyteMeetingInfoV2: DyteMeetingInfoV2) {
    println("DyteMobileClient | Controller init started")
    var invalidateAudio = false
    var invalidateVideo = false
    if (dyteMeetingInfoV2.enableAudio) {
      invalidateAudio = true
    }
    if (dyteMeetingInfoV2.enableVideo) {
      invalidateVideo = true
    }
    _init(
      MeetingInfo(
        dyteMeetingInfoV2.authToken,
        MeetingConfig(
          dyteMeetingInfoV2.enableAudio,
          dyteMeetingInfoV2.enableVideo,
          invalidateVideo,
          invalidateAudio
        ), dyteMeetingInfoV2.baseUrl
      )
    )
    println("DyteMobileClient | Controller init completed")
  }

  private suspend fun _init(meetingInfo: MeetingInfo) {
    try {
      _platformUtilsProvider.init(this@Controller)
      _eventController.triggerEvent(OnMeetingRoomInitStarted)

      _metaController = MetaController(meetingInfo, this@Controller)

      _loggerController = LoggerController(this@Controller)
      _loggerController.init()

      try {
        _metaController.init()
      } catch (e: Exception) {
        e.printStackTrace()
        loggerController.traceWarning("Invalid auth token")
        eventController.triggerEvent(
          OnMeetingRoomInitFailed(
            IllegalArgumentException(
              "Invalid auth token"
            )
          )
        )
      }
      _permissionController = getPermissionController(this@Controller, meetingInfo)
      doInit()
    } catch (e: Exception) {
      e.printStackTrace()
      println("Exception is raised in Controller.kt")
      eventController.triggerEvent(OnMeetingRoomInitFailed(e))
    }
  }

  private suspend fun getPermissionController(
    parent: Controller,
    meetingInfo: MeetingInfo
  ): PermissionController {
    val permissionController = PermissionController(parent)
    permissionController.init()
    val config = meetingInfo.config
    val audioEnabled = config.enableAudio
    val videoEnable = config.enableVideo
    if (audioEnabled || videoEnable) {
      permissionController.askPermissions(
        audioEnabled,
        videoEnable,
        {
          loggerController.traceLog("permissions granted")
        },
        {
          loggerController.traceLog("------permissions denied")
        }
      )
    }
    return permissionController
  }

  private suspend fun doInit() {
    println("DyteMobileClient | Controller doInit started")
    try {
      _platformUtilsProvider.getLogger().enableLogger(true)

      _connectionController = ConnectionController(this@Controller)
      _apiClient = ApiClient(this@Controller)


      val participantInfo = apiClient.getParticipantInfo()

      /*
          * Initializing the Feature Flags.
          * IMP: The shared API client is sending our Authorization header because of the default headers set.
          * We will need to pass a new instance of HttpClient but an ideal way would be to use the
          * shared instance only. In most of the Dyte API calls > 90%, we have manually added the
          * Authorization header, so the default header block inside the current HttpClient initialisation
          * can be removed.
          * */
      featureFlagService = FlagsmithService(
        loggerController = loggerController,
        environmentKey = FEATURE_FLAG_KEY,
        httpClient = _apiClient.getClient()
      )

      featureFlagService.initializeWithUserContext(
        identifier = getFeatureFlagIdentifier(
          featureFlagEntity = DyteFeatureFlagEntity.PEER,
          peerId = metaController.getPeerId()
        ),
        userAttributes = getFeatureFlagUserAttributes(
          featureFlagEntity = DyteFeatureFlagEntity.PEER,
          participantInfo = participantInfo,
          metaController = metaController,
          platformUtilsProvider = platformUtilsProvider
        )
      )

      loggerController.setPostInitInfo(
        participantInfo.organizationId,
        metaController.getRoomName()
      )
      _metaController.setParticipantInfo(participantInfo)
      _presetController = PresetController(participantInfo, this@Controller)

      /*
      * Initializing both SocketIO and SocketService sockets.
      * Needed as dependency for initialisation of Chat and Plugin features.
      * */
      _socketController = SocketController(this@Controller)
      _socketMessageResponseParser = SocketMessageResponseParser()
      _socketService = DefaultSocketService(
        baseUrl = provideSocketServiceBaseUrl(metaController.getBaseUrl()),
        peerId = _metaController.getPeerId(),
        roomName = _metaController.getRoomName(),
        authToken = _metaController.getAuthToken()
      )

      /*
      * Creating ChatController.
      * */
      val chatSocketServer =
        featureFlagService.getConfigValue(DyteFeatureFlags.CHAT_SOCKET_SERVER)
      println("DyteClient: Controller chatSocketServer -> $chatSocketServer")
      val chatSocketHandler = DefaultChatSocketHandler(
        chatSocketServer = chatSocketServer,
        roomNodeSocket = socketController,
        socketService = socketService,
        metaController = metaController,
        loggerController = loggerController,
        platformUtilsProvider = platformUtilsProvider,
        socketIoMessageResponseParser = socketMessageResponseParser
      )
      val presignedUrlApiService = PresignedUrlApiService(metaController.getBaseUrl())
      _chatController =
        ChatController(this@Controller, chatSocketHandler, presignedUrlApiService)

      _pollController = PollController(this@Controller)
      _hostController = HostController(this@Controller)
      _recordingController = RecordingController(this@Controller)
      _webinarController = WebinarController(this@Controller)

      loggerController.setPostInitInfo(
        participantInfo.organizationId,
        metaController.getRoomName()
      )
      _mediaSoupController = MediaSoupController(this@Controller)
      _presetController = PresetController(participantInfo, this@Controller)
      _waitlistController = WaitlistController(this@Controller)
      _participantController = ParticipantController(this@Controller)
      _selfController = SelfController(participantInfo, this@Controller)
      _pollController = PollController(this@Controller)
      _hostController = HostController(this@Controller)
      _recordingController = RecordingController(this@Controller)
      _pluginsController = PluginsController(this@Controller)
      _webinarController = WebinarController(this@Controller)
      _liveStreamController = LiveStreamController(this@Controller, socketService)
      _roomNodeController = RoomNodeController(this@Controller)

      recording = DyteRecording(this@Controller)
      poll = DytePoll(this@Controller)
      meta = DyteMeta(this@Controller)
      plugins = DytePlugins(this@Controller)
      webinar = DyteWebinar(this@Controller)
      liveStream = DyteLiveStream(this@Controller)
      chat = DyteChat(this@Controller)

      _selfController.init()

      withContext(Dispatchers.Main) {
        (platformUtilsProvider.getSFUUtils() as? IDyteMediaSoupUtils)?.init(this@Controller)
      }

      _mediaSoupController.init()
      _permissionController.init()
      _connectionController.init()
      _presetController.init()
      _participantController.init()
      _chatController.init()
      _pollController.init()
      _hostController.init()
      _webinarController.init()
      _recordingController.init()
      _liveStreamController.init()
      _pluginsController.init()
      _roomNodeController.init()
      _waitlistController.init()


      withContext(Dispatchers.Main) {
        println("DyteMobileClient | Controller doInit init start")
        platformUtilsProvider.getMediaUtils().init()
        println("DyteMobileClient | Controller doInit init complete")
      }

      _socketController.init()
      _socketController.connect()
      socketService.connect()
      eventController.triggerEvent(DyteEventType.OnMeetingRoomInitCompleted)

    } catch (e: Exception) {

      eventController.triggerEvent(OnMeetingRoomInitFailed(e))
      loggerController.traceError("do init crashed ${e.message}")
    }
  }

  override suspend fun joinRoom() {
    tempSendJoinRoomRequestToSocketService(
      _metaController.getPeerId(),
      _selfController.getSelf().name,
      _selfController.getSelf().userId
    )
    this.loggerController.traceLog("join room started")
    this.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
    roomNodeController.joinRoom()
  }

  override suspend fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit) {
    println("DyteMobileClient | Controller joinRoom start")
    joinRoom()
    println("DyteMobileClient | Controller joinRoom end")
  }

  override suspend fun leaveRoom() {
    eventController.triggerEvent(OnMeetingRoomLeaveStarted)
    platformUtilsProvider.getVideoUtils().destroyAll()
    platformUtilsProvider.getSFUUtils().leaveCall()
    socketController.disconnect()
    socketService.disconnect()
    eventController.triggerEvent(OnMeetingRoomLeave)
  }

  override suspend fun release() {
    platformUtilsProvider.getVideoUtils().destroyAll()
    platformUtilsProvider.getSFUUtils().leaveCall()
    socketController.disconnect()
    socketService.disconnect()
  }

  override fun getSelf(): DyteSelfParticipant {
    return this.selfController.getSelf()
  }

  override fun getMeetingParticipant(): DyteRoomParticipants {
    return this.participantController.meetingRoomParticipants
  }

  override fun getChat(): DyteChat {
    return chat
  }

  override fun getRecording(): DyteRecording {
    return recording
  }

  override fun getPolls(): DytePoll {
    return poll
  }

  override fun getMeta(): DyteMeta {
    return meta
  }

  override fun getPlugins(): DytePlugins {
    return plugins
  }

  override fun getWebinar(): DyteWebinar {
    return webinar
  }

  override fun getLiveStream(): DyteLiveStream {
    return liveStream
  }

  /**
   * Fetches roomUuid from Room node.
   */
  private suspend fun getRoomUuid(): String {
    try {
      val roomStateResponse = _socketController.sendMessage(GET_ROOM_STATE, null)
      val roomState =
        requireNotNull(
          socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel
        )

      return roomState.roomState?.roomUUID ?: error("Failed to fetch the room state.")
    } catch (e: Exception) {
      throw Exception("Invalid room state", e)
    }
  }

  // note(swapnil): This can later depend on some config. eg. project's BuildConfig
  private fun provideSocketServiceBaseUrl(apiBaseUrl: String): String {
    val socketServiceHost = if (apiBaseUrl.contains("staging.dyte")) {
      "socket-edge-staging.dyte.io"
    } else {
      "socket-edge.dyte.io"
    }

    return "wss://${socketServiceHost}"
  }

  /*
  note(swapnil): This method or step can be moved to some other initializer class once
  the roomUUID retrieval is made cleaner.
   */
  private suspend fun tempSendJoinRoomRequestToSocketService(
    peerId: String,
    displayName: String,
    userId: String
  ) {
    try {
      val roomUuid = getRoomUuid()
      /*
      * Note(swapnil): We are storing the roomUuid in the MetaController for using it inside
      * ChatController and PluginsController. Ideal way would be to fetch the roomUuid before
      * initialisation of the feature modules/controllers.
      *
      * WARNING: This is a temporary hack to make the roomUuid available to feature modules.
      * Doing this after connecting to sockets in doInit() was breaking on iOS.
      * TODO: Improve the initialization order.
      * */
      metaController.setRoomUuid(roomUuid)
      val joinRoomRequest = JoinRoomRequest(
        peer = Peer(peer_id = peerId, display_name = displayName, user_id = userId),
        room_uuid = metaController.getRoomUuid()
      )

      val response = socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
        payload = JoinRoomRequest.ADAPTER.encode(joinRoomRequest)
      )
      println("DyteClient: Controller | Socket requestResponse -> $response")
    } catch (e: Exception) {
      println("DyteClient: SocketService -> join room request error: $e")
    }
  }

  private fun getFeatureFlagIdentifier(featureFlagEntity: String, peerId: String): String {
    return "${featureFlagEntity}_${peerId}"
  }

  private fun getFeatureFlagUserAttributes(
    featureFlagEntity: String,
    participantInfo: ParticipantInfo,
    metaController: IMetaController,
    platformUtilsProvider: IDytePlatformUtilsProvider
  ): Map<String, Any> {
    return mapOf(
      UserAttributeKeys.ENTITY to featureFlagEntity,
      UserAttributeKeys.CLIENT_ID to participantInfo.organizationId,
      UserAttributeKeys.ROOM_NAME to metaController.getRoomName(),
      UserAttributeKeys.ANONYMOUS_USER to participantInfo.organizationId.isBlank(),
      UserAttributeKeys.SDK_VERSION to BuildKonfig.sdkVersion,
      UserAttributeKeys.IS_MOBILE to true,
      UserAttributeKeys.OS_NAME to platformUtilsProvider.getPlatformUtils().getOsName(),
      UserAttributeKeys.OS_VERSION_NAME to platformUtilsProvider.getPlatformUtils()
        .getOsVersion()
    )
  }

  companion object {
    private const val FEATURE_FLAG_KEY = "hXgU8Wc8pwuGNq9ms5q9Hh"
  }
}

internal interface IController {
  suspend fun init(dyteMeetingInfoV2: DyteMeetingInfoV2)
  suspend fun init(dyteMeetingInfo: DyteMeetingInfo)
  suspend fun joinRoom()
  suspend fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit)
  suspend fun leaveRoom()

  suspend fun release()

  fun getSelf(): DyteSelfParticipant
  fun getRecording(): DyteRecording

  fun getMeetingParticipant(): DyteRoomParticipants
  fun getChat(): DyteChat
  fun getPolls(): DytePoll
  fun getMeta(): DyteMeta
  fun getPlugins(): DytePlugins
  fun getWebinar(): DyteWebinar
  fun getLiveStream(): DyteLiveStream
}

internal interface IControllerContainer {
  val chatController: IChatController
  val participantController: IParticipantController
  val pollsController: IPollController
  val metaController: IMetaController
  val selfController: ISelfController
  val roomNodeController: IRoomNodeController
  val presetController: IPresetController
  val hostController: IHostController
  val apiClient: IApiClient
  val socketService: SocketService
  val socketController: ISocketController
  val socketMessageResponseParser: ISocketMessageResponseParser
  val platformUtilsProvider: IDytePlatformUtilsProvider
  val recordingController: IRecordingController
  val eventController: IEventController
  val connectionController: IConnectionController
  val permissionController: IPermissionController
  val pluginsController: IPluginsController
  val loggerController: ILoggerController
  val mediaSoupController: IMediaSoupController
  val webinarController: IWebinarController
  val liveStreamController: ILiveStreamController
  val waitlistController: IWaitlistController
}