package io.dyte.core.platform

import android.app.Application
import android.content.Context
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import org.webrtc.AudioSource
import org.webrtc.Camera1Enumerator
import org.webrtc.Camera2Enumerator
import org.webrtc.CameraEnumerator
import org.webrtc.CameraVideoCapturer
import org.webrtc.CameraVideoCapturer.CameraEventsHandler
import org.webrtc.DefaultVideoDecoderFactory
import org.webrtc.DefaultVideoEncoderFactory
import org.webrtc.EglBase
import org.webrtc.MediaConstraints
import org.webrtc.PeerConnectionFactory
import org.webrtc.SurfaceTextureHelper
import org.webrtc.VideoDecoderFactory
import org.webrtc.VideoEncoderFactory
import org.webrtc.VideoSource
import org.webrtc.VideoTrack
import org.webrtc.audio.AudioDeviceModule
import org.webrtc.audio.JavaAudioDeviceModule
import org.webrtc.audio.JavaAudioDeviceModule.AudioRecordErrorCallback
import org.webrtc.audio.JavaAudioDeviceModule.AudioTrackErrorCallback

internal class DyteAndroidPeerConnection(private val utilsProvider: IDytePlatformUtilsProvider) :
  IDytePeerConnectionUtils {
  private var mPeerConnectionFactory: PeerConnectionFactory? = null
  private var mAudioSource: AudioSource? = null
  private var mVideoSource: VideoSource? = null
  private var mCamCapture: CameraVideoCapturer? = null

  private var eglBase: EglBase = EglBase.create()

  private var selfPreviewTrack: VideoTrack? = null

  init {
    // utilsProvider.getPlatformUtils().printThread("DyteMobileClient | DyteAndroidPeerConnection init ")
  }

  override fun switchCamera() {
    utilsProvider.getControllerContainer().loggerController.traceLog("DyteAndroidPeerConnection - switchCamera ")
    requireNotNull(mCamCapture).switchCamera(null)
  }

  override fun createAudioTrack(id: String): Any? {
    utilsProvider.getPlatformUtils().printThread("DyteMobileClient | DyteAndroidPeerConnection createAudioTrack ")
    utilsProvider.getControllerContainer().loggerController.traceLog("DyteAndroidPeerConnection - createAudioTrack ")
    if (utilsProvider.getControllerContainer().permissionController.isPermissionGrated(MICROPHONE)
        .not()
    ) {
      return null
    }
    if (utilsProvider.getControllerContainer().presetController.canPublishAudio().not()) {
      return null
    }
    if (mAudioSource == null) {
      createAudioSource(
        utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application
      )
    }
    return requireNotNull(mPeerConnectionFactory).createAudioTrack(id, mAudioSource)
  }

  override fun createVideoTrack(id: String): Any? {
    utilsProvider.getControllerContainer().loggerController.traceLog("DyteAndroidPeerConnection createVideoTrack")
    if (utilsProvider.getControllerContainer().permissionController.isPermissionGrated(CAMERA)
        .not()
    ) {
      return null
    }

    if (utilsProvider.getControllerContainer().presetController.canPublishVideo().not()) {
      return null
    }

    if (mVideoSource == null) {
      createVideoSource(
        utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application
      )
    }
    if (selfPreviewTrack == null) {
      selfPreviewTrack = requireNotNull(mPeerConnectionFactory).createVideoTrack(id, mVideoSource)
    }
    return requireNotNull(selfPreviewTrack)
  }

  override fun getEglContext(): EglBase.Context {
    return eglBase.eglBaseContext
  }

  override fun dispose() {
    if (mCamCapture != null) {
      try {
        mCamCapture?.stopCapture()
        mCamCapture?.dispose()
        mCamCapture = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mVideoSource != null) {
      try {
        mVideoSource?.dispose()
        mVideoSource = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mAudioSource != null) {
      try {
        mAudioSource?.dispose()
        mAudioSource = null
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
    if (mPeerConnectionFactory != null) {
      mPeerConnectionFactory?.dispose()
      mPeerConnectionFactory = null
    }

    releaseEglContext()
  }

  override fun onCameraClosed() {
    mCamCapture = null
    mVideoSource = null
  }

  override fun stopVideo() {
    selfPreviewTrack?.setEnabled(false)
  }

  override fun resumeVideo() {
    selfPreviewTrack?.setEnabled(true)
    // mCamCapture?.startCapture(640, 480, 30)
  }

  private fun createPeerConnectionFactory(context: Context) {
    utilsProvider.getControllerContainer().loggerController.traceLog("create peer connection factory")
    val builder = PeerConnectionFactory.builder()
    builder.setOptions(null)
    val adm = createJavaAudioDevice(context)
    val encoderFactory: VideoEncoderFactory = DefaultVideoEncoderFactory(
      getEglContext(), true /* enableIntelVp8Encoder */, true
    )
    val decoderFactory: VideoDecoderFactory =
      DefaultVideoDecoderFactory(getEglContext())

    val peerConnectionFactoryOpptionsBuilder =
      PeerConnectionFactory.InitializationOptions
        .builder(utilsProvider.getPlatformUtils().getAndroidApplicationContext() as Application)
        .setNativeLibraryName("mediasoupclient_so")
        .createInitializationOptions()

    PeerConnectionFactory.initialize(peerConnectionFactoryOpptionsBuilder)

    mPeerConnectionFactory = builder
      .setAudioDeviceModule(adm)
      .setVideoEncoderFactory(encoderFactory)
      .setVideoDecoderFactory(decoderFactory)
      .createPeerConnectionFactory()
  }

  private fun createJavaAudioDevice(appContext: Context): AudioDeviceModule {
    utilsProvider.getControllerContainer().loggerController.traceLog("create java audio device")
    val audioRecordErrorCallback: AudioRecordErrorCallback = object : AudioRecordErrorCallback {
      override fun onWebRtcAudioRecordInitError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record init error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneInitError(
            errorMessage
          )
        )
      }

      override fun onWebRtcAudioRecordStartError(
        errorCode: JavaAudioDeviceModule.AudioRecordStartErrorCode,
        errorMessage: String
      ) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record start error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneStartError(
            errorMessage
          )
        )
      }

      override fun onWebRtcAudioRecordError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio record error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneError(
            errorMessage
          )
        )
      }
    }
    val audioTrackErrorCallback: AudioTrackErrorCallback = object : AudioTrackErrorCallback {
      override fun onWebRtcAudioTrackInitError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track init error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneInitError(
            errorMessage
          )
        )
      }

      override fun onWebRtcAudioTrackStartError(
        errorCode: JavaAudioDeviceModule.AudioTrackStartErrorCode,
        errorMessage: String
      ) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track start error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneStartError(
            errorMessage
          )
        )
      }

      override fun onWebRtcAudioTrackError(errorMessage: String) {
        utilsProvider.getControllerContainer().loggerController.traceError("web rtc audio track error $errorMessage")
        utilsProvider.getControllerContainer().eventController.triggerEvent(
          DyteEventType.OnMicrophoneError(
            errorMessage
          )
        )
      }
    }
    return JavaAudioDeviceModule.builder(appContext)
      .setAudioRecordErrorCallback(audioRecordErrorCallback)
      .setAudioTrackErrorCallback(audioTrackErrorCallback)
      .createAudioDeviceModule()
  }

  private fun createAudioSource(context: Context) {
    utilsProvider.getControllerContainer().loggerController.traceLog("DyteAndroidPeerConnection - createAudioSource ")
    if (mPeerConnectionFactory == null) {
      createPeerConnectionFactory(context)
    }
    mAudioSource = mPeerConnectionFactory!!.createAudioSource(MediaConstraints())
  }

  private fun createCamCapture(context: Context) {
    utilsProvider.getControllerContainer().loggerController.traceLog("createCamCapture start")
    val isCamera2Supported = Camera2Enumerator.isSupported(context)
    val cameraEnumerator: CameraEnumerator = if (isCamera2Supported) {
      utilsProvider.getControllerContainer().loggerController.traceLog("createCamCapture camera 2 enumerator")
      Camera2Enumerator(context)
    } else {
      utilsProvider.getControllerContainer().loggerController.traceLog("createCamCapture camera 1 enumerator")
      Camera1Enumerator()
    }
    mCamCapture = cameraEnumerator.createCapturer(
      cameraEnumerator.deviceNames.last(), // TODO fix this
      object : CameraEventsHandler {
        override fun onCameraError(s: String) {
          utilsProvider.getControllerContainer().loggerController.traceError("camera error $s")
          utilsProvider.getControllerContainer().eventController.triggerEvent(
            DyteEventType.OnCameraError(
              s
            )
          )
        }

        override fun onCameraDisconnected() {
          utilsProvider.getControllerContainer().loggerController.traceLog("on camera disconnected")
          utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnCameraDisconnected)
        }

        override fun onCameraFreezed(s: String) {
          utilsProvider.getControllerContainer().loggerController.traceLog("on camera freezed")
          utilsProvider.getControllerContainer().eventController.triggerEvent(
            DyteEventType.OnCameraFreezed(
              s
            )
          )
        }

        override fun onCameraOpening(s: String) {
          utilsProvider.getControllerContainer().loggerController.traceLog("on camera opening $s")
          utilsProvider.getControllerContainer().eventController.triggerEvent(
            DyteEventType.OnCameraOpening(
              s
            )
          )
        }

        override fun onFirstFrameAvailable() {
          utilsProvider.getControllerContainer().loggerController.traceLog("on first frame available")
          utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnFirstFrameAvailable)
        }

        override fun onCameraClosed() {
          utilsProvider.getControllerContainer().loggerController.traceLog("on camera closed")
          utilsProvider.getControllerContainer().eventController.triggerEvent(DyteEventType.OnCameraClosed)
        }
      })

    if (mCamCapture == null) {
      utilsProvider.getControllerContainer().loggerController.traceError("can not create mCamCapture")
    }
  }

  private fun createVideoSource(context: Context) {
    utilsProvider.getControllerContainer().loggerController.traceLog("createVideoSource start")
    if (mPeerConnectionFactory == null) {
      createPeerConnectionFactory(context)
    }
    if (mCamCapture == null) {
      createCamCapture(context)
    }
    mVideoSource = mPeerConnectionFactory!!.createVideoSource(false)
    val surfaceTextureHelper =
      SurfaceTextureHelper.create("CaptureThread", getEglContext())
    mCamCapture!!.initialize(surfaceTextureHelper, context, mVideoSource?.capturerObserver)
    mCamCapture!!.startCapture(640, 480, 24)
    utilsProvider.getControllerContainer().loggerController.traceLog("createVideoSource end")
  }

  override fun releaseEglContext() {
    // eglBase.release()
    // eglBase = EglBase.create()
  }

  override fun createVideoView(id: String): Any {
    return 1
  }
}