package io.dyte.core.socket.socketservice

import io.dyte.core.observability.ILoggerController
import io.dyte.core.observability.LoggerController
import io.dyte.sockrates.client.*
import io.dyte.sockrates.client.ResponseException as SockratesResponseException
import io.dyte.sockrates.client.ResponseTimeoutException as SockratesResponseTimeoutException
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import message.v1.SocketMessage
import kotlin.coroutines.cancellation.CancellationException
import kotlin.random.Random

interface SocketService {
  suspend fun connect()

  suspend fun disconnect()

  suspend fun send(event: Int, payload: ByteArray?)

  @Throws(ResponseTimeoutException::class, CancellationException::class)
  suspend fun requestResponse(event: Int, payload: ByteArray?): ByteArray?

  fun subscribe(event: Int, listener: SocketServiceEventListener)

  fun unsubscribe(event: Int, listener: SocketServiceEventListener)

  fun addConnectionStateListener(listener: SocketServiceConnectionStateListener)

  fun removeConnectionStateListener(listener: SocketServiceConnectionStateListener)
}

interface SocketServiceEventListener {
  fun onEvent(event: Int, payload: ByteArray?)
}

interface SocketServiceConnectionStateListener {
  fun onConnectionStateChanged(newState: SocketServiceConnectionState)
}

class ResponseTimeoutException(override val message: String) : RuntimeException(message)

internal class DefaultSocketService(
  baseUrl: String,
  private val peerId: String,
  roomName: String,
  authToken: String,
  private val loggerController: ILoggerController,
  private val workContext: CoroutineDispatcher = Dispatchers.Default
) : SocketService {
  private val url: String
  private val wsClient: Sockrates
  // private val mainScope = MainScope()

  private val activeEventListeners: MutableMap<Int, MutableSet<SocketServiceEventListener>> =
    HashMap()

  private val connectionStateListeners: MutableSet<SocketServiceConnectionStateListener> =
    LinkedHashSet()

  init {
    url = createSocketServiceUrl(baseUrl, peerId, roomName, authToken)
    wsClient = Sockrates(url, config = SockratesConfiguration(disconnectOnPingTimeout = false))
  }

  override suspend fun connect() {
    withContext(workContext) {
      wsClient.connect(object : SockratesWSListener {
        override fun onClosed(client: Sockrates, code: Int, reason: String) {
          super.onClosed(client, code, reason)
          loggerController.traceLog("DyteMobileClient: SocketService onClosed[$reason]")
        }

        override fun onConnectionStateChanged(
          client: Sockrates,
          newState: WebSocketConnectionState
        ) {
          super.onConnectionStateChanged(client, newState)
          loggerController.traceLog("DyteMobileClient: SocketService onConnectionStateChanged[$newState]")
          try {
            val socketServiceConnectionState =
              SocketServiceConnectionState.createFromSockratesConnectionState(newState)
            notifyConnectionStateListeners(socketServiceConnectionState)
          } catch (e: IllegalArgumentException) {
            loggerController.traceError("DyteMobileClient: SocketService onConnectionStateChanged exception -> ${e.message}")
          }
        }

        override fun onFailure(client: Sockrates, exception: Exception, reason: String) {
          super.onFailure(client, exception, reason)
          loggerController.traceError("DyteMobileClient: SocketService onFailure[$reason]")
        }

        override fun onMessage(client: Sockrates, message: WebSocketMessage) {
          super.onMessage(client, message)
          loggerController.traceLog("DyteMobileClient: SocketService onMessage[$message]")
        }

        override fun onMessage(client: Sockrates, message: SocketMessage) {
          super.onMessage(client, message)
          loggerController.traceLog("DyteMobileClient: SocketService onMessage[$message]")
          notifyEventSubscribers(message)
        }

        override fun onOpen(client: Sockrates) {
          super.onOpen(client)
          loggerController.traceLog("DyteMobileClient: SocketService onOpen")
        }
      })
    }
  }

  override suspend fun disconnect() {
    withContext(workContext) {
      wsClient.disconnect()
      clear()
    }
  }

  override suspend fun send(event: Int, payload: ByteArray?) {
    withContext(workContext) {
      val messageId = generateMessageId(peerId)
      wsClient.send(event, messageId, payload)
      println("DyteMobileClient: SocketService send")
    }
  }

  override suspend fun requestResponse(event: Int, payload: ByteArray?): ByteArray? {
    return withContext(workContext) {
      val messageId = generateMessageId(peerId)
      try {
        val response = wsClient.requestResponse(event, messageId, payload)
        loggerController.traceLog("DyteMobileClient: SocketService requestResponse[$response]")
        response.payload?.toByteArray()
      } catch (e: SockratesResponseTimeoutException) {
        loggerController.traceWarning("DyteMobileClient: SocketService requestResponse timeout exception -> [${e.message}]")
        throw ResponseTimeoutException(e.message)
      } catch (e: SockratesResponseException) {
        loggerController.traceWarning("DyteMobileClient: SocketService requestResponse exception -> [${e.message}]")
        null
      }
    }
  }

  override fun subscribe(event: Int, listener: SocketServiceEventListener) {
    val existingEventListeners = activeEventListeners[event]
    if (existingEventListeners == null) {
      val newEventListenersSet = LinkedHashSet<SocketServiceEventListener>()
      newEventListenersSet.add(listener)
      activeEventListeners[event] = newEventListenersSet
    } else {
      existingEventListeners.add(listener)
    }
  }

  override fun unsubscribe(event: Int, listener: SocketServiceEventListener) {
    val existingEventListeners = activeEventListeners[event]
    if (existingEventListeners != null) {
      existingEventListeners.remove(listener)
    }
  }

  override fun addConnectionStateListener(listener: SocketServiceConnectionStateListener) {
    connectionStateListeners.add(listener)
  }

  override fun removeConnectionStateListener(listener: SocketServiceConnectionStateListener) {
    connectionStateListeners.remove(listener)
  }

  private fun notifyEventSubscribers(socketMessage: SocketMessage) {
    if (activeEventListeners.isEmpty()) return

    val eventListeners = activeEventListeners[socketMessage.event]
    if (eventListeners != null && eventListeners.isNotEmpty()) {
      for (listener in eventListeners) {
        listener.onEvent(socketMessage.event, socketMessage.payload?.toByteArray())
      }
    }
  }

  private fun notifyConnectionStateListeners(connectionState: SocketServiceConnectionState) {
    connectionStateListeners.forEach {
      it.onConnectionStateChanged(connectionState)
    }
  }

  private fun clear() {
    activeEventListeners.clear()
    connectionStateListeners.clear()
  }

  companion object {
    private fun createSocketServiceUrl(
      baseUrl: String,
      peerId: String,
      roomName: String,
      authToken: String
    ): String {
      return "$baseUrl/ws?roomID=${roomName}&peerID=${peerId}&authToken=${authToken}"
    }

    private fun generateMessageId(peerId: String): String {
      val randomSuffix = generateRandomString(5)
      return "${peerId}-${randomSuffix}"
    }

    private val characterSet = charArrayOf(
      '0', '1', '2', '3', '4', '5',
      '6', '7', '8', '9', 'a', 'b',
      'c', 'd', 'e', 'f', 'g', 'h',
      'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't',
      'u', 'v', 'w', 'x', 'y', 'z'
    )

    /*
    This is not finalised yet. Currently using the most straightforward method to
    generate a random 5 character alphanumeric string.
    Will need to come-up with a better efficient method or implement Base36 encoding in
    Kotlin.
     */
    private fun generateRandomString(idLength: Int = 5): String {
      val idBuilder: StringBuilder = StringBuilder()
      var randomIndex = 0
      for (i in 0 until idLength) {
        randomIndex = (Random.nextInt(until = characterSet.size))
        idBuilder.append(characterSet[randomIndex])
      }

      return idBuilder.toString()
    }
  }
}