package io.dyte.core.feat

import io.dyte.core.ThreadSafeMap
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.platform.DytePluginFile
import io.dyte.core.platform.IDyteWebView
import io.dyte.core.platform.IDyteWebViewListener
import io.dyte.core.platform.WebView
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonObject

/**
 * Dyte Plugin
 *
 * @property controllerContainer
 * @constructor Create un-activated Dyte Plugin
 */

class DytePlugin internal constructor(
  val id: String,
  val name: String,
  val description: String,
  val picture: String,
  val isPrivate: Boolean,
  val staggered: Boolean,
  val baseURL: String,
  private val controller: IControllerContainer
) {

  private var enabledBy: String? = null

  companion object {
    private const val KEY_DEFAULT_WEBVIEW = "default"
  }

  private var config: PluginConfigResponse? = null

  private var authToken: String? = null
  private var urlToLoad: String? = null
  private val webViews: ThreadSafeMap<String, IDyteWebView> = ThreadSafeMap()

  private var _isActive: Boolean = false
  val isActive: Boolean
    get() = _isActive

  private val webViewListener: IDyteWebViewListener = object : IDyteWebViewListener {

    override fun onWebViewMessage(message: JsonObject) {
      controller.pluginsController.onPluginWebViewMessage(id, message)
    }

    override fun onFileRequest() {
      controller.pluginsController.onPluginFileRequest(this@DytePlugin)
    }
  }

  /**
   * Submits the event to the Plugin SDK through WebView.
   *
   * @param payload event JSON
   */
  internal fun submitEvent(payload: JsonObject) {
    webViews[KEY_DEFAULT_WEBVIEW]?.submitEvent(payload)
  }

  /**
   * Activate this plugin for all participants.
   */

  fun activate(callBack: () -> Unit) {
    controller.loggerController.traceLog("DytePlugin : $name :  activate")
    if (_isActive.not()) {
      controller.pluginsController.activatePlugin(this@DytePlugin)
      callBack()
    } else {
      callBack()
    }
  }

  /**
   * Deactivate this plugin for all participants.
   */
  fun deactivate(callBack: () -> Unit) {
    controller.loggerController.traceLog("DytePlugin : $name :  deactivate")
    if (isActive) {
      controller.pluginsController.deactivatePlugin(this@DytePlugin)
      callBack()
    } else {
      callBack()
    }
  }

  fun getPluginView(): WebView {
    return webViews.getOrElse(KEY_DEFAULT_WEBVIEW) {
      val webView = controller.platformUtilsProvider.createWebView()
      webView.initializeWebView()
      webView.setListener(webViewListener)
      urlToLoad?.let {
        webView.loadURL(it)
      }
      webViews[KEY_DEFAULT_WEBVIEW] = webView
      webViews.getValue(KEY_DEFAULT_WEBVIEW)
    }.getWebView()
  }

  private fun removePluginView() {
    webViews.remove(KEY_DEFAULT_WEBVIEW)
  }

  /**
   * Uploads selected file to Plugin.
   */
  fun uploadFile(file: DytePluginFile) {
    webViews[KEY_DEFAULT_WEBVIEW]?.uploadFile(file)
  }

  /**
   * Enable this plugin for the local user.
   */
  internal fun enableLocal(
    authToken: String,
    url: String,
    enabledBy: String
  ) {
    if (!_isActive) {
      this@DytePlugin.authToken = authToken
      this@DytePlugin.enabledBy = enabledBy
      this@DytePlugin.urlToLoad = url
      _isActive = true
    }
  }

  /**
   * Disable this plugin for the local user.
   */
  internal fun disableLocal() {
    if (_isActive) {
      webViews[KEY_DEFAULT_WEBVIEW]?.removeListener()
      removePluginView()
      _isActive = false
    }
  }
}