package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class SelfController(
  private val participantInfo: ParticipantInfo,
  controllerContainer: IControllerContainer
) : ISelfController, BaseController(controllerContainer) {
  private lateinit var selfParticipant: DyteSelfParticipant

  private var _roomJoined = false
  override val roomJoined: Boolean
    get() = _roomJoined

  override fun init() {
    this.selfParticipant =
      DyteSelfParticipant(
        controllerContainer.metaController.getPeerId(),
        requireNotNull(participantInfo.id),
        controllerContainer.metaController.getDisplayName(),
        participantInfo.picture,
        false,
        participantInfo.clientSpecificId,
        ParticipantFlags(controllerContainer.presetController.canRecord(), false),
        controllerContainer
      )
  }

  override fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  override fun disableAudio() {
    controllerContainer.loggerController.traceLog("disable video")
    serialScope.launch {
      if (_roomJoined) {
        controllerContainer.platformUtilsProvider.getSFUUtils().muteSelfAudio()
        controllerContainer.socketController.sendMessage(MUTE_SELF_AUDIO, null)
      } else {
        selfParticipant._audioEnabled = false
        controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
      }
    }
  }

  override fun onAudioDisabled() {
    selfParticipant._audioEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun enableAudio() {
    controllerContainer.loggerController.traceLog("enable audio")
    if (!controllerContainer.permissionController.isPermissionGrated(MICROPHONE)) {
      return
    }
    if (!controllerContainer.presetController.canPublishAudio()) {
      return
    }
    if (!controllerContainer.metaController.isAudioEnabled()) {
      return
    }

    if (_roomJoined) {
      controllerContainer.platformUtilsProvider.getSFUUtils().unmuteSelfAudio()
      controllerContainer.socketController.sendMessage(UN_MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    }
  }

  override fun onAudioEnabled() {
    selfParticipant._audioEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
  }

  override fun disableVideo() {
    controllerContainer.loggerController.traceLog("disable video")
    serialScope.launch {
      if (_roomJoined) {
        controllerContainer.platformUtilsProvider.getSFUUtils().muteSelfVideo()
        controllerContainer.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
        val producerId =
          controllerContainer.mediaSoupController.videoProduceId
        producerId?.let {
          val content = HashMap<String, JsonElement>()
          content["producerId"] = JsonPrimitive(producerId)
          controllerContainer.socketController.sendMessage(MUTE_VIDEO, JsonObject(content))
        } ?: run {
          controllerContainer.loggerController.traceWarning("no producer id to disable video")
        }
      } else {
        controllerContainer.platformUtilsProvider.getPeerConnectionUtils().stopVideo()
        selfParticipant._videoEnabled = false
        controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
      }
    }
  }

  override fun onVideoDisabled() {
    selfParticipant._videoEnabled = false
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun onVideoEnabled(videoProducerId: String) {
    selfParticipant._videoEnabled = true
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun enableVideo() {
    controllerContainer.loggerController.traceLog("enable video")
    if (!controllerContainer.permissionController.isPermissionGrated(CAMERA)) {
      return
    }
    if (!controllerContainer.presetController.canPublishVideo()) {
      return
    }
    if (!controllerContainer.metaController.isVideoEnabled()) {
      return
    }

    if (roomJoined) {
      controllerContainer.mediaSoupController.produceVideo()
      controllerContainer.platformUtilsProvider.getSFUUtils().unmuteSelfVideo()
    }
    selfParticipant._videoEnabled = true
    controllerContainer.platformUtilsProvider.getPeerConnectionUtils().resumeVideo()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override fun setDevice(dyteAudioDevice: DyteAudioDevice) {
    controllerContainer.platformUtilsProvider.getMediaUtils().setAudioDevice(dyteAudioDevice)
  }

  override fun getAudioDevices(): List<DyteAudioDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getAudioDevices()
  }

  override fun getSelectedAudioDevice(): DyteAudioDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedAudioDevice()
  }

  override fun onAudioDevicesChanged() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnAudioDevicesChanged)
  }

  override fun getVideoDevices(): List<DyteVideoDevice> {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getVideoDevices()
  }

  override fun setDevice(dyteVideoDevice: DyteVideoDevice) {
    controllerContainer.loggerController.traceLog("set video device ${dyteVideoDevice.type.displayName}")
    controllerContainer.platformUtilsProvider.getMediaUtils().setVideoDevice(dyteVideoDevice)
  }

  override fun getSelectedVideoDevice(): DyteVideoDevice {
    return controllerContainer.platformUtilsProvider.getMediaUtils().getSelectedVideoDevice()
  }

  override fun onVideoDevicesChanged() {
    TODO("Not yet implemented")
  }

  override fun onEnteredInRoom() {
    _roomJoined = true
  }

  override fun onUpdate() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfUpdate)
  }

  override fun onRemovedFromMeeting() {
    controllerContainer.loggerController.traceLog("removed from meeting")
    serialScope.launch {
      controllerContainer.socketController.disconnect()
      controllerContainer.socketService.disconnect()
    }

    controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
    controllerContainer.platformUtilsProvider.getSFUUtils().leaveCall()

    controllerContainer.eventController.triggerEvent(DyteEventType.OnHostKicked)
  }
}

interface ISelfController {
  val roomJoined: Boolean
  fun getSelf(): DyteSelfParticipant
  fun disableAudio()
  fun onAudioDisabled()
  fun enableAudio()
  fun onAudioEnabled()
  fun disableVideo()
  fun onVideoDisabled()
  fun enableVideo()
  fun onVideoEnabled(videoProducerId: String)

  fun getAudioDevices(): List<DyteAudioDevice>
  fun setDevice(dyteAudioDevice: DyteAudioDevice)
  fun getSelectedAudioDevice(): DyteAudioDevice
  fun onAudioDevicesChanged()
  fun getVideoDevices(): List<DyteVideoDevice>
  fun setDevice(dyteVideoDevice: DyteVideoDevice)
  fun getSelectedVideoDevice(): DyteVideoDevice
  fun onVideoDevicesChanged()
  fun onEnteredInRoom()
  fun onUpdate()
  fun onRemovedFromMeeting()
}