package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE

internal class PermissionController(controller: IControllerContainer) : IPermissionController,
  BaseController(controller) {

  override val permissions: List<Permission>
    get() = _permissions.toList()

  private val _permissions = arrayListOf<Permission>()

  private var isNotified = false

  override fun init() {
    super.init()

    /*askPermissions(true, true, {
      //no-op
    }, {
      //no-op
    })*/
  }

  override fun isPermissionGrated(permissionType: PermissionType): Boolean {
    return controllerContainer.platformUtilsProvider.getPermissionUtils().isPermissionGranted(Permission(permissionType, false))
  }

  override fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  ) {
    isNotified = false
    val permissionsToAsk = arrayListOf<PermissionType>()
    if (videoEnabled) {
      controllerContainer.loggerController.traceLog("adding camera permission to ask")
      permissionsToAsk.add(CAMERA)
    }
    if (audioEnabled) {
      controllerContainer.loggerController.traceLog("adding microphone permission to ask")
      permissionsToAsk.add(MICROPHONE)
    }

    _permissions.clear()
    permissionsToAsk.forEach {
      _permissions.add(Permission(it, false))
    }

    controllerContainer.platformUtilsProvider.getPermissionUtils()
      .askPermissions(_permissions, {
        controllerContainer.loggerController.traceLog("we've both permission")
        if (isNotified.not()) {
          onPermissionGranted.invoke()
          isNotified = true
        }
      }, {
        controllerContainer.loggerController.traceLog("we've some denied permissions")
        if(isNotified.not()) {
          onPermissionDenied.invoke()
          isNotified = true
        }
      })
  }
}

interface IPermissionController {
  val permissions: List<Permission>
  fun isPermissionGrated(permissionType: PermissionType): Boolean
  fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  )
}

data class Permission(
  val type: PermissionType,
  var isGranted: Boolean
)

enum class PermissionType(val displayName: String) {
  CAMERA("Camera"),
  MICROPHONE("Mic"),
  STORAGE("Storage");
}