package io.dyte.core.observability

import defaults.BuildKonfig
import io.dyte.core.controllers.BaseController
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.observability.LogSeverity.error
import io.dyte.core.observability.LogSeverity.log
import io.dyte.core.observability.LogSeverity.warn
import io.dyte.core.observability.LogStatus.NONE
import io.dyte.core.observability.LogStatus.PROCESSIGN
import io.dyte.core.observability.LogStatus.PUSHED
import kotlinx.coroutines.async
import kotlinx.coroutines.delay
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

internal class LoggerController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), ILoggerController {
  private var logs: ArrayList<DyteLog> = arrayListOf()

  private lateinit var meetingMetaData: MeetingMetaData
  private var roomName: String = "no-room"

  private var logId = 0

  // TODO create util class timer
  private suspend fun startTimerContinuous(tickInMillis: Long) {
    while (true) {
      processLogs()
      delay(tickInMillis)
    }
  }

  private suspend fun processLogs(shouldPushAll: Boolean = false) {
    val logArray = ArrayList(logs)
    val logsToBeProcessed =
      logArray.filter { it.status == NONE }.take(if (shouldPushAll) Int.MAX_VALUE else 20)
    if (logsToBeProcessed.isEmpty()) {
      return
    }

    logsToBeProcessed.map { it.status = PROCESSIGN }
    try {
      controllerContainer.apiClient.uploadLogs(
        PostDyteLogs(
          meetingMetaData,
          logsToBeProcessed,
          "mobile-core"
        )
      )
      logsToBeProcessed.forEach { processedLog ->
        logArray.find { it.id == processedLog.id }!!.status = PUSHED
      }
    } catch (e: Exception) {
      logsToBeProcessed.forEach { processedLog ->
        logArray.find { it.id == processedLog.id }!!.status = NONE
      }
    }
  }

  override fun init() {
    meetingMetaData = MeetingMetaData(
      controllerContainer.metaController.getPeerId(),
      roomName,
      "no-org",
      BuildKonfig.sdkVersion,
      true,
      controllerContainer.platformUtilsProvider.getPlatformUtils().getDeviceInfo()
    )
    serialScope.async {
      startTimerContinuous(7000)
    }
  }



  override fun traceLog(message: String) {
    trace(message, log)
  }

  override fun traceWarning(message: String) {
    trace(message, warn)
  }

  override fun traceError(message: String) {
    trace(message, error)
  }

  private fun trace(message: String, severity: LogSeverity) {
    logs.add(
      DyteLog(
        logId,
        message,
        controllerContainer.platformUtilsProvider.getPlatformUtils().getUtcTimeNow(),
        NONE,
        severity
      )
    )
    logId++
  }

  override suspend fun pushAllNow() {
    processLogs(true)
  }

  override fun setPostInitInfo(organizationId: String, roomName: String) {
    meetingMetaData = meetingMetaData.copy(organizationId = organizationId, roomName = roomName)
  }
}

@Serializable
data class DyteLog(
  val id: Int,
  val message: String,
  val loggedAt: String,
  var status: LogStatus,
  val logSeverity: LogSeverity
)

@Serializable
data class PostDyteLogs(
  val meetingMetadata: MeetingMetaData,
  val logs: List<DyteLog>,
  val serviceName: String
)

@Serializable
data class MeetingMetaData(
  @SerialName("meetingMetadata.peerId") val peerId: String,
  @SerialName("meetingMetadata.roomName") val roomName: String,
  @SerialName("meetingMetadata.organizationId") val organizationId: String,
  @SerialName("meetingMetadata.sdkVersion") val sdkVersion: String,
  @SerialName("meetingMetadata.deviceInfo.isMobile") val isMobile: Boolean,
  @SerialName("meetingMetadata.deviceInfo.deviceInfo") val deviceInfo: String
)

enum class LogStatus {
  NONE,
  PROCESSIGN,
  PUSHED;
}

enum class LogSeverity {
  info,
  error,
  debug,
  log,
  warn;
}

interface ILoggerController {
  fun traceLog(message: String)
  fun traceWarning(message: String)
  fun traceError(message: String)

  fun setPostInitInfo(organizationId: String, roomName: String)

  suspend fun pushAllNow()
}