package io.dyte.core.controllers

import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.feat.DytePollVote
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_POLLS
import io.dyte.core.socket.events.OutboundMeetingEventType.NEW_POLL
import io.dyte.core.socket.events.OutboundMeetingEventType.VOTE_POLL
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPoll
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollsModel
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class PollController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IPollController {
  private var _polls: ArrayList<DytePollMessage> = arrayListOf()

  override val polls: List<DytePollMessage>
    get() = _polls

  override fun loadPolls() {
    serialScope.launch {
      val pollsResponse = controllerContainer.socketController.sendMessage(GET_POLLS, null)
      val socketParsedResponse =
        controllerContainer.socketMessageResponseParser.parseResponse(pollsResponse)
      val webSocketPollsModel = socketParsedResponse.payload as WebSocketPollsModel
      webSocketPollsModel.polls?.values?.forEach {
        handleSocketPoll(it, true)
      }
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
    }
  }

  override fun onNewPoll(websocketPoll: WebSocketPoll) {
    handleSocketPoll(websocketPoll, true)
  }

  override fun vote(
    dytePollMessage: DytePollMessage,
    dytePollOption: DytePollOption
  ) {
    if (controllerContainer.presetController.canVoteOnPoll().not()) {
      return
    }

    val content = HashMap<String, JsonElement>()
    content["index"] = JsonPrimitive(dytePollMessage.options.indexOf(dytePollOption))
    content["pollId"] = JsonPrimitive(dytePollMessage.id)
    controllerContainer.socketController.sendMessage(VOTE_POLL, JsonObject(content))
    loadPolls()
  }
  override fun create(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean
  ) {
    if (controllerContainer.presetController.canCreatePoll().not()) {
      return
    }

    val content = HashMap<String, JsonElement>()
    content["question"] = JsonPrimitive(question)
    content["options"] = JsonArray(options.map { JsonPrimitive(it) })
    content["anonymous"] = JsonPrimitive(anonymous)
    content["hideVotes"] = JsonPrimitive(hideVotes)
    controllerContainer.socketController.sendMessage(NEW_POLL, JsonObject(content))
    return
  }

  private fun handleSocketPoll(webSocketPoll: WebSocketPoll, updateUi: Boolean) {
    val newPoll = handlePoll(webSocketPoll)
    if (updateUi) {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnNewMeetingPollReceived(
          newPoll
        )
      )
    }
  }

  private fun handlePoll(webSocketPoll: WebSocketPoll): DytePollMessage {
    if (!controllerContainer.presetController.canViewPoll()) {
      throw UnsupportedOperationException("not allowed to view poll")
    }

    val options = ArrayList<DytePollOption>()
    webSocketPoll.options.forEach { option ->
      val votes = ArrayList<DytePollVote>()
      val updateOption = DytePollOption(option.text, votes, option.count)
      option.votes.forEach { vote ->
        votes.add(DytePollVote(vote.id, vote.name))
      }
      options.add(updateOption)
    }
    val ifPollExists = polls.find { it.id == webSocketPoll.id } != null
    if (ifPollExists) {
      val existingPoll = requireNotNull(polls.find { it.id == webSocketPoll.id })
      val updatedPoll = existingPoll.copy(options = options)
      val existingIndex = polls.indexOf(existingPoll)
      _polls.removeAt(existingIndex)
      _polls.add(existingIndex, updatedPoll)
      return updatedPoll
    } else {
      val newPoll = DytePollMessage(
        webSocketPoll.id,
        webSocketPoll.question,
        webSocketPoll.anonymous,
        webSocketPoll.hideVotes,
        webSocketPoll.createdBy,
        options
      )
      _polls.add(newPoll)
      return newPoll
    }
  }
}

interface IPollController {
  val polls: List<DytePollMessage>

  fun loadPolls()
  fun onNewPoll(websocketPoll: WebSocketPoll)

  fun vote(dytePollMessage: DytePollMessage, dytePollOption: DytePollOption)
  fun create(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean
  )
}