package io.dyte.core.controllers

import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_ALL
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_PEER
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_ALL_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.PIN_PARTICIPANT
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class HostController(controllerContainer: IControllerContainer) : IHostController,
  BaseController(controllerContainer) {
  override fun kick(id: String) {
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      throw UnsupportedOperationException("not allowed to kick participant")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(id)
    controllerContainer.socketController.sendMessage(KICK_PEER, JsonObject(content))
  }

  // TODO : is there any differance between kick single and kick all ?
  override fun kickAll() {
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      throw UnsupportedOperationException("not allowed to kick all")
    }
    controllerContainer.socketController.sendMessage(KICK_ALL, null)
  }

  // TODO : is there any differance between mute single and mute all ?
  override fun muteAllAudio(allowUnmute: Boolean) {
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      throw UnsupportedOperationException("not allowed to mute all audio")
    }
    val content = HashMap<String, JsonElement>()
    content["allowUnMute"] = JsonPrimitive(allowUnmute)
    controllerContainer.socketController.sendMessage(MUTE_ALL_AUDIO, JsonObject(content))
  }

  override fun muteAllVideo() {
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      throw UnsupportedOperationException("not allowed to mute all video")
    }
    controllerContainer.socketController.sendMessage(MUTE_ALL_VIDEO, null)
  }

  override fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant) {
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      throw UnsupportedOperationException("cant disable participant audio")
    }
    serialScope.launch {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessage(MUTE_PEER_AUDIO, JsonObject(content))
    }
  }

  override fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant) {
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      throw UnsupportedOperationException("cant produce video")
    }
    serialScope.launch {
      val content = HashMap<String, JsonElement>()
      content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
      controllerContainer.socketController.sendMessage(MUTE_PEER_VIDEO, JsonObject(content))
    }
  }

  override fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant) {
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      throw UnsupportedOperationException("not allowed to pin participant")
    }

    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    controllerContainer.socketController.sendMessage(PIN_PARTICIPANT, JsonObject(content))
  }

  override fun unpinParticipant(){
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      throw UnsupportedOperationException("not allowed to un-pin participant")
    }
    val content = HashMap<String, JsonElement>()
    val nullString: String? = null
    content["id"] = JsonPrimitive(nullString)
    controllerContainer.socketController.sendMessage(PIN_PARTICIPANT, JsonObject(content))
  }
}

interface IHostController {
  fun muteAllAudio(allowUnmute: Boolean)
  fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant)
  fun muteAllVideo()
  fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant)
  fun kickAll()
  fun kick(id: String)
  fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant)
  fun unpinParticipant()
}
