package io.dyte.core.network.info

import io.dyte.core.network.models.ParticipantPreset
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.WaitingRoomType

data class ParticipantInfo internal constructor(
  val id: String,
  val name: String,
  val email: String?,
  val picture: String?,
  val loggedIn: Boolean,
  val clientSpecificId: String,
  val organizationId: String,
  val scope: String?,
  val hiddenParticipant: Boolean,
  val isRecorder: Boolean,
  val presetInfo: PresetInfo
) {
  companion object {
    fun getParticipantInfoFromV1(
      userData: UserDataWrapperV1,
      userPresetData: UserPresetDataWrapper
    ): ParticipantInfo {
      val presetInfo = PresetInfo.getPresetInfoFromV1(userPresetData)
      return ParticipantInfo(
        userData.user.id,
        userData.user.name,
        null,
        userData.user.picture,
        false,
        userData.user.clientSpecificId,
        userData.user.organizationId,
        null,
        false,
        false,
        presetInfo
      )
    }

    fun getParticipantInfoFromV2(userData: UserDataWrapper): ParticipantInfo {
      val preset = PresetInfo.getPresetInfoFromV2(userData.data.preset)
      val participant = userData.data.participant
      return ParticipantInfo(
        participant.id,
        participant.name,
        null,
        userData.data.participant.picture,
        false,
        participant.clientSpecificId,
        participant.organizationId,
        null,
        false,
        false,
        preset
      )
    }
  }
}

data class PresetInfo internal constructor(
  val viewType: String,
  val name: String,
  val waitingRoomType: WaitingRoomType,
  val gridInfo: GridInfo,
  val permissions: SelfPermissions
) {
  companion object {
    fun getPresetInfoFromV1(userPresetData: UserPresetDataWrapper): PresetInfo {
      userPresetData.data?.preset?.let { preset ->
        val selfPermissions =
          SelfPermissions.getParticipantPermissionsFromV1(userPresetData)
        return PresetInfo(
          preset.permissions?.viewType ?: "",
          preset.presetName ?: "",
          WaitingRoomType.valueOf(preset.permissions?.waitingRoomType?.name ?: ""),
          GridInfo(preset.theme?.grid?.multi?.maxVideoCount ?: 6),
          selfPermissions
        )
      } ?: run {
        throw IllegalArgumentException("not enough data to build preset info from v1")
      }
    }

    fun getPresetInfoFromV2(participantPreset: ParticipantPreset): PresetInfo {
      val selfPermissions =
        SelfPermissions.getParticipantPermissionsFromV2(participantPreset)
      return PresetInfo(
        participantPreset.config.viewType,
        participantPreset.name,
        WaitingRoomType.valueOf(participantPreset.permissions.waitingRoomType),
        GridInfo(participantPreset.config.streamsConfig.mobile),
        selfPermissions
      )
    }
  }
}

data class GridInfo internal constructor(
  val maxParticipantsPerPage: Int
)

data class SelfPermissions internal constructor(
  val miscellaneous: MiscellaneousPermissions,
  val host: HostPermissions,
  val plugins: PluginPermissions,
  val polls: PollPermissions,
  val media: MediaPermissions,
  val chat: ChatPermissions,
  val waitingRoom: WaitingRoomPermissions
) {
  companion object {
    fun getParticipantPermissionsFromV1(userPresetData: UserPresetDataWrapper): SelfPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        val miscPermissions = MiscellaneousPermissions(permissions.canEditDisplayName ?: false, permissions.showParticipantList ?: false)
        val hostPermissions = HostPermissions.getHostPermissionsFromV1(userPresetData)
        val pluginPermissions = PluginPermissions.getPluginPermissionsFromV1(userPresetData)
        val pollPermissions = PollPermissions.getPollPermissionsFromV1(userPresetData)
        val mediaPermissions = MediaPermissions.getMediaPermissionsFromV1(userPresetData)
        val chatPermissions = ChatPermissions.getChatPermissionsFromV1(userPresetData)
        val waitingRoomPermission = WaitingRoomPermissions.getPermissionsFromV1(userPresetData)
        return SelfPermissions(
          miscPermissions,
          hostPermissions,
          pluginPermissions,
          pollPermissions,
          mediaPermissions,
          chatPermissions,
          waitingRoomPermission
        )
      } ?: run {
        throw IllegalArgumentException("not enough data to build participant permissions from v1")
      }
    }

    // TODO : fix can edit display name
    fun getParticipantPermissionsFromV2(participantPreset: ParticipantPreset): SelfPermissions {
      val miscPermissions = MiscellaneousPermissions(true, participantPreset.permissions.showParticipantList)
      val hostPermissions = HostPermissions.getHostPermissionsFromV2(participantPreset)
      val pluginPermissions = PluginPermissions.getPluginPermissionsFromV2(participantPreset)
      val pollPermissions = PollPermissions.getPollPermissionsFromV2(participantPreset)
      val mediaPermissions = MediaPermissions.getMediaPermissionsFromV2(participantPreset)
      val chatPermissions = ChatPermissions.getChatPermissionsFromV2(participantPreset)
      val waitingRoomPermission = WaitingRoomPermissions.getPermissionsFromV2(participantPreset)
      return SelfPermissions(
        miscPermissions,
        hostPermissions,
        pluginPermissions,
        pollPermissions,
        mediaPermissions,
        chatPermissions,
        waitingRoomPermission
      )
    }
  }

  open fun toMap(): Map<String, Any?> {
    return hashMapOf(
      "miscellaneous" to miscellaneous.toMap(),
      "media" to media.toMap(),
      "plugins" to plugins.toMap(),
      "polls" to polls.toMap(),
      "chat" to chat.toMap(),
      "host" to host.toMap(),
    )
  }
}

data class MiscellaneousPermissions internal constructor(
  val canEditDisplayName: Boolean,
  val isHiddenParticipant: Boolean,
) {
  fun toMap() : Map<String, Any>{
    return hashMapOf(
      "canEditDisplayName" to canEditDisplayName,
      "isHiddenParticipant" to isHiddenParticipant,
    );
  }
}

data class HostPermissions internal constructor(
  val canKickParticipant: Boolean,
  val canMuteAudio: Boolean,
  val canMuteVideo: Boolean,
  val canPinParticipant: Boolean,
  val canTriggerRecording: Boolean
) {
  companion object {
    fun getHostPermissionsFromV1(userPresetData: UserPresetDataWrapper): HostPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return HostPermissions(
          canMuteAudio = permissions.canDisableParticipantAudio ?: false,
          canMuteVideo = permissions.canDisableParticipantVideo ?: false,
          canKickParticipant = permissions.kickParticipant ?: false,
          canPinParticipant = permissions.pinParticipant ?: false,
          canTriggerRecording = permissions.canRecord ?: false
        )
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build host permissions from v1 api")
      }
    }

    // TODO : fix canPresent, canAcceptPresentRequest,
    fun getHostPermissionsFromV2(participantPreset: ParticipantPreset): HostPermissions {
      return HostPermissions(
        canMuteAudio = participantPreset.permissions.disableParticipantAudio,
        canMuteVideo = participantPreset.permissions.disableParticipantVideo,
        canKickParticipant = participantPreset.permissions.kickParticipant,
        canPinParticipant = participantPreset.permissions.pinParticipant,
        canTriggerRecording = participantPreset.permissions.canRecord
      )
    }
  }
  fun toMap() : Map<String,Any>{
    return hashMapOf(
      "canMuteAudio" to canMuteAudio,
      "canMuteVideo" to canMuteVideo,
      "canKickParticipant" to canKickParticipant,
      "canPinParticipant" to canPinParticipant,
      "canTriggerRecording" to canTriggerRecording,
    );
  }
}

data class PluginPermissions internal constructor(
  val canClose: Boolean,
  val canLaunch: Boolean,
  private val canEditAcl: Boolean
) {
  companion object {
    fun getPluginPermissionsFromV1(userPresetData: UserPresetDataWrapper): PluginPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PluginPermissions(
          permissions.plugins?.canClose ?: false,
          permissions.plugins?.canStart ?: false,
          permissions.plugins?.canEditAcl ?: false
        )
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build plugins permissions from v1 api")
      }
    }

    fun getPluginPermissionsFromV2(participantPreset: ParticipantPreset): PluginPermissions {
      return PluginPermissions(
        participantPreset.permissions.plugins.canClose,
        participantPreset.permissions.plugins.canStart,
        false
      )
    }
  }

  fun toMap() : Map<String,Any>{
    return hashMapOf(
      "canClose" to canClose,
      "canLaunch" to canLaunch
    );
  }
}

data class PollPermissions internal constructor(
  val canCreate: Boolean,
  val canVote: Boolean,
  val canView: Boolean
) {
  companion object {
    fun getPollPermissionsFromV1(userPresetData: UserPresetDataWrapper): PollPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PollPermissions(
          permissions.polls?.canCreate ?: false,
          permissions.polls?.canVote ?: false,
          permissions.polls?.canView ?: false
        )
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build polls permissions from v1 api")
      }
    }

    fun getPollPermissionsFromV2(participantPreset: ParticipantPreset): PollPermissions {
      return PollPermissions(
        participantPreset.permissions.polls.canCreate,
        participantPreset.permissions.polls.canVote,
        participantPreset.permissions.polls.canView
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf<String, Any>(
      "canCreate" to canCreate,
      "canView" to canView,
      "canVote" to canVote,
    );
  }
}

data class MediaPermissions internal constructor(
  private val video: VideoPermissions,
  val canPublishAudio: Boolean,
) {
  val canPublishVideo = video.canPublish
  companion object {
    fun getMediaPermissionsFromV1(userPresetData: UserPresetDataWrapper): MediaPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return MediaPermissions(
          VideoPermissions.getVideoPermissionsFromV1(userPresetData),
          permissions.produce?.audio ?: false
        )
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build media permissions from v1 api")
      }
    }

    fun getMediaPermissionsFromV2(participantPreset: ParticipantPreset): MediaPermissions {
      return MediaPermissions(
        VideoPermissions.getVideoPermissionsFromV2(participantPreset),
        participantPreset.permissions.media.audio.canProduce == "ALLOWED"
      )
    }
  }
  fun toMap() : Map<String, Any>{
    return hashMapOf(
      "canPublishAudio" to canPublishAudio,
      "canPublishVideo" to canPublishVideo,
    );
  }
}

data class VideoPermissions internal constructor(
  val canPublish: Boolean,
  val quality: String,
  val frameRate: Int
) {
  companion object {
    fun getVideoPermissionsFromV1(userPresetData: UserPresetDataWrapper): VideoPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return VideoPermissions(
          permissions.produce?.video?.allow ?: false || permissions.viewType == "GROUP_CALL",
          permissions.produce?.video?.quality ?: "720p",
          permissions.produce?.video?.frameRate ?: 30
        )
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build video permissions from v1 api")
      }
    }

    // TODO : fix quality and framerate
    fun getVideoPermissionsFromV2(participantPreset: ParticipantPreset): VideoPermissions {
      return VideoPermissions(
        participantPreset.permissions.media.video.canProduce == "ALLOWED",
        "720p",
        30
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canPublish" to canPublish,
      "quality" to quality,
      "frameRate" to frameRate,
    );
  }
}

data class ChatPermissions internal constructor(
  val canSendText: Boolean,
  val canSendFiles: Boolean
) {
  companion object {
    fun getChatPermissionsFromV1(userPresetData: UserPresetDataWrapper): ChatPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return ChatPermissions(
          permissions.chat?.chatPublicProps?.text ?: false,
          permissions.chat?.chatPublicProps?.files ?: false
        )
      } ?: run {
        throw IllegalArgumentException("not sufficent data to build chat permissions from v1 api")
      }
    }

    fun getChatPermissionsFromV2(participantPreset: ParticipantPreset): ChatPermissions {
      return ChatPermissions(
        participantPreset.permissions.chat.public.text,
        participantPreset.permissions.chat.public.files
      )
    }
  }
  fun toMap() : Map<String,Any> {
    return hashMapOf(
      "canSendText" to canSendText,
      "canSendFiles" to canSendFiles,
    )
  }
}

data class WaitingRoomPermissions internal constructor(
  val canAcceptRequests: Boolean,
  val behaviour: WaitingRoomType
) {
  companion object {
    fun getPermissionsFromV1(userPresetData: UserPresetDataWrapper): WaitingRoomPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return WaitingRoomPermissions(permissions.acceptPresentRequests ?: false, permissions.waitingRoomType)
      } ?: run {
        throw IllegalArgumentException("not sufficent data to build waiting permissions from v1 api")
      }
    }

    fun getPermissionsFromV2(participantPreset: ParticipantPreset): WaitingRoomPermissions {
      return WaitingRoomPermissions(
        participantPreset.permissions.acceptWaitingRequests,
        WaitingRoomType.valueOf(participantPreset.permissions.waitingRoomType)
      )
    }
  }
}