package io.dyte.core.controllers

import io.dyte.core.models.ProduceData
import io.dyte.core.network.BaseApiService
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.PRODUCE
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class MediaSoupController(controllerContainer: IControllerContainer) :
  IMediaSoupController, BaseController(controllerContainer) {
  private lateinit var _produceData: ProduceData
  override val produceData: ProduceData
    get() = _produceData

  private var _videoProduceId: String? = null
  override val videoProduceId: String?
    get() = _videoProduceId

  private val consumers = arrayListOf<WebSocketConsumerModel>()

  override fun onReceiveTransportConnected(transportId: String, dtlsParameters: String) {
    println("DyteMobileClient | MediaSoupController onReceiveTransportConnected ")
    controllerContainer.loggerController.traceLog("onReceiveTransportConnected")
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
    val payload = JsonObject(content)
    serialScope.launch {
      val resp = controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
        payload
      )
      println("DyteMobileClient | MediaSoupController onReceiveTransportConnected recv transport $resp")
    }
  }

  override fun onSendTransportConnected(transportId: String, dtlsParameters: String) {
    println("DyteMobileClient | MediaSoupController onSendTransportConnected ")
    controllerContainer.loggerController.traceLog("onSendTransportConnected")
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
    val payload = JsonObject(content)
    serialScope.launch {

      val resp = controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
        payload
      )
      println("DyteMobileClient | MediaSoupController onReceiveTransportConnected send transport $resp")
    }
  }

  override fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String {
    println("DyteMobileClient | MediaSoupController onProduce ")
    if (kind == "video") {
      _produceData = ProduceData(transportId, kind, rtpParameters, appData)
      if (controllerContainer.selfController.getSelf().videoEnabled.not()) {
        return ""
      }
    }

    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(transportId)
    rtpParameters.let { content["rtpParameters"] = BaseApiService.json.parseToJsonElement(it) }
    content["kind"] = JsonPrimitive(kind)
    appData?.let { content["appData"] = BaseApiService.json.parseToJsonElement(it) }
    serialScope.launch {
      val resp = controllerContainer.socketController.sendMessage(PRODUCE, JsonObject(content))
      val producerConnectModel =
        controllerContainer.socketMessageResponseParser.parseResponse(resp).payload as WebSocketProducerConnectModel
      if (kind == "video") {
        _videoProduceId = producerConnectModel.id
      }
    }
    return ""
  }

  override fun produceVideo() {
    println("DyteMobileClient | MediaSoupController produceVideo")
    controllerContainer.platformUtilsProvider.getPlatformUtils().printThread()
    if (this::_produceData.isInitialized.not()) {
      return
    }
    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(produceData.transportId)
    content["rtpParameters"] =
      BaseApiService.json.parseToJsonElement(produceData.rtpParameters!!)
    content["kind"] = JsonPrimitive(produceData.kind)
		content["appData"] = BaseApiService.json.parseToJsonElement(produceData.appData!!)

    serialScope.launch {
      runBlocking {
        try {
          val resp =
            controllerContainer.socketController.sendMessage(PRODUCE, JsonObject(content))
          val parsedResp =
            controllerContainer.socketMessageResponseParser.parseResponse(resp).payload as WebSocketProducerConnectModel
          _videoProduceId = requireNotNull(parsedResp.id)
        } catch (e: Exception) {
          // TODO : handle this, this gives error
          e.printStackTrace()
        }
      }
    }
  }

  override fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel) {
    consumers.add(webSocketConsumerModel)
  }

  override fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData {
    return requireNotNull(consumers.find { it.id == consumerId }!!.appData)
  }

  override fun getConsumerType(consumerId: String): String {
    val consumer = consumers.find { it.id == consumerId }
    return consumer?.kind ?: throw IllegalArgumentException("consumer with $consumerId not found")
  }
}

internal interface IMediaSoupController {
  val produceData: ProduceData
  val videoProduceId: String?
  fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String

  fun onReceiveTransportConnected(transportId: String, dtlsParameters: String)
  fun onSendTransportConnected(transportId: String, dtlsParameters: String)
  fun produceVideo()
  fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel)
  fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData?
  fun getConsumerType(consumerId: String): String
}
