package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE

internal class PermissionController(controller: IControllerContainer) : IPermissionController,
  BaseController(controller) {

  override val permissions: List<Permission>
    get() = _permissions.toList()

  private val _permissions = arrayListOf<Permission>()

  private var isNotified = false

  override fun init() {
    super.init()

    /*askPermissions(true, true, {
      //no-op
    }, {
      //no-op
    })*/
  }

  override fun isPermissionGrated(permissionType: PermissionType): Boolean {
    return controllerContainer.platformUtilsProvider.getPermissionUtils().isPermissionGranted(Permission(permissionType, false))
  }

  override fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  ) {
    isNotified = false
    val permissionsToAsk = arrayListOf<PermissionType>()
    if (videoEnabled) {
      println("DyteMobileClient | PermissionController askPermissions adding camera")
      permissionsToAsk.add(CAMERA)
    }
    if (audioEnabled) {
      println("DyteMobileClient | PermissionController askPermissions adding microphone")
      permissionsToAsk.add(MICROPHONE)
    }

    _permissions.clear()
    permissionsToAsk.forEach {
      _permissions.add(Permission(it, false))
    }

    println("DyteMobileClient | PermissionController askPermissions asking for $_permissions")

    controllerContainer.platformUtilsProvider.getPermissionUtils()
      .askPermissions(_permissions, {
        println("DyteMobileClient | PermissionController askPermissions granted")
        if (isNotified.not()) {
          onPermissionGranted.invoke()
          isNotified = true
        }
      }, {
        println("DyteMobileClient | PermissionController askPermissions denied")
        if(isNotified.not()) {
          onPermissionDenied.invoke()
          isNotified = true
        }
      })
  }
}

interface IPermissionController {
  val permissions: List<Permission>
  fun isPermissionGrated(permissionType: PermissionType): Boolean
  fun askPermissions(
    audioEnabled: Boolean,
    videoEnabled: Boolean,
    onPermissionGranted: () -> Unit,
    onPermissionDenied: () -> Unit
  )
}

data class Permission(
  val type: PermissionType,
  var isGranted: Boolean
)

enum class PermissionType(val displayName: String) {
  CAMERA("Camera"),
  MICROPHONE("Mic"),
  STORAGE("Storage");
}