package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveParticipantsChanged
import io.dyte.core.controllers.DyteEventType.OnParticipantsUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareEnded
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareStarted
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.platform.VideoView
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlin.math.ceil

internal class ParticipantController(
  controllerContainer: IControllerContainer
) : IParticipantController, BaseController(controllerContainer) {

  private val joinedParticipants = arrayListOf<DyteMeetingParticipant>()
  private val activeParticipants = arrayListOf<DyteMeetingParticipant>()
  private val screenshareParticipants = arrayListOf<DyteMeetingParticipant>()

  private val hiddenParticipants = arrayListOf<DyteMeetingParticipant>()

  override val meetingRoomParticipants = DyteRoomParticipants(
    controllerContainer.waitlistController.waitlistedParticipants,
    joinedParticipants,
    activeParticipants,
    screenshareParticipants,
    controllerContainer
  )

  private var _pinnedParticipant: DyteMeetingParticipant? = null
  override val pinnedParticipant: DyteMeetingParticipant?
    get() = _pinnedParticipant

  private var _activeSpeaker: DyteMeetingParticipant? = null
  override val activeSpeaker: DyteMeetingParticipant?
    get() = _activeSpeaker

  private lateinit var gridPagesInfo: GridInfo

  private var requestedPageNumber = 0

  override fun init() {
    gridPagesInfo = GridInfo(
      0,
      0,
      false,
      false
    )
  }

  override fun getVideoView(dyteMeetingParticipant: DyteMeetingParticipant): VideoView {
    return controllerContainer.platformUtilsProvider.getVideoUtils()
      .getVideoView(dyteMeetingParticipant)
  }

  override fun getScreenShareView(dyteMeetingParticipant: DyteMeetingParticipant): VideoView {
    return controllerContainer.platformUtilsProvider.getVideoUtils()
      .getScreenShareView(dyteMeetingParticipant)
  }

  /**
   * On peer joined
   *
   * when a remote peer joins, this does not contain audio track or video track
   *
   * @see onParticipantStreamConnected
   *
   * @param meetingPeerUser
   */
  override fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    val participant = getDyteParticipant(meetingPeerUser)
    if (participant.flags.hiddenParticipant || participant.flags.recorder) {
      hiddenParticipants.add(participant)
      if (participant.flags.recorder) {
        controllerContainer.recordingController.onRecordingPeerJoined()
      }
    } else {
      joinedParticipants.add(participant)
      updatePageCount()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerJoin(participant))
    }
  }

  override fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    var isHiddentParticipantLeft = false
    var peerToRemove = joinedParticipants.find { it.id == webSocketPeerLeftModel.id }
    if (peerToRemove == null) {
      peerToRemove = hiddenParticipants.find { it.id == webSocketPeerLeftModel.id }
      if (peerToRemove != null) {
        if (peerToRemove.flags.recorder) {
          controllerContainer.recordingController.onRecordingPeerLeft()
        }
        hiddenParticipants.remove(peerToRemove)
        isHiddentParticipantLeft = true
      }
    }

    if (!isHiddentParticipantLeft) {
      peerToRemove?.let {
        val screenSharePeer = screenshareParticipants.find { it.id == peerToRemove.id }
        if (screenSharePeer != null) {
          screenshareParticipants.remove(screenSharePeer)
          controllerContainer.eventController.triggerEvent(OnPeerScreenShareEnded(screenSharePeer))
        }

        joinedParticipants.remove(peerToRemove)
        val removedFromActive = activeParticipants.removeAll { it.id == peerToRemove.id }
        if (removedFromActive) {
          updatePageCount()
          controllerContainer.eventController.triggerEvent(
            OnActiveParticipantsChanged(
              activeParticipants
            )
          )
        }
        if (peerToRemove.id == pinnedParticipant?.id) {
          controllerContainer.eventController.triggerEvent(
            OnPeerUnpinned(
              requireNotNull(
                pinnedParticipant
              )
            )
          )
          _pinnedParticipant = null
        }
        controllerContainer.eventController.triggerEvent(OnPeerLeft(peerToRemove))
      }
    }
  }

  private fun updatePageCount(currentPageNumber: Int = gridPagesInfo.currentPageNumber) {
    // always for active
    var pageCount = 1

    var nextPages = 0
    if (joinedParticipants.size > controllerContainer.presetController.getMaxVideoCount()) {
      nextPages =
        ceil(
          joinedParticipants.size / controllerContainer.presetController.getMaxVideoCount()
            .toFloat()
        ).toInt()
    }
    pageCount += nextPages
    val newGridInfo = gridPagesInfo.copy(
      pageCount = pageCount,
      currentPageNumber = currentPageNumber,
      isNextPagePossible = currentPageNumber > pageCount,
      isPreviousPagePossible = currentPageNumber > 0
    )
    if (newGridInfo != gridPagesInfo) {
      gridPagesInfo = newGridInfo
      controllerContainer.eventController.triggerEvent(OnParticipantsUpdate(meetingRoomParticipants))
    }
  }

  override fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = false
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel) {
    val consumerType =
      controllerContainer.mediaSoupController.getConsumerType(
        requireNotNull(webSocketConsumerClosedModel.id)
      )
    if ("audio" == consumerType) {
      return
    }
    webSocketConsumerClosedModel.id.let {
      val appdata =
        controllerContainer.mediaSoupController.getAppDataFromConsumerId(it)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          if (appdata.screenShare == false) {
            participant._videoEnabled = false
            controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
          }
        }
      }
    }
  }

  override fun onPeerVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel) {
    webSocketConsumerClosedModel.id?.let {
      serialScope.launch {
        withContext(Dispatchers.Main) {
          (controllerContainer.platformUtilsProvider.getSFUUtils() as? IDyteMediaSoupUtils)?.resumeConsumer(it)
        }
      }
    }
  }

  override fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerPinnedModel.peerId }
    participant?.let {
      _pinnedParticipant = participant
      controllerContainer.eventController.triggerEvent(OnPeerPinned(participant))
    }
  }

  override fun onPeerUnpinned() {
    _pinnedParticipant?.let {
      controllerContainer.eventController.triggerEvent(OnPeerUnpinned(it))
    }
    _pinnedParticipant = null
  }

  override fun onActiveSpeaker(id: String) {
    val participant = joinedParticipants.find { it.id == id }
    _activeSpeaker = participant
    _activeSpeaker?.let {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnActiveSpeakerChanged(
          requireNotNull(_activeSpeaker)
        )
      )
    }
  }

  override fun onNoActiveSpeaker() {
    _activeSpeaker = null
    controllerContainer.eventController.triggerEvent(DyteEventType.OnNoActiveSpeaker)
  }

  override fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel) {
    webSocketRoomStateModel.roomState?.peers?.forEach {
      onPeerJoined(it)
    }
  }

  override fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    joinedParticipants.add(controllerContainer.selfController.getSelf())
    if (webSocketJoinRoomModel.pinnedPeerId?.isNotEmpty() == true) {
      val pinnedParticipant =
        joinedParticipants.find { it.id == webSocketJoinRoomModel.pinnedPeerId }
      pinnedParticipant?.let {
        _pinnedParticipant = pinnedParticipant
      }
    }
  }

  override fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    if (requestedPageNumber == 0) {
      val peerIds = arrayListOf<String>()
      peerIds.addAll(webSocketSelectedPeersModel.peerIds ?: emptyList())
      if (controllerContainer.metaController.getRoomType() == "WEBINAR") {
        if (controllerContainer.webinarController.isPresenting()) {
          peerIds.add(controllerContainer.selfController.getSelf().id)
        }
      } else {
        peerIds.add(controllerContainer.selfController.getSelf().id)
      }
      processNewPage(peerIds, 0)
    }
  }

  override fun setPage(newPageNumber: Int) {
    if (newPageNumber < 0) {
      throw IllegalArgumentException("page number cant be less than 0")
    }
    requestedPageNumber = newPageNumber
    val content = HashMap<String, JsonElement>()
    content["pageNum"] = JsonPrimitive(newPageNumber)
    val prevPageResponse =
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.GET_PAGE,
        JsonObject(content)
      )
    val pageModel =
      controllerContainer.socketMessageResponseParser.parseResponse(prevPageResponse).payload as WebSocketGetPageModel
    if (pageModel.peerIds?.isEmpty() == false) {
      processNewPage(pageModel.peerIds, newPageNumber)
    }
  }

  private fun processNewPage(peerIds: List<String>, pageNumber: Int) {
    activeParticipants.clear()
    peerIds.forEach { peerId ->
      val peer = joinedParticipants.find { it.id == peerId }
      peer?.let {
        activeParticipants.add(peer)
      } ?: run {
        if (peerId == controllerContainer.selfController.getSelf().id) {
          activeParticipants.add(controllerContainer.selfController.getSelf())
        }
      }
    }
    updatePageCount(pageNumber)
    controllerContainer.eventController.triggerEvent(OnActiveParticipantsChanged(activeParticipants))
  }

  override fun onPeerScreenShareStarted(
    participant: DyteMeetingParticipant,
    screenshareTrack: Any
  ) {
    participant._screenShareTrack = screenshareTrack
    screenshareParticipants.add(participant)
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareStarted(participant))
    updatePageCount()
  }

  override fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant) {
    controllerContainer.platformUtilsProvider.getVideoUtils()
      .destroyView(screenshareParticipants.find { it.id == participant.id }!!)
    screenshareParticipants.removeAll { it.id == participant.id }
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareEnded(participant))
    updatePageCount()
  }



  // override fun onSelfJoined() {
  // joinedParticipants.add(controllerContainer.selfController.getSelf())
  // activeParticipants.add(controllerContainer.selfController.getSelf())
  // updatePageCount()
  // controllerContainer.eventController.triggerEvent(OnActiveParticipantsChanged(activeParticipants))
  // }

  private fun printRoomParticipants() {
    val stringBuilder = StringBuilder()
    stringBuilder.append("DyteMobileClient")
    stringBuilder.appendLine()
    stringBuilder.append("Joined participants count : ${joinedParticipants.size}")
    stringBuilder.appendLine()
    joinedParticipants.map {
      stringBuilder.append(it.name)
      stringBuilder.append("\t")
    }
    stringBuilder.appendLine()

    stringBuilder.append("Active participants count : ${activeParticipants.size}")
    stringBuilder.appendLine()
    activeParticipants.map {
      stringBuilder.append(it.name)
      stringBuilder.append("\t")
    }
    stringBuilder.appendLine()

    if (pinnedParticipant != null) {
      stringBuilder.append("pinned participant ${pinnedParticipant?.name}")
    }
    stringBuilder.appendLine()
    println("DyteMobileClient | ParticipantController printRoomParticipants")
    println("<<<<<<<<<<<<${stringBuilder}>>>>>>>>>>>>>>")
  }

  private fun getDyteParticipant(meetingPeerUser: WebSocketMeetingPeerUser): DyteMeetingParticipant {
    val flags = ParticipantFlags(
      meetingPeerUser.flags?.recordere ?: false,
      meetingPeerUser.flags?.hiddenParticipant ?: false
    )
    return DyteParticipant(
      requireNotNull(meetingPeerUser.id),
      requireNotNull(meetingPeerUser.userId),
      meetingPeerUser.name ?: "",   // null in case of recorder
      meetingPeerUser.picture,
      meetingPeerUser.isHost ?: false,
      meetingPeerUser.clientSpecificId
        ?: controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid(),
      flags,
      controllerContainer
    )
  }

  override fun getGridInfo(): GridInfo {
    return gridPagesInfo
  }
}

interface IParticipantController {
  val meetingRoomParticipants: DyteRoomParticipants
  val pinnedParticipant: DyteMeetingParticipant?
  val activeSpeaker: DyteMeetingParticipant?

  fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel)
  fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel)
  fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser)
  fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel)
  fun onPeerVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel)
  fun onPeerVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel)
  fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel)
  fun onPeerUnpinned()
  fun onActiveSpeaker(id: String)
  fun onNoActiveSpeaker()
  fun onPeerScreenShareStarted(participant: DyteMeetingParticipant, screenshareTrack: Any)
  fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant)  fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel)
  fun setPage(newPageNumber: Int)
  fun getVideoView(dyteMeetingParticipant: DyteMeetingParticipant): VideoView
  fun getScreenShareView(dyteMeetingParticipant: DyteMeetingParticipant): VideoView
  fun getGridInfo(): GridInfo
}
data class GridInfo(
  val pageCount: Int,
  val currentPageNumber: Int,
  val isNextPagePossible: Boolean,
  val isPreviousPagePossible: Boolean
) {
  fun toMap(): Map<String, Any?> {
    val args = HashMap<String, Any?>()
    args["pageCount"] = pageCount
    args["currentPageNumber"] = currentPageNumber
    args["isNextPagePossible"] = isNextPagePossible
    args["isPreviousPagePossible"] = isPreviousPagePossible
    return args
  }
}