package io.dyte.core.platform

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import android.os.CountDownTimer
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import io.dyte.core.controllers.Permission
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE

class DyteAndroidPlatformPermissions(private val activity: Activity) :
  IDytePlatformPermissionUtils {
  @RequiresApi(VERSION_CODES.M)
  override fun arePermissionsGranted(permissions: List<Permission>): Boolean {
    println("DyteMobileClient | DyteAndroidPlatformPermissions arePermissionsGranted ")
    val grantedPermissions = arrayListOf<Permission>()
    val deniedPermissions = arrayListOf<Permission>()

    permissions.forEach { permission ->
      if (isPermissionGrantedInternal(activity, permission)) {
        grantedPermissions.add(permission)
      } else {
        deniedPermissions.add(permission)
      }
    }
    return deniedPermissions.isEmpty()
  }

  @RequiresApi(VERSION_CODES.M)
  override fun askPermissions(
    permissions: List<Permission>,
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit
  ) {
    println("DyteMobileClient | DyteAndroidPlatformPermissions askPermissions ")
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
      val deniedPermissions = arrayListOf<Permission>()
      permissions.forEach { permission ->
        if (!isPermissionGrantedInternal(activity, permission)) {
          deniedPermissions.add(permission)
        }
      }
      if (deniedPermissions.isNotEmpty()) {
        onPermissionsDenied()
      } else {
        onPermissionsGranted()
      }
    } else {
      val platformPermissions = permissions.map {
        getPlatformPermission(it)
      }
      activity.requestPermissions(platformPermissions.toTypedArray(), 111)
      pollForPermissions(permissions, onPermissionsGranted, onPermissionsDenied)
    }
  }

  @SuppressLint("NewApi")
  private fun pollForPermissions(
    permissions: List<Permission>,
    onPermissionGranted: () -> Unit,
    onPermissionsDenied: () -> Unit
  ) {
    // timer for 30 seconds, ticked every 1 second
    val timer = object : CountDownTimer(30000, 1000) {
      override fun onTick(millisUntilFinished: Long) {
        permissions.forEach {
          val deniedPermissions = arrayListOf<Permission>()
          val grantedPermissions = arrayListOf<Permission>()
          if (isPermissionGranted(it)) {
            grantedPermissions.add(it)
          } else {
            deniedPermissions.add(it)
          }
          if (deniedPermissions.isEmpty()) {
            println("DyteMobileClient | DyteAndroidPlatformPermissions onTick no denied permissions")
            onPermissionGranted.invoke()
            cancel()
          }
        }
      }

      override fun onFinish() {
        println("DyteMobileClient | DyteAndroidPlatformPermissions onFinish ")
        permissions.forEach {
          val deniedPermissions = arrayListOf<Permission>()
          val grantedPermissions = arrayListOf<Permission>()
          if (isPermissionGranted(it)) {
            grantedPermissions.add(it)
          } else {
            deniedPermissions.add(it)
          }
          if (deniedPermissions.isEmpty()) {
            println("DyteMobileClient | DyteAndroidPlatformPermissions onTick all permissions granted")
            onPermissionGranted.invoke()
          } else {
            println("DyteMobileClient | DyteAndroidPlatformPermissions onFinish denied permissions $deniedPermissions")
            onPermissionsDenied.invoke()
          }
        }
      }
    }
    timer.start()
  }

  @RequiresApi(VERSION_CODES.M)
  override fun isPermissionGranted(permission: Permission): Boolean {
    return isPermissionGrantedInternal(activity, permission)
    // return activity.checkSelfPermission(getPlatformPermission(permission)) == PackageManager.PERMISSION_GRANTED
  }

  private fun getPlatformPermission(permission: Permission): String {
    return when (permission.type) {
      CAMERA -> {
        Manifest.permission.CAMERA
      }
      MICROPHONE -> {
        Manifest.permission.RECORD_AUDIO
      }
      PermissionType.STORAGE -> {
        if (VERSION.SDK_INT >= VERSION_CODES.R) {
          Manifest.permission.MANAGE_EXTERNAL_STORAGE
        } else {
          Manifest.permission.WRITE_EXTERNAL_STORAGE
        }
      }
      else -> {
        throw IllegalArgumentException("permission type not supported ${permission.type}")
      }
    }
  }

  private fun isPermissionGrantedInternal(context: Context, permission: Permission): Boolean {
    return ContextCompat.checkSelfPermission(
      context,
      getPlatformPermission(permission)
    ) == PackageManager.PERMISSION_GRANTED
  }
}