package io.dyte.core.platform

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.app.Application
import android.content.ContentResolver
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import android.os.CountDownTimer
import android.provider.OpenableColumns
import android.util.Base64
import android.util.Log
import android.webkit.MimeTypeMap
import androidx.annotation.RequiresApi
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import org.json.JSONObject
import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileNotFoundException
import java.io.IOException
import java.io.UnsupportedEncodingException
import java.net.URLEncoder
import java.text.DateFormat
import java.text.SimpleDateFormat
import java.util.Date
import java.util.TimeZone
import java.util.UUID

/**
 * Dyte android platform
 *
 * Place to put misc platform specific code
 *
 * @constructor Create empty Dyte android platform
 */
internal class DyteAndroidPlatform(
  private val activity: Activity,
  private val utilsProvider: IDytePlatformUtilsProvider
) :
  IDytePlatformUtils {

  /**
   * Get current time
   *
   * @return current time in millis
   */
  override fun getCurrentTime(): Long {
    return System.currentTimeMillis()
  }

  /**
   * Get uuid
   *
   * @return random uuid
   */
  override fun getUuid(): String {
    return UUID.randomUUID().toString()
  }

  override fun getAndroidApplicationContext(): Any {
    return activity.application
  }

  override fun getActivity(): Any {
    return activity
  }

  // "yyyy-MM-dd'T'HH:mm:ss'Z'"
  @SuppressLint("SimpleDateFormat")
  override fun getDiff(startTime: String, endTime: String): String {
    val dates = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    val date1 = dates.parse(startTime)
    val date2 = dates.parse(endTime)
    val differenceInMillis =
      Math.abs(requireNotNull(date1).time - requireNotNull(date2).time).toInt()

    val seconds: Int = differenceInMillis / 1000 % 60
    val minutes: Int = (differenceInMillis - seconds) / 1000 / 60

    var newSeconds = seconds.toString()
    if (seconds < 10) {
      newSeconds = "0$newSeconds"
    }

    var newMinutes = minutes.toString()
    if (minutes < 10) {
      newMinutes = "0$newMinutes"
    }

    return "$newMinutes:$newSeconds"
  }

  // "YYYY-MM-DDThh:mm:ss.sTZD"
  @SuppressLint("SimpleDateFormat")
  override fun getUtcTimeNow(): String {
    val m_ISO8601Local: DateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
    m_ISO8601Local.timeZone = TimeZone.getTimeZone("UTC")
    return m_ISO8601Local.format(Date())
  }

  @SuppressLint("SimpleDateFormat")
  override fun getUserDisplayableTime(time: Long): String {
    val dateFormat: DateFormat = SimpleDateFormat("HH:mm")
    return dateFormat.format(Date(time))
  }

  override fun getFileContent(filePath: String): ByteArray {
    return File(filePath).readBytes()
  }

  override fun printThread(desc: String?) {
    Log.v("DyteMobileClient", "current thread ${Thread.currentThread().name} for $desc")
  }

  override fun listenForCrashes() {
    val defaultHandler = Thread.getDefaultUncaughtExceptionHandler()
    Thread.setDefaultUncaughtExceptionHandler { thread, throwable ->
      /*utilsProvider.getControllerContainer().platformUtilsProvider.getDispatchers().bgScope.launch {
        utilsProvider.getControllerContainer().loggerController.traceError(
          throwable.message ?: "crash happened, reason unknown"
        )
        utilsProvider.getControllerContainer().loggerController.pushAllNow()
      }*/
      if (defaultHandler != null) {
        defaultHandler.uncaughtException(thread, throwable)
      }
    }
  }

  override fun getOsName(): String {
    return "android"
  }

  override fun getOsVersion(): String {
    return VERSION.SDK_INT.toString()
  }

  override fun getDeviceInfo(): String {
    return Build.MODEL + " | " + Build.MANUFACTURER + " | " + VERSION.SDK_INT
  }

  override fun decodeAuthToken(authToken: String): String? {
    var meetingId: String? = null
    try {
      val split = authToken.split("\\.".toRegex()).toTypedArray()
      val body = JSONObject(getJson(split[1]))
      if (body.has("meetingId")) {
        meetingId = body.getString("meetingId")
      }
    } catch (e: UnsupportedEncodingException) {
      throw IllegalArgumentException("Invalid auth token")
    }
    return meetingId
  }

  @Throws(UnsupportedEncodingException::class)
  private fun getJson(strEncoded: String): String {
    val decodedBytes: ByteArray = Base64.decode(strEncoded, Base64.URL_SAFE)
    return io.ktor.utils.io.core.String(decodedBytes)
  }

  override fun getUrlEncodedString(stringToEncode: String): String {
    return URLEncoder.encode(stringToEncode, "UTF-8")
  }

  override fun getPlatformFile(path: String): PlatformFile? {
    return try {
      val file = File(path)
      val mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(file.extension)
      return PlatformFile(
        name = file.name,
        content = file.readBytes(),
        size = file.length(),
        mimeType = mimeType
      )
    } catch (e: Exception) {
      // In case if provided path is not absolute or file doesn't exist we return null
      utilsProvider.getControllerContainer().loggerController.traceError(
        "DyteAndroidPlatform | getPlatformFile | ${e.message} | path=$path"
      )
      null
    }
  }

  override fun getPlatformFile(uri: Uri): PlatformFile? {
    (getAndroidApplicationContext() as? Application)?.applicationContext?.let { context ->
      val contentResolver = context.contentResolver
      try {
        contentResolver.query(uri, null, null, null, null)?.use { cursor ->
          val nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
          val sizeIndex = cursor.getColumnIndex(OpenableColumns.SIZE)
          if (cursor.moveToFirst()) {
            val name = cursor.getString(nameIndex)
            val size = cursor.getLong(sizeIndex)
            val mimeType = contentResolver.getType(uri)
            val content = readFileContent(contentResolver, uri)
            return PlatformFile(name, content, size, mimeType)
          }
        }
      } catch (e: FileNotFoundException) {
        utilsProvider.getControllerContainer().loggerController.traceError(
          "DyteAndroidPlatform | getPlatformFile | ${e.message} | uri=$uri"
        )
      } catch (e: IOException) {
        utilsProvider.getControllerContainer().loggerController.traceError(
          "DyteAndroidPlatform | getPlatformFile | ${e.message} | uri=$uri"
        )
      } catch (e: Exception) {
        utilsProvider.getControllerContainer().loggerController.traceError(
          "DyteAndroidPlatform | getPlatformFile | ${e.message} | uri=$uri"
        )
      }
    }

    return null
  }

  @Throws(FileNotFoundException::class, IOException::class)
  private fun readFileContent(contentResolver: ContentResolver, uri: Uri): ByteArray {
    contentResolver.openInputStream(uri)?.use { inputStream ->
      return inputStream.readBytes()
    } ?: throw IOException("Could not open input stream to read file")
  }
}

// note: This annotation is currently needed as Android's Uri is an abstract class and we want to
// provide a class as its typealias. This may be removed in the future versions of KMM and Kotlin.
@Suppress("ACTUAL_WITHOUT_EXPECT")
actual typealias Uri = android.net.Uri