package io.dyte.callstats

import io.dyte.callstats.datasinks.DataSink
import io.dyte.callstats.datasinks.HTTPDataSink
import io.dyte.callstats.errors.NotAcceptableDeviceType
import io.dyte.callstats.events.*
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.*
import io.dyte.callstats.observers.CallStatsObserver
import io.dyte.callstats.observers.InternalTestObserver
import io.dyte.callstats.observers.TestObserver
import io.dyte.callstats.platform.PlatformInfo
import io.dyte.callstats.utils.DependencyProvider
import io.dyte.webrtc.IceServer
import kotlinx.coroutines.*
import kotlinx.datetime.Clock

class CallStats(
  debug: Boolean = false,
  private val callStatsObserver: CallStatsObserver,
  private val env: Environment,
  private val coroutineScope: CoroutineScope
) {

  private var producingTransport: TransportFacade? = null
  private var consumingTransport: TransportFacade? = null
  private val producers: MutableMap<String, ProducerFacade> = mutableMapOf()
  private val consumers: MutableMap<String, ConsumerFacade> = mutableMapOf()
  private val provider = DependencyProvider(debug)
  private var dataSink: DataSink
  private var eventHandler: EventHandler
  private var iceServers: List<IceServer> = listOf()
  private val measurements = CallStatsMeasurements(provider, callStatsObserver, coroutineScope)

  private val logger = this.provider.getLogger()
  private lateinit var setIntervalHandler: Job

  init {
    val ipDetails = runBlocking { measurements.getIpDetails() }
    this.callStatsObserver.onInitialized(ipDetails)
    this.dataSink = HTTPDataSink(this.provider.getLogger(), env)
    this.eventHandler = EventHandler(this.dataSink, callStatsObserver, this.provider)
  }

  // NOTE: For functions which accept their own observers, callStatsObserver will not be used
  fun authenticate(authPayload: AuthPayload) {
    try {
      this.dataSink.authenticate(authPayload)
    } catch (ex: Exception) {
      this.logger.logError("Exception while authenticating: $ex")
    }
  }

  fun registerIceServers(servers: IceServersWrapper) {
    try {
      this.iceServers = servers.toPcIceServers()
    } catch (ex: Exception) {
      this.logger.logError("Exception while registering ice servers: $ex")
    }
  }

  // Send Transport related
  fun registerProducingTransport(transport: TransportFacade) {
    this.producingTransport = transport
  }

  fun registerProducer(producer: ProducerFacade) {
    try {
      this.producers[producer.getId()] = producer
      this.measurements.registerProducer(producer)
      this.logger.log("Registered Producer with id: ${producer.getId()}")
    } catch (ex: Exception) {
      this.logger.logError("Exception while registering producer: $ex")
    }
  }

  fun registerProducerListenerOnTransportClose(producer: ProducerFacade) {
    deRegisterProducer(producer)

    // For js callstats, mediasoup-client provides an event for when transport closes

    // In cpp client no such event/callback exists, but each producer/consumer on that
    // transport is called individually for closing them, so even though transport
    // closes after a bit, we can close it over here itself ( and yes it will be set to
    // null multiple times if transport has multiple producers/consumers under it )

    this.producingTransport = null
  }

  fun deRegisterProducer(producer: ProducerFacade) {
    this.producers.remove(producer.getId())
    this.measurements.deregisterProducer(producer)
  }

  fun disconnectProducingTransport() {
    this.producingTransport = null
  }

  // Receive Transport related
  fun registerConsumingTransport(transport: TransportFacade) {
    this.consumingTransport = transport
  }

  fun registerConsumer(consumer: ConsumerFacade) {
    try {
      this.consumers[consumer.getId()] = consumer
      this.measurements.registerConsumer(consumer)
      this.logger.log("Registered Consumer with id: ${consumer.getId()}")
    } catch (ex: Exception) {
      this.logger.logError("Exception while registering consumer: $ex")
    }
  }

  fun registerConsumerListenerOnTransportClose(consumer: ConsumerFacade) {
    deRegisterConsumer(consumer)

    // For js callstats, mediasoup-client provides an event for when transport closes

    // In cpp client no such event/callback exists, but each producer/consumer on that
    // transport is called individually for closing them, so even though transport
    // closes after a bit, we can close it over here itself ( and yes it will be set to
    // null multiple times if transport has multiple producers/consumers under it )
    this.consumingTransport = null
  }

  fun deRegisterConsumer(consumer: ConsumerFacade) {
    this.consumers.remove(consumer.getId())
    this.measurements.deregisterConsumer(consumer)
  }

  fun disconnectConsumingTransport() {
    this.consumingTransport = null
  }

  fun sendPreCallTestBeginEvent(observer: TestObserver) {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.PreCallTestBeginEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)

      val onDoneCb = { data: Any ->
        val doneNow = Clock.System.now()
        val doneEventEntry =
          EventEntry.PreCallTestCompleteEntry(
            doneNow,
            EventData.PreCallTestData(connectionInfo = data as NetworkInformation?)
          )
        this.eventHandler.callEvent(doneEventEntry, doneNow)
        observer.onDone(data)
      }

      val onFailureCb = { reason: String, lastUpdatedResults: Any? ->
        observer.onFailure(reason, lastUpdatedResults)
      }

      val onErrorCb = { ex: Exception -> observer.onError(ex) }

      val resultObserver = InternalTestObserver(onDoneCb, onFailureCb, onErrorCb)
      coroutineScope.launch { measurements.getNetworkInfo(iceServers, resultObserver) }
    } catch (ex: Exception) {
      observer.onError(ex)
    }
  }

  // TODO: think should this be Double? ( question mark does not denote kotlin nullable one )
  fun screenShareToggleEvent(on: Boolean, ssrc: Double) {
    try {
      val now = Clock.System.now()
      val eventEntry =
        if (on) {
          EventEntry.ScreenShareStartedEntry(now, EventData.Object(mapOf("ssrc" to ssrc)))
        } else {
          EventEntry.ScreenShareStoppedEntry(now, EventData.Object(mapOf("ssrc" to ssrc)))
        }
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendScreenShareRequestedEvent() {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.ScreenShareRequestedEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendActiveSpeakerEvent(peerId: String) {
    try {
      val now = Clock.System.now()
      val eventEntry =
        EventEntry.DominantSpeakerEntry(now, EventData.Object(mapOf("peerId" to peerId)))
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  // NOTE: Do not remove these commented unimplemented functions
  // fun devices() {}

  // fun selectedDevices() {}

  fun mediaPermission(deviceType: MediaDeviceType, permission: Any) {
    try {
      val now = Clock.System.now()
      val eventEntry =
        EventEntry.MediaPermissionEntry(
          now,
          EventData.Object(mapOf("deviceType" to deviceType, "permission" to permission))
        )
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun mediaPlaybackFailed(deviceType: MediaDeviceType) {
    try {
      val now = Clock.System.now()
      val eventEntry =
        when (deviceType) {
          MediaDeviceType.AUDIO ->
            EventEntry.AudioPlaybackFailureEntry(
              now,
              EventData.Object(mapOf("deviceType" to deviceType))
            )
          MediaDeviceType.VIDEO ->
            EventEntry.VideoPlaybackFailureEntry(
              now,
              EventData.Object(mapOf("deviceType" to deviceType))
            )
          else -> throw NotAcceptableDeviceType
        }

      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun legacySwitch(on: Boolean) {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.LegacySwitchEntry(now, EventData.Object(mapOf("on" to on)))
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendCallJoinBeginEvent(peerMetaData: PeerMetaData) {
    val platformInfo = PlatformInfo()
    try {
      peerMetaData.meetingEnv = this.env.str
      peerMetaData.deviceInfo.isMobile = true
      peerMetaData.deviceInfo.osName = platformInfo.getOSName()
      peerMetaData.deviceInfo.osVersionName = platformInfo.getOSVersionName()
      peerMetaData.deviceInfo.userAgent = platformInfo.getUserAgent()
      peerMetaData.deviceInfo.cpus = platformInfo.getCpuCount()
      peerMetaData.deviceInfo.memory = platformInfo.getTotalMemory()

      val now = Clock.System.now()
      val eventEntry = EventEntry.CallJoinBeginEntry(now, EventData.CallJoinData(peerMetaData))
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  suspend fun sendNetworkQualityTestBeginEvent(regionalInformations: List<RegionalIceInformation>) {
    try {
      val onDone = { data: Any ->
        val regionData = data as MutableList<EventData.RegionalNetworkQualityTestData>
        val doneNow = Clock.System.now()
        val doneEventEntry =
          EventEntry.NetworkQualityTestEndEntry(
            doneNow,
            EventData.NetworkQualityTestData(regionData)
          )
        this.eventHandler.callEvent(doneEventEntry, doneNow)
      }
      val onFailure = { _: String, _: Any? -> }
      val onError = { _: Exception -> }
      val observer = InternalTestObserver(onDone, onFailure, onError)

      val now = Clock.System.now()
      val eventEntry = EventEntry.NetworkQualityTestBeginEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)

      val regionData: MutableList<EventData.RegionalNetworkQualityTestData> = mutableListOf()
      val lastRegionalInformation = regionalInformations.last()

      for (regionalInformation in regionalInformations) {
        try {
          if (regionalInformation.iceServers.isNotEmpty()) {
            val onDoneCb = { data: Any ->
              val networkQualityInfo = data as NetworkQualityInformation
              regionData.add(
                EventData.RegionalNetworkQualityTestData(networkResults = networkQualityInfo)
              )

              if (regionalInformation == lastRegionalInformation) {
                observer.onDone(regionData)
              }
            }

            val onFailureCb = { reason: String, _: Any? ->
              this.logger.logError(
                "Error in sendNetworkQualityTestBeginEvent for regionInformation: $regionalInformation, failure: $reason"
              )
            }

            val onErrorCb = { ex: Exception ->
              this.logger.logError(
                "Error in sendNetworkQualityTestBeginEvent for regionInformation: $regionalInformation, exception: $ex"
              )
            }

            val resultObserver = InternalTestObserver(onDoneCb, onFailureCb, onErrorCb)

            this.measurements.getNetworkQuality(iceServers, resultObserver)
          }
        } catch (ex: Exception) {
          observer.onError(ex)
        }
      }
    } catch (ex: Exception) {
      this.logger.logError("Error in sendNetworkQualityTestBeginEvent, exception: $ex")
    }
  }

  fun sendWebSocketConnectedEvent() {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.WebSocketConnectedEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendTransportConnectedEvent() {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.TransportConnectedEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendAudioToggleEvent(on: Boolean) {
    try {
      val now = Clock.System.now()

      val eventEntry =
        if (on) {
          EventEntry.AudioOnEntry(now, EventData.EmptyData)
        } else {
          EventEntry.AudioOffEntry(now, EventData.EmptyData)
        }

      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendVideoToggleEvent(on: Boolean) {
    try {
      val now = Clock.System.now()

      val eventEntry =
        if (on) {
          EventEntry.VideoOnEntry(now, EventData.EmptyData)
        } else {
          EventEntry.VideoOffEntry(now, EventData.EmptyData)
        }

      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendParticipantRoleToggleEvent(role: EventData.ParticipantRoleData) {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.ParticipantRoleToggleEntry(now, role)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun startPingStats(interval: Long) {
    val onInterval: suspend () -> Unit = { this.sendPingStatsEvent() }

    setInterval(interval, onInterval)
  }

  private fun setInterval(timeMillis: Long, handler: suspend () -> Unit) {
    this.setIntervalHandler =
      GlobalScope.launch {
        while (true) {
          delay(timeMillis)
          handler()
        }
      }
  }

  fun stopPingStats() {
    this.setIntervalHandler.cancel()
  }

  suspend fun sendPingStatsEvent() {
    try {
      var proReport: ProcessedStatsReport? = null
      var conReport: ProcessedStatsReport? = null

      if (this.producingTransport != null) {
        proReport =
          this.measurements.getProcessedStats(
            this.producingTransport!!,
            consuming = false,
            producing = true
          )
        if (proReport.producerReport == null) {
          this.logger.log("Regenerating producer stream maps!")
          val report = this.measurements.getProducersReport(this.producers.values.toList())

          if (report.isNotEmpty()) {
            proReport.producerReport = report
          } else {
            proReport =
              this.measurements.getProcessedStats(
                this.producingTransport!!,
                consuming = false,
                producing = true
              )
            if (proReport.producerReport == null) {
              this.logger.log("Producer stream maps invalid despite regenerating!")
            }
          }
        }
      }

      if (this.consumingTransport != null) {
        conReport =
          this.measurements.getProcessedStats(
            this.consumingTransport!!,
            consuming = true,
            producing = false
          )
        if (conReport.consumerReport == null) {
          this.logger.log("Regenerating consumer stream maps!")
          val report = this.measurements.getConsumersReport(this.consumers.values.toList())

          if (report.isNotEmpty()) {
            conReport.consumerReport = report
          } else {
            conReport =
              this.measurements.getProcessedStats(
                this.consumingTransport!!,
                consuming = true,
                producing = false
              )
            if (conReport.consumerReport == null) {
              this.logger.log("Consumer stream maps invalid despite regenerating!")
            }
          }
        }
      }

      var producerStats: List<ProducerStatistics>? = null
      var consumerStats: List<ConsumerStatistics>? = null

      if (conReport?.producerReport != null) {
        producerStats = proReport?.producerReport?.plus(conReport.producerReport!!)
      }
      if (proReport?.consumerReport != null) {
        consumerStats = conReport?.consumerReport?.plus(proReport.consumerReport!!)
      }

      val metaData =
        EventData.PingStatsData(
          producingTransportStats = proReport?.transportReport,
          consumingTransportStats = conReport?.transportReport,
          producerStats = producerStats,
          consumerStats = consumerStats
        )

      val now = Clock.System.now()
      val eventEntry = EventEntry.PingStatsEntry(now, metaData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.logger.log("Exception $ex")
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendDisconnectEvent() {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.DisconnectEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }

  fun sendReconnectEvent() {
    try {
      val now = Clock.System.now()
      val eventEntry = EventEntry.ReconnectAttemptEntry(now, EventData.EmptyData)
      this.eventHandler.callEvent(eventEntry, now)
    } catch (ex: Exception) {
      this.callStatsObserver.onError(ex)
    }
  }
}
