package io.dyte.callstats.events

import io.dyte.callstats.datasinks.DataSink
import io.dyte.callstats.observers.CallStatsObserver
import io.dyte.callstats.utils.DependencyProvider
import kotlinx.coroutines.*
import kotlinx.datetime.Instant

class EventStore {
  private var currentChunk: EventChunk = EventChunk(entries = mutableListOf())
  private var flushedChunks: MutableList<EventChunk> = mutableListOf()

  fun add(entry: EventEntry) {
    this.currentChunk.entries.add(entry)
  }

  fun flush(): EventChunk {
    val chunk = this.currentChunk
    this.currentChunk = EventChunk(entries = mutableListOf())
    this.flushedChunks.add(chunk)
    return chunk
  }
}

class EventHandler(
  private val dataSink: DataSink,
  private val observer: CallStatsObserver,
  provider: DependencyProvider,
) {
  private val eventStore: EventStore = EventStore()
  private val logger = provider.getLogger()

  fun callEvent(entry: EventEntry, timestamp: Instant) {
    entry.timestamp = timestamp
    this.eventStore.add(entry)
    this.triggerCallback(entry)
    this.logger.log("Received event entry: ${entry.event} with metadata: ${entry.metaData}")

    val eventCategory: EventCategory? = eventCategoryMap[entry.event]
    if (eventCategory != null && eventCategory == EventCategory.MAJOR_EVENT) {
      val chunk = this.eventStore.flush()
      val globalDatasink = this.dataSink
      GlobalScope.launch {
        try {
          globalDatasink.sendChunk(chunk)
        } catch (e: Exception) {
          /*
           * note: currently we are getting a SocketTimeout exception which can be due to slow network.
           * For now we are just catching the exception and logging it. We will need to mostly increase
           * the timeout in the Http client slightly and also make sure that the events which were not
           * sent due to failure are queued and retried in the existing order.
           * */
          logger.logError("Failed to send chunk, error -> ${e.message}")
        }
      }
    }
  }

  private fun triggerCallback(entry: EventEntry) {
    when (entry) {
      is EventEntry.AudioOffEntry -> this.observer.onAudioOffEntry(entry.metaData)
      is EventEntry.AudioOnEntry -> this.observer.onAudioOnEntry(entry.metaData)
      is EventEntry.AudioPlaybackFailureEntry ->
        this.observer.onAudioPlaybackFailureEntry(entry.metaData)
      is EventEntry.AudioTrackFailureEntry -> this.observer.onAudioTrackFailureEntry(entry.metaData)
      is EventEntry.CallJoinBeginEntry -> this.observer.onCallJoinBeginEntry(entry.metaData)
      is EventEntry.DisconnectEntry -> this.observer.onDisconnectEntry(entry.metaData)
      is EventEntry.DominantSpeakerEntry -> this.observer.onDominantSpeakerEntry(entry.metaData)
      is EventEntry.LegacySwitchEntry -> this.observer.onLegacySwitchEntry(entry.metaData)
      is EventEntry.MediaPermissionEntry -> this.observer.onMediaPermissionEntry(entry.metaData)
      is EventEntry.NetworkQualityTestBeginEntry ->
        this.observer.onNetworkQualityTestBeginEntry(entry.metaData)
      is EventEntry.NetworkQualityTestEndEntry ->
        this.observer.onNetworkQualityTestEndEntry(entry.metaData)
      is EventEntry.ParticipantRoleToggleEntry ->
        this.observer.onParticipantRoleToggleEntry(entry.metaData)
      is EventEntry.PingStatsEntry -> this.observer.onPingStatsEntry(entry.metaData)
      is EventEntry.PreCallTestBeginEntry -> this.observer.onPreCallTestBeginEntry(entry.metaData)
      is EventEntry.PreCallTestCompleteEntry ->
        this.observer.onPreCallTestCompleteEntry(entry.metaData)
      is EventEntry.ReconnectAttemptEntry -> this.observer.onReconnectAttemptEntry(entry.metaData)
      is EventEntry.ScreenShareRequestedEntry ->
        this.observer.onScreenShareRequestedEntry(entry.metaData)
      is EventEntry.ScreenShareStartedEntry ->
        this.observer.onScreenShareStartedEntry(entry.metaData)
      is EventEntry.ScreenShareStoppedEntry ->
        this.observer.onScreenShareStoppedEntry(entry.metaData)
      is EventEntry.TransportConnectedEntry ->
        this.observer.onTransportConnectedEntry(entry.metaData)
      is EventEntry.VideoOffEntry -> this.observer.onVideoOffEntry(entry.metaData)
      is EventEntry.VideoOnEntry -> this.observer.onVideoOnEntry(entry.metaData)
      is EventEntry.VideoPlaybackFailureEntry ->
        this.observer.onVideoPlaybackFailureEntry(entry.metaData)
      is EventEntry.VideoTrackFailureEntry -> this.observer.onVideoTrackFailureEntry(entry.metaData)
      is EventEntry.WebSocketConnectedEntry ->
        this.observer.onWebSocketConnectedEntry(entry.metaData)
    }
  }
}
