package io.dyte.callstats

import io.dyte.callstats.api.IPApiService
import io.dyte.callstats.events.TransportStatistics
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.DataThroughputTestResults
import io.dyte.callstats.models.TestResult
import io.dyte.callstats.observers.CallStatsObserver
import io.dyte.callstats.observers.InternalTestObserver
import io.dyte.callstats.observers.TestObserver
import io.dyte.callstats.tests.*
import io.dyte.callstats.utils.DependencyProvider
import io.dyte.webrtc.IceServer
import io.dyte.webrtc.RtcConfiguration
import io.dyte.webrtc.RtcStatsReport
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.runBlocking

class CallStatsMeasurements(
  private val provider: DependencyProvider,
  private val callStatsObserver: CallStatsObserver,
  private val coroutineScope: CoroutineScope,
) : Measurements<ProducerFacade, ConsumerFacade, TransportFacade, IceServer> {
  private val outboundProducerMap: MutableMap<String, String> = mutableMapOf()
  private val inboundConsumerMap: MutableMap<String, String> = mutableMapOf()
  private val consumerPeerIdMap: MutableMap<String, ConsumerPeerIdValue> = mutableMapOf()
  private val overallConsumerStatsMap: MutableMap<String, OverallConsumerStatsValue> =
    mutableMapOf()
  private var consumerIdsWithFreezedVideo: MutableSet<String> = mutableSetOf()
  private var consumerIdsWithFreezedAudio: MutableSet<String> = mutableSetOf()
  private var logger = this.provider.getLogger()

  private val fallBackIceServers =
    listOf(
      IceServer(
        urls = listOf("turn:turn.dyte.in:443?transport=tcp"),
        username = "dyte",
        password = "dytein",
      ),
      IceServer(
        urls = listOf("turn:turn.dyte.in:3478?transport=udp"),
        username = "dyte",
        password = "dytein",
      ),
    )

  override suspend fun registerProducer(producer: ProducerFacade) {
    this.generateProducerStreamMap(producer, true)
  }

  // private
  override fun processInboundConsumerVideoStats(
    consumerId: String,
    overallConsumerStatsValue: OverallConsumerStatsValue,
    streamStatsData: InboundVideoStreamStats,
  ) {
    if (overallConsumerStatsValue.totalVideoPacketReceived == streamStatsData.packetsReceived) {
      this.consumerIdsWithFreezedVideo.add(consumerId)
      this.logger.log("video freezed for consumer id: $consumerId")
      callStatsObserver.onReceivingConsumerVideoStatus("pause", consumerId)
    } else {
      overallConsumerStatsValue.totalVideoPacketReceived = streamStatsData.packetsReceived
      if (this.consumerIdsWithFreezedAudio.contains(consumerId)) {
        this.logger.log("video working fine for consumer id: $consumerId")
        this.consumerIdsWithFreezedVideo.remove(consumerId)
        callStatsObserver.onReceivingConsumerVideoStatus("resume", consumerId)
      }
    }
  }

  // private
  override fun processInboundConsumerAudioStats(
    consumerId: String,
    overallConsumerStatsValue: OverallConsumerStatsValue,
    streamStatsData: InboundAudioStreamStats,
  ) {
    if (overallConsumerStatsValue.totalAudioPacketReceived == streamStatsData.packetsReceived) {
      this.consumerIdsWithFreezedAudio.add(consumerId)
      this.logger.log("audio freezed for consumer id: $consumerId")
      callStatsObserver.onReceivingConsumerAudioStatus("pause", consumerId)
    } else {
      overallConsumerStatsValue.totalAudioPacketReceived = streamStatsData.packetsReceived
      if (this.consumerIdsWithFreezedAudio.contains(consumerId)) {
        this.logger.log("audio working fine for consumer id: $consumerId")
        this.consumerIdsWithFreezedAudio.remove(consumerId)
        callStatsObserver.onReceivingConsumerAudioStatus("resume", consumerId)
      }
    }
  }

  override suspend fun registerConsumer(consumer: ConsumerFacade) {
    this.generateConsumerStreamMap(consumer, true)

    val appData = consumer.getAppData()
    this.consumerPeerIdMap[consumer.getId()] =
      ConsumerPeerIdValue(
        producerId = consumer.getProducerId(),
        peerId = consumer.getProducingPeerId(),
      )
  }

  override suspend fun generateProducerStreamMap(
    producer: ProducerFacade,
    parse: Boolean,
  ): ProducerStatistics? {
    val stats = producer.getStats() ?: return null
    val id = producer.getId()
    val result =
      if (parse) {
        getProducerStatsFromReport(
            parseRTCReport(listOf(stats), listOf("outbound-rtp", "remote-inbound-rtp"), id)
          )
          .firstOrNull()
      } else {
        null
      }

    stats.stats.values.forEach { entry ->
      if (entry.type == "outbound-rtp") outboundProducerMap[entry.id] = id
    }

    return result
  }

  override suspend fun generateConsumerStreamMap(
    consumer: ConsumerFacade,
    parse: Boolean,
  ): ConsumerStatistics? {
    val stats = consumer.getStats() ?: return null
    val id = consumer.getId()
    val result =
      if (parse) {
        getConsumerStatsFromReport(parseRTCReport(listOf(stats), listOf("inbound-rtp"), id))
          .firstOrNull()
      } else {
        null
      }

    stats.stats.values.forEach { entry ->
      if (entry.type == "inbound-rtp") inboundConsumerMap[entry.id] = id
    }

    return result
  }

  override fun deregisterProducer(producer: ProducerFacade) {
    val producerIterator = this.outboundProducerMap.iterator()
    while (producerIterator.hasNext()) {
      val item = producerIterator.next()
      if (producer.getId() == item.value) {
        producerIterator.remove()
      }
    }
  }

  override fun deregisterConsumer(consumer: ConsumerFacade) {
    val consumerIterator = this.inboundConsumerMap.iterator()
    while (consumerIterator.hasNext()) {
      val item = consumerIterator.next()
      if (consumer.getId() == item.value) {
        consumerIterator.remove()
      }
    }
    val consumerPeerIterator = this.consumerPeerIdMap.iterator()
    while (consumerPeerIterator.hasNext()) {
      val item = consumerPeerIterator.next()
      if (consumer.getId() == item.key) {
        consumerPeerIterator.remove()
      }
    }
  }

  private fun parseBoolean(value: Any?): Boolean {
    return when (value) {
      is Boolean -> value
      is String -> value.toBoolean()
      is Int -> value == 1
      else -> false
    }
  }

  // private
  override fun parseRTCReport(
    statsList: List<RtcStatsReport>,
    statsTypeFilter: List<String>,
    ownerId: String?,
  ): ParsedRTCStats {
    val result = ParsedRTCStats()
    var candidatePair: IceCandidatePairStats
    var transport: WebRtcTransportStats
    var remoteInboundStreamStats: RemoteInboundStreamStats
    var outboundVideoStreamStats: OutboundVideoStreamStats
    var outboundAudioStreamStats: OutboundAudioStreamStats
    var inboundVideoStreamStats: InboundVideoStreamStats
    var inboundAudioStreamStats: InboundAudioStreamStats

    val rtcStatsList = statsList.map { it.stats }.map { it.values }.flatten()
    for (stat in rtcStatsList) {
      val statsType = stat.type
      val stats = stat.members

      val foundStatsType = statsTypeFilter.find { statsType == it }
      if (foundStatsType != null) {
        this.logger.log("Current Stats Type: $statsType")
        when (statsType) {
          "candidate-pair" -> {
            val nominated = stats["nominated"]

            if (nominated != null && parseBoolean(nominated)) {
              candidatePair =
                IceCandidatePairStats(
                  bytesReceived = stats["bytesReceived"].toString().toDoubleOrNull(),
                  bytesSent = stats["bytesSent"].toString().toDoubleOrNull(),
                  currentRoundTripTime = stats["currentRoundTripTime"].toString().toDoubleOrNull(),
                  totalRoundTripTime = stats["totalRoundTripTime"].toString().toDoubleOrNull(),
                  availableOutgoingBitrate =
                    stats["availableOutgoingBitrate"].toString().toDoubleOrNull(),
                )
              result.candidatePair = candidatePair
              this.logger.log("Candidate Pair: $candidatePair")
            }
          }
          "transport" -> {
            val dtlsStateList = listOf("closed", "connected", "connecting", "failed", "new")
            val receivedDTLSState = stats["dtlsState"] as String?
            val identifiedDTLSState = dtlsStateList.find { receivedDTLSState == it }
            if (identifiedDTLSState == null) {
              this.logger.log("dtlsState not in identified state of list")
            }

            val iceRoleList = listOf("controlled", "controlling", "unknown")
            val receivedIceRole = stats["iceRole"] as String?
            val identifiedIceRole = iceRoleList.find { receivedIceRole == it }
            if (identifiedIceRole == null) {
              this.logger.log("iceRole not in identified state of list")
            }

            transport =
              WebRtcTransportStats(
                bytesReceived = stats["bytesReceived"].toString().toDoubleOrNull(),
                bytesSent = stats["bytesSent"].toString().toDoubleOrNull(),
                roundTripTime = stats["roundTripTime"].toString().toDoubleOrNull(),
                totalRoundTripTime = stats["totalRoundTripTime"].toString().toDoubleOrNull(),
                availableOutgoingBitrate =
                  stats["availableOutgoingBitrate"].toString().toDoubleOrNull(),
                dtlsCipher = stats["dtlsCipher"] as String?,
                packetsReceived = stats["packetsReceived"].toString().toDoubleOrNull(),
                packetsSent = stats["packetsSent"].toString().toDoubleOrNull(),
                dtlsState = receivedDTLSState,
                iceRole = receivedIceRole,
              )

            result.transport = transport

            this.logger.log("Transport Stats: $transport")
          }
          "remote-inbound-rtp" -> {
            remoteInboundStreamStats =
              RemoteInboundStreamStats(
                jitter = stats["jitter"].toString().toDoubleOrNull(),
                fractionLost = stats["fractionLost"].toString().toDoubleOrNull(),
                roundTripTime = stats["roundTripTime"].toString().toDoubleOrNull(),
                roundTripTimeMeasurements =
                  stats["roundTripTimeMeasurements"].toString().toDoubleOrNull(),
                totalRoundTripTime = stats["totalRoundTripTime"].toString().toDoubleOrNull(),
                packetsLost = stats["packetsLost"].toString().toDoubleOrNull(),
                localId = stats["localId"] as String?,
              )

            if (remoteInboundStreamStats.localId != null) {
              result.remoteInboundRtp[remoteInboundStreamStats.localId!!] = remoteInboundStreamStats
            }

            this.logger.log("Remote Inbound Stream Stats: $remoteInboundStreamStats")
          }
          "outbound-rtp" -> {
            val id = stat.id
            val producerId = ownerId ?: outboundProducerMap[id]
            if (producerId == null) continue

            val mediaType = stats["mediaType"]
            val foundMediaType = listOf("video", "audio").find { mediaType == it }

            this.logger.log("Media Type: $mediaType and foundMediaType: $foundMediaType")
            if (foundMediaType != null) {
              if (this.outboundProducerMap[id] == null) {
                result.staleProducerStreamMap = true
                this.logger.log("Continuing because outboundProducerMap does not contain given id")
                continue
              }
              if (!result.producerStreamMap.containsKey(producerId)) {
                this.logger.log("producer id not found in producerStreamMap")
                result.producerStreamMap[producerId] =
                  ParsedProducerStats(
                    outboundVideoRtpId = mutableListOf(),
                    outboundAudioRtpId = mutableListOf(),
                  )

                if (mediaType == "video") {
                  val decoderImplementation = stats["decoderImplementation"]
                  if (decoderImplementation != null && decoderImplementation == "unknown") {
                    continue
                  }

                  remoteInboundStreamStats =
                    RemoteInboundStreamStats(
                      jitter = stats["jitter"].toString().toDoubleOrNull(),
                      fractionLost = stats["fractionLost"].toString().toDoubleOrNull(),
                      roundTripTime = stats["roundTripTime"].toString().toDoubleOrNull(),
                      roundTripTimeMeasurements =
                        stats["roundTripTimeMeasurements"].toString().toDoubleOrNull(),
                      totalRoundTripTime = stats["totalRoundTripTime"].toString().toDoubleOrNull(),
                      packetsLost = stats["packetsLost"].toString().toDoubleOrNull(),
                      localId = stats["localId"] as String?,
                    )

                  outboundVideoStreamStats =
                    OutboundVideoStreamStats(
                      hugeFramesSent = stats["hugeFramesSent"].toString().toDoubleOrNull(),
                      pliCount = stats["pliCount"].toString().toDoubleOrNull(),
                      qpSum = stats["qpSum"].toString().toDoubleOrNull(),
                      framesEncoded = stats["framesEncoded"].toString().toDoubleOrNull(),
                      framesSent = stats["framesSent"].toString().toDoubleOrNull(),
                      keyFramesEncoded = stats["keyFramesEncoded"].toString().toDoubleOrNull(),
                      encoderImplementation = stats["encoderImplementation"] as String?,
                      qualityLimitationReason = stats["qualityLimitationReason"] as String?,
                      qualityLimitationResolutionChanges =
                        stats["qualityLimitationResolutionChanges"].toString().toDoubleOrNull(),
                      totalEncodeTime = stats["totalEncodeTime"].toString().toDoubleOrNull(),
                      totalPacketSendDelay =
                        stats["totalPacketSendDelay"].toString().toDoubleOrNull(),
                      frameHeight = stats["frameHeight"].toString().toDoubleOrNull(),
                      frameWidth = stats["frameWidth"].toString().toDoubleOrNull(),
                      droppedFrames = stats["droppedFrames"].toString().toDoubleOrNull(),
                      frameRateMean = stats["frameRateMean"].toString().toDoubleOrNull(),
                      framesDropped = stats["framesDropped"].toString().toDoubleOrNull(),
                      framesPerSecond = stats["framesPerSecond"].toString().toDoubleOrNull(),
                      firCount = stats["firCount"].toString().toDoubleOrNull(),
                      bytesSent = stats["bytesSent"].toString().toDoubleOrNull(),
                      packetsSent = stats["packetsSent"].toString().toDoubleOrNull(),
                      retransmittedBytesSent =
                        stats["retransmittedBytesSent"].toString().toDoubleOrNull(),
                      retransmittedPacketsSent =
                        stats["retransmittedPacketsSent"].toString().toDoubleOrNull(),
                      remoteData = remoteInboundStreamStats,
                      nackCount = stats["nackCount"].toString().toDoubleOrNull(),
                    )

                  this.logger.log("Outbound Video Stream Stats: $outboundVideoStreamStats")
                  result.outboundVideoRtp[id] = outboundVideoStreamStats
                  result.producerStreamMap[producerId]?.outboundVideoRtpId?.add(id)
                } else if (mediaType == "audio") {
                  this.logger.log("Entered else if audio section")
                  outboundAudioStreamStats =
                    OutboundAudioStreamStats(
                      bytesSent = stats["bytesSent"].toString().toDoubleOrNull(),
                      packetsSent = stats["packetsSent"].toString().toDoubleOrNull(),
                      retransmittedBytesSent =
                        stats["retransmittedBytesSent"].toString().toDoubleOrNull(),
                      retransmittedPacketsSent =
                        stats["retransmittedPacketsSent"].toString().toDoubleOrNull(),
                      remoteData = null,
                      nackCount = stats["nackCount"].toString().toDoubleOrNull(),
                    )

                  this.logger.log("Outbound Audio Stream Stats: $outboundAudioStreamStats")
                  result.outboundAudioRtp[id] = outboundAudioStreamStats
                  result.producerStreamMap[producerId]?.outboundAudioRtpId?.add(id)
                  this.logger.log("Added id to outboundAudioRtpId")
                }
                this.logger.log("outbound-rtp ended!")
              } else {
                this.logger.log("producer id found in producerStreamMap")
              }
            }
          }
          "inbound-rtp" -> {
            val id = stat.id
            val consumerId = ownerId ?: inboundConsumerMap[id]
            if (consumerId == null) continue

            if (this.overallConsumerStatsMap[consumerId] == null) {
              this.overallConsumerStatsMap[consumerId] =
                OverallConsumerStatsValue(
                  totalVideoPacketReceived = 0.0,
                  totalAudioPacketReceived = 0.0,
                )
            }

            val overallStatsForConsumer = overallConsumerStatsMap[consumerId]!!

            val mediaType = stats["mediaType"]
            val foundMediaType = listOf("video", "audio").find { mediaType == it }
            if (foundMediaType != null) {
              if (this.inboundConsumerMap[id] == null) {
                result.staleConsumerStreamMap = true
                continue
              }

              if (!result.consumerStreamMap.containsKey(consumerId)) {
                result.consumerStreamMap[consumerId] =
                  ParsedConsumerStats(
                    inboundVideoRtpId = mutableListOf(),
                    inboundAudioRtpId = mutableListOf(),
                  )

                if (mediaType == "video") {
                  val decoderImplementation = stats["decoderImplementation"]
                  if (decoderImplementation != null && decoderImplementation == "unknown") {
                    continue
                  }

                  inboundVideoStreamStats =
                    InboundVideoStreamStats(
                      framesDecoded = stats["framesDecoded"].toString().toDoubleOrNull(),
                      keyFramesDecoded = stats["keyFramesDecoded"].toString().toDoubleOrNull(),
                      framesReceived = stats["framesReceived"].toString().toDoubleOrNull(),
                      decoderImplementation = stats["decoderImplementation"] as String?,
                      frameHeight = stats["frameHeight"].toString().toDoubleOrNull(),
                      frameWidth = stats["frameWidth"].toString().toDoubleOrNull(),
                      droppedFrames = stats["droppedFrames"].toString().toDoubleOrNull(),
                      frameRateMean = stats["frameRateMean"].toString().toDoubleOrNull(),
                      framesDropped = stats["framesDropped"].toString().toDoubleOrNull(),
                      framesPerSecond = stats["framesPerSecond"].toString().toDoubleOrNull(),
                      firCount = stats["firCount"].toString().toDoubleOrNull(),
                      bytesReceived = stats["bytesReceived"].toString().toDoubleOrNull(),
                      packetsReceived = stats["packetsReceived"].toString().toDoubleOrNull(),
                      packetsLost = stats["packetsLost"].toString().toDoubleOrNull(),
                      jitter = stats["jitter"].toString().toDoubleOrNull(),
                      nackCount = stats["nackCount"].toString().toDoubleOrNull(),
                    )

                  result.inboundVideoRtp[id] = inboundVideoStreamStats
                  result.consumerStreamMap[consumerId]?.inboundVideoRtpId?.add(id)

                  this.processInboundConsumerVideoStats(
                    consumerId,
                    overallStatsForConsumer,
                    inboundVideoStreamStats,
                  )
                } else if (mediaType == "audio") {
                  inboundAudioStreamStats =
                    InboundAudioStreamStats(
                      audioLevel = stats["audioLevel"].toString().toDoubleOrNull(),
                      concealedSamples = stats["concealedSamples"].toString().toDoubleOrNull(),
                      concealmentEvents = stats["concealmentEvents"].toString().toDoubleOrNull(),
                      jitterBufferDelay = stats["jitterBufferDelay"].toString().toDoubleOrNull(),
                      jitterBufferEmittedCount =
                        stats["jitterBufferEmittedCount"].toString().toDoubleOrNull(),
                      totalAudioEnergy = stats["totalAudioEnergy"].toString().toDoubleOrNull(),
                      totalSamplesDuration =
                        stats["totalSamplesDuration"].toString().toDoubleOrNull(),
                      totalSamplesReceived =
                        stats["totalSamplesReceived"].toString().toDoubleOrNull(),
                      bytesReceived = stats["bytesReceived"].toString().toDoubleOrNull(),
                      packetsReceived = stats["packetsReceived"].toString().toDoubleOrNull(),
                      packetsLost = stats["packetsLost"].toString().toDoubleOrNull(),
                      jitter = stats["jitter"].toString().toDoubleOrNull(),
                      nackCount = stats["nackCount"].toString().toDoubleOrNull(),
                    )

                  result.inboundAudioRtp[id] = inboundAudioStreamStats
                  result.consumerStreamMap[consumerId]?.inboundAudioRtpId?.add(id)
                  this.processInboundConsumerAudioStats(
                    consumerId,
                    overallStatsForConsumer,
                    inboundAudioStreamStats,
                  )
                }
              }
            }
          }
        }
      }
    }

    for ((key, value) in result.outboundVideoRtp) {
      value.remoteData = result.remoteInboundRtp[key]
    }

    for ((key, value) in result.outboundAudioRtp) {
      value.remoteData = result.remoteInboundRtp[key]
    }

    if (result.candidatePair != null) {
      if (result.transport != null) {
        result.transport!!.totalRoundTripTime = result.candidatePair!!.totalRoundTripTime
        result.transport!!.availableOutgoingBitrate =
          result.candidatePair!!.availableOutgoingBitrate
        result.transport!!.roundTripTime = result.candidatePair!!.currentRoundTripTime
      } else {
        WebRtcTransportStats(
            bytesReceived = result.candidatePair!!.bytesReceived,
            bytesSent = result.candidatePair!!.bytesSent,
            roundTripTime = result.candidatePair!!.currentRoundTripTime,
            totalRoundTripTime = result.candidatePair!!.totalRoundTripTime,
            availableOutgoingBitrate = result.candidatePair!!.availableOutgoingBitrate,
            dtlsCipher = null,
            packetsReceived = null,
            packetsSent = null,
            dtlsState = null,
            iceRole = null,
          )
          .also { result.transport = it }
      }
    }

    if (result.transport != null && result.transport!!.roundTripTime != null) {
      var rtt = 0.0
      var totalRTT = 0.0
      for ((_, value) in result.remoteInboundRtp) {
        if (value.roundTripTime != null && value.roundTripTime > rtt) {
          rtt = value.roundTripTime
          totalRTT = value.totalRoundTripTime!!
        }
      }

      result.transport!!.roundTripTime = rtt
      result.transport!!.totalRoundTripTime = totalRTT
    }

    return result
  }

  override suspend fun getProducersReport(
    producers: List<ProducerFacade>
  ): List<ProducerStatistics> {
    return producers.mapNotNull { producer -> generateProducerStreamMap(producer, true) }
  }

  override suspend fun getConsumersReport(
    consumers: List<ConsumerFacade>
  ): List<ConsumerStatistics> {
    return consumers.mapNotNull { consumer -> generateConsumerStreamMap(consumer, true) }
  }

  override suspend fun getTransportReport(transport: TransportFacade): List<RtcStatsReport> {
    return transport.getStats()
  }

  override suspend fun getProcessedStats(
    transport: TransportFacade,
    consuming: Boolean,
    producing: Boolean,
  ): ProcessedStatsReport {
    val transportStats = this.getTransportReport(transport)
    val result =
      this.parseRTCReport(
        transportStats,
        listOf("transport", "candidate-pair", "inbound-rtp", "outbound-rtp", "remote-inbound-rtp"),
      )

    val transportReport =
      TransportStatistics(
        stats = result.transport,
        transportId = transport.getId(),
        consuming = consuming,
        producing = producing,
      )

    val producerReport =
      if (result.staleProducerStreamMap) {
        null
      } else {
        this.getProducerStatsFromReport(result)
      }

    val consumerReport =
      if (result.staleConsumerStreamMap) {
        null
      } else {
        this.getConsumerStatsFromReport(result)
      }

    return ProcessedStatsReport(transportReport, producerReport, consumerReport)
  }

  // private
  override fun getProducerStatsFromReport(report: ParsedRTCStats): List<ProducerStatistics> {
    val producerReport: MutableList<ProducerStatistics> = mutableListOf()
    try {
      for ((key, value) in report.producerStreamMap) {
        producerReport.add(
          ProducerStatistics(
            producerId = key,
            videoStats = value.outboundVideoRtpId.map { report.outboundVideoRtp[it]!! },
            audioStats = value.outboundAudioRtpId.map { report.outboundAudioRtp[it]!! },
          )
        )
      }
    } catch (err: Error) {
      this.logger.logError("getProducersStatsFromReport:  $err, $report")
    }

    return producerReport
  }

  // private
  override fun getConsumerStatsFromReport(report: ParsedRTCStats): List<ConsumerStatistics> {
    val consumerReport: MutableList<ConsumerStatistics> = mutableListOf()
    try {
      for ((key, value) in report.consumerStreamMap) {
        consumerReport.add(
          ConsumerStatistics(
            consumerId = key,
            peerId = this.consumerPeerIdMap[key]?.peerId,
            producerId = this.consumerPeerIdMap[key]?.producerId,
            videoStats = value.inboundVideoRtpId.map { report.inboundVideoRtp[it]!! },
            audioStats = value.inboundAudioRtpId.map { report.inboundAudioRtp[it]!! },
          )
        )
      }
    } catch (err: Error) {
      this.logger.logError("getConsumerStatsFromReport:  $err, $report")
    }

    return consumerReport
  }

  // private
  override suspend fun getConnectivity(iceServers: List<IceServer>, observer: TestObserver) {
    try {
      var finalIceServers = iceServers
      if (iceServers.isEmpty()) {
        finalIceServers = fallBackIceServers
      }
      val config = RtcConfiguration(iceServers = finalIceServers)

      var hostConnTestResult: TestResult? = null
      var relayConnTestResult: TestResult? = null
      var reflexiveConnTestResult: TestResult? = null

      val onHostConnTestDone: suspend (Any) -> Unit = { hostTestResult: Any ->
        hostConnTestResult = hostTestResult as TestResult
        this.logger.log("Host Connectivity Done!")

        val onRelayConnTestDone: suspend (Any) -> Unit = { relayTestResult: Any ->
          relayConnTestResult = relayTestResult as TestResult
          this.logger.log("Relay Connectivity Done!")

          val onReflexiveConnTestDone = { reflexiveTestResult: Any ->
            reflexiveConnTestResult = reflexiveTestResult as TestResult
            this.logger.log("Reflexive Connectivity Done!")

            observer.onDone(
              IceConnectivity(
                host = hostConnTestResult?.connectivity,
                relay = relayConnTestResult?.connectivity,
                reflexive = reflexiveConnTestResult?.connectivity,
              )
            )
          }

          val onReflexiveConnTestFailure = { reason: String ->
            this.logger.log("Reflexive Connectivity Failed! reason: $reason")
            observer.onFailure(
              reason,
              IceConnectivity(
                host = hostConnTestResult?.connectivity,
                relay = relayConnTestResult?.connectivity,
                reflexive = reflexiveConnTestResult?.connectivity,
              ),
            )
          }

          val reflexiveConnTest =
            ReflexiveConnectivityTest(
              config,
              this.provider,
              onReflexiveConnTestDone,
              onReflexiveConnTestFailure,
              coroutineScope,
            )
          reflexiveConnTest.start(7000)
        }

        val onRelayConnTestFailure: suspend (String) -> Unit = { reason: String ->
          this.logger.log("Relay Connectivity Failed! reason: $reason")
          observer.onFailure(
            reason,
            IceConnectivity(
              host = hostConnTestResult?.connectivity,
              relay = relayConnTestResult?.connectivity,
              reflexive = reflexiveConnTestResult?.connectivity,
            ),
          )
        }

        val relayConnTest =
          RelayConnectivityTest(
            config,
            this.provider,
            onRelayConnTestDone,
            onRelayConnTestFailure,
            coroutineScope,
          )
        relayConnTest.start(7000)
      }

      val onHostConnTestFailure = { reason: String ->
        this.logger.log("Host Connectivity Failed! reason: $reason")
        observer.onFailure(
          reason,
          IceConnectivity(
            host = hostConnTestResult?.connectivity,
            relay = relayConnTestResult?.connectivity,
            reflexive = reflexiveConnTestResult?.connectivity,
          ),
        )
      }

      val hostConnTest =
        HostConnectivityTest(
          config,
          this.provider,
          onHostConnTestDone,
          onHostConnTestFailure,
          coroutineScope,
        )
      hostConnTest.start(5000)
    } catch (ex: Exception) {
      this.logger.logError("Error while executing connectivity tests: $ex")
      observer.onError(ex)
    }
  }

  // private
  override suspend fun getThroughput(iceServers: List<IceServer>, observer: TestObserver) {
    try {
      var finalIceServers = iceServers
      if (iceServers.isEmpty()) {
        finalIceServers = fallBackIceServers
      }
      val config = RtcConfiguration(iceServers = finalIceServers)

      val testDone = { data: Any ->
        val testResults = data as DataThroughputTestResults
        observer.onDone(
          ThroughputInformation(
            throughput = testResults.throughput,
            fractionLoss = 0,
            RTT = testResults.RTT,
            jitter = 0,
            backendRTT = testResults.backendRTT,
          )
        )
      }

      val testFailed = { reason: String ->
        this.logger.log("Throughput test failed! Reason: $reason")
        observer.onFailure(reason, null)
      }

      val test = DataThroughputTest(config, this.provider, testDone, testFailed, coroutineScope)
      test.start(20000)
    } catch (ex: Exception) {
      this.logger.logError("Error while executing throughput tests: $ex")
      observer.onError(ex)
    }
  }

  override suspend fun getIpDetails(): IPDetails {
    return IPApiService().getIpDetails()
  }

  override suspend fun getNetworkQuality(iceServers: List<IceServer>, observer: TestObserver) {
    try {
      var connTestResult: IceConnectivity? = null
      var throughputTestResult: ThroughputInformation? = null

      val onConnDone = { data: Any ->
        connTestResult = data as IceConnectivity
        if (connTestResult != null && throughputTestResult != null) {
          observer.onDone(
            NetworkQualityInformation(
              connectivity = connTestResult!!,
              throughput = throughputTestResult!!.throughput,
              fractionLoss = throughputTestResult!!.fractionLoss,
              RTT = throughputTestResult!!.RTT,
              jitter = throughputTestResult!!.jitter,
              backendRTT = throughputTestResult!!.backendRTT,
            )
          )
        }
      }

      val onConnFailure = { reason: String, _: Any? -> observer.onFailure(reason, null) }

      val onErrorCb = { ex: Exception -> observer.onError(ex) }

      val onThroughputDone = { data: Any ->
        throughputTestResult = data as? ThroughputInformation
        if (connTestResult != null && throughputTestResult != null) {
          observer.onDone(
            NetworkQualityInformation(
              connectivity = connTestResult!!,
              throughput = throughputTestResult!!.throughput,
              fractionLoss = throughputTestResult!!.fractionLoss,
              RTT = throughputTestResult!!.RTT,
              jitter = throughputTestResult!!.jitter,
              backendRTT = throughputTestResult!!.backendRTT,
            )
          )
        }
      }

      val onThroughputFailure = { reason: String, _: Any? -> observer.onFailure(reason, null) }

      val connTestObserver = InternalTestObserver(onConnDone, onConnFailure, onErrorCb)
      val throughputTestObserver =
        InternalTestObserver(onThroughputDone, onThroughputFailure, onErrorCb)

      this.getConnectivity(iceServers, connTestObserver)
      this.getThroughput(iceServers, throughputTestObserver)
    } catch (ex: Exception) {
      this.logger.logError("Error while executing network quality test: $ex")
      observer.onError(ex)
    }
  }

  override suspend fun getNetworkInfo(iceServers: List<IceServer>, observer: TestObserver) {
    try {
      val ipDetails: IPDetails?

      ipDetails = runBlocking { getIpDetails() }

      val onDoneCb = { data: Any ->
        val results = data as NetworkQualityInformation
        val connTestResult = results.connectivity
        observer.onDone(
          NetworkInformation(
            ipDetails,
            // TODO: Check what to fill here
            effectiveNetworkType = null,
            location = ipDetails.location,
            turnConnectivity =
              if (connTestResult != null) {
                connTestResult.host == true ||
                  connTestResult.relay == true ||
                  connTestResult.reflexive == true
              } else {
                false
              },
            connectivity = connTestResult,
            throughput = results.throughput,
            fractionLoss = results.fractionLoss,
            RTT = results.RTT,
            jitter = results.jitter,
            backendRTT = results.backendRTT,
          )
        )
      }

      val onFailureCb = { reason: String, lastUpdatedResults: Any? ->
        observer.onFailure(reason, lastUpdatedResults)
      }

      val onErrorCb = { ex: Exception -> observer.onError(ex) }

      val resultObserver = InternalTestObserver(onDoneCb, onFailureCb, onErrorCb)

      getNetworkQuality(iceServers, resultObserver)
    } catch (ex: Exception) {
      this.logger.logError("Error while executing network quality test: $ex")
      observer.onError(ex)
    }
  }
}

// TODO: Think Remove this or not?
// fun getNetworkClass(context: Context): String? {
//    val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
//    val info = cm.activeNetworkInfo
//    if (info == null || !info.isConnected) return "-" // not connected
//    if (info.type == ConnectivityManager.TYPE_WIFI) return "WIFI"
//    if (info.type == ConnectivityManager.TYPE_MOBILE) {
//        val networkType = info.subtype
//        return when (networkType) {
//            TelephonyManager.NETWORK_TYPE_GPRS, TelephonyManager.NETWORK_TYPE_EDGE,
// TelephonyManager.NETWORK_TYPE_CDMA, TelephonyManager.NETWORK_TYPE_1xRTT,
// TelephonyManager.NETWORK_TYPE_IDEN, TelephonyManager.NETWORK_TYPE_GSM -> "2G"
//            TelephonyManager.NETWORK_TYPE_UMTS, TelephonyManager.NETWORK_TYPE_EVDO_0,
// TelephonyManager.NETWORK_TYPE_EVDO_A, TelephonyManager.NETWORK_TYPE_HSDPA,
// TelephonyManager.NETWORK_TYPE_HSUPA, TelephonyManager.NETWORK_TYPE_HSPA,
// TelephonyManager.NETWORK_TYPE_EVDO_B, TelephonyManager.NETWORK_TYPE_EHRPD,
// TelephonyManager.NETWORK_TYPE_HSPAP, TelephonyManager.NETWORK_TYPE_TD_SCDMA -> "3G"
//            TelephonyManager.NETWORK_TYPE_LTE, TelephonyManager.NETWORK_TYPE_IWLAN, 19 -> "4G"
//            TelephonyManager.NETWORK_TYPE_NR -> "5G"
//            else -> "?"
//        }
//    }
//    return "?"
// }

//    suspend fun getTest(observer: TestObserver) {
//        try {
//            val iceServers = IceServersApiService().getIceServers(this.authToken)
////            val results = this.getConnectivity(iceServers)
////            this.logger.log("All Test Results: $results")
//            val results = this.getThroughput(iceServers, observer)
//            this.logger.log("Throughput Test Results: $results")
//        } catch (ex: Exception) {
//            this.logger.logError("Exception while fetching ice servers: $ex")
//            observer.onError(ex)
//        }
//    }
