package io.dyte.callstats.api

import io.dyte.callstats.models.Environment
import io.dyte.callstats.models.SQSPacketStructure
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.encodeToJsonElement

class CollectorApiService(environment: Environment) : BaseApiService() {
  private val endpoint = if (environment == Environment.PRODUCTION) {
    "https://collector.prod.da.dyte.io/api/v1/message"
  } else {
    "https://collector.non-prod.da.dyte.io/api/v1/message"
  }

  suspend fun sendCallStats(packet: SQSPacketStructure) {
    val body = json.encodeToString(SQSPacketStructure.serializer(), packet)

    val data = (json.parseToJsonElement(body) as JsonObject).toMutableMap()
    val payload = (data[KEY_PAYLOAD] as JsonObject).toMutableMap()
    val entries = (payload[KEY_ENTRIES] as JsonArray).toMutableList()

    entries.forEachIndexed { index, it ->
      // Remove type field added to every EventEntry subtype by Kotlinx serialization
      val entry = (it as JsonObject).toMutableMap()
      entry.remove(KEY_ENTRY_TYPE)

      // Remove type field from EventData subtype, if added by Kotlinx serialization
      val metaDataMap = (entry[KEY_METADATA] as JsonObject).toMutableMap()
      if (metaDataMap.containsKey(KEY_METADATA_TYPE)) {
        metaDataMap.remove(KEY_METADATA_TYPE)
        entry[KEY_METADATA] = json.encodeToJsonElement(metaDataMap)
      }

      /*
      * Hack to change the event name of CALL_JOIN_BEGIN to "call_join".
      * TODO: verify the events and their names with what is expected by the backend
      * */
      val event = (entry[KEY_ENTRY_EVENT] as? JsonPrimitive)?.content
      if (event != null && event == "call_join_begin") {
        entry[KEY_ENTRY_EVENT] = json.encodeToJsonElement("call_join")
      }

      entries[index] = json.encodeToJsonElement(entry)
    }

    payload[KEY_ENTRIES] = json.encodeToJsonElement(entries)
    data[KEY_PAYLOAD] = json.encodeToJsonElement(payload)

    client.post(endpoint) { setBody(json.encodeToString(data)) }
  }

  companion object {
    private const val KEY_PAYLOAD = "payload"
    private const val KEY_ENTRIES = "entries"
    private const val KEY_ENTRY_TYPE = "type"
    private const val KEY_METADATA = "metaData"
    private const val KEY_METADATA_TYPE = "type"
    private const val KEY_ENTRY_EVENT = "event"
  }
}
