package io.dyte.callstats.events

import io.dyte.callstats.*
import kotlinx.datetime.Instant
import kotlinx.serialization.Contextual
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.Serializer
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

enum class MediaDeviceType {
  AUDIO,
  VIDEO,
  SPEAKER
}

enum class EventCategory {
  MAJOR_EVENT,
  MINOR_EVENT
}

@Serializable(with = EventSerializer::class)
enum class Event {
  PRECALL_TEST_BEGIN,
  PRECALL_TEST_COMPLETE,
  CALL_JOIN_BEGIN,
  NET_QUALITY_TEST_BEGIN,
  NET_QUALITY_TEST_END,
  WEBSOCKET_CONNECTED,
  TRANSPORT_CONNECTED,
  AUDIO_ON,
  AUDIO_OFF,
  VIDEO_ON,
  VIDEO_OFF,
  PARTICIPANT_ROLE,
  PING_STAT,
  DISCONNECT,
  RECONNECT_ATTEMPT,
  SCREENSHARE_START_REQUESTED,
  SCREENSHARE_STARTED,
  SCREENSHARE_STOPPED,
  TAB_CHANGE,
  BROWSER_BACKGROUNDED,
  BROWSER_FOREGROUNDED,
  DOMINANT_SPEAKER,
  AUDIO_DEVICES_UPDATES,
  VIDEO_DEVICES_UPDATES,
  SPEAKER_DEVICES_UPDATES,
  SELECTED_MICROPHONE_UPDATE,
  SELECTED_CAMERA_UPDATE,
  SELECTED_SPEAKER_UPDATE,
  MEDIA_PERMISSION,
  LEGACY_SWITCH,
  AUDIO_PLAY_FAILED,
  VIDEO_PLAY_FAILED,
  AUDIO_TRACK_MUTED,
  VIDEO_TRACK_MUTED
}

val eventCategoryMap: Map<Event, EventCategory> =
  mapOf(
    Event.PRECALL_TEST_BEGIN to EventCategory.MINOR_EVENT,
    Event.PRECALL_TEST_COMPLETE to EventCategory.MINOR_EVENT,
    Event.CALL_JOIN_BEGIN to EventCategory.MAJOR_EVENT,
    Event.NET_QUALITY_TEST_BEGIN to EventCategory.MINOR_EVENT,
    Event.NET_QUALITY_TEST_END to EventCategory.MINOR_EVENT,
    Event.WEBSOCKET_CONNECTED to EventCategory.MINOR_EVENT,
    Event.TRANSPORT_CONNECTED to EventCategory.MAJOR_EVENT,
    Event.AUDIO_ON to EventCategory.MINOR_EVENT,
    Event.AUDIO_OFF to EventCategory.MINOR_EVENT,
    Event.VIDEO_ON to EventCategory.MINOR_EVENT,
    Event.VIDEO_OFF to EventCategory.MINOR_EVENT,
    Event.PARTICIPANT_ROLE to EventCategory.MINOR_EVENT,
    Event.PING_STAT to EventCategory.MAJOR_EVENT,
    Event.DISCONNECT to EventCategory.MAJOR_EVENT,
    Event.RECONNECT_ATTEMPT to EventCategory.MAJOR_EVENT,
    Event.SCREENSHARE_START_REQUESTED to EventCategory.MINOR_EVENT,
    Event.SCREENSHARE_STARTED to EventCategory.MINOR_EVENT,
    Event.SCREENSHARE_STOPPED to EventCategory.MINOR_EVENT,
    Event.TAB_CHANGE to EventCategory.MINOR_EVENT,
    Event.BROWSER_BACKGROUNDED to EventCategory.MINOR_EVENT,
    Event.BROWSER_FOREGROUNDED to EventCategory.MINOR_EVENT,
    Event.DOMINANT_SPEAKER to EventCategory.MINOR_EVENT,
    Event.AUDIO_DEVICES_UPDATES to EventCategory.MINOR_EVENT,
    Event.VIDEO_DEVICES_UPDATES to EventCategory.MINOR_EVENT,
    Event.SPEAKER_DEVICES_UPDATES to EventCategory.MINOR_EVENT,
    Event.SELECTED_MICROPHONE_UPDATE to EventCategory.MINOR_EVENT,
    Event.SELECTED_CAMERA_UPDATE to EventCategory.MINOR_EVENT,
    Event.SELECTED_SPEAKER_UPDATE to EventCategory.MINOR_EVENT,
    Event.MEDIA_PERMISSION to EventCategory.MINOR_EVENT,
    Event.LEGACY_SWITCH to EventCategory.MINOR_EVENT,
    Event.AUDIO_PLAY_FAILED to EventCategory.MINOR_EVENT,
    Event.VIDEO_PLAY_FAILED to EventCategory.MINOR_EVENT,
    Event.AUDIO_TRACK_MUTED to EventCategory.MINOR_EVENT,
    Event.VIDEO_TRACK_MUTED to EventCategory.MINOR_EVENT,
  )

@Serializable
data class DeviceInfo(
  var isMobile: Boolean = true,
  val browserName: String? = null,
  val browserVersion: String? = null,
  var osName: String,
  var osVersionName: String,
  val engineName: String? = null,
  var userAgent: String? = null,
  var cpus: Int,
  var memory: Long?,
  val webGLSupport: Int? = null // 0, 1, 2 or undefined
)

@Serializable
data class PeerMetaData(
  val metaData: Map<String, @Contextual Any?>,
  var deviceInfo: DeviceInfo,
  val displayName: String,
  var meetingEnv: String?,
  val peerId: String,
  val userId: String,
  val clientSpecificId: String?,
  val roomName: String,
  val roomUUID: String,
  val permissions: Map<String, @Contextual Any?>,
  val participantRole: String?,
  val roomViewType: String,
  val sdkName: String,
  val sdkVersion: String
)

@Serializable
data class TransportStatistics(
  val transportId: String?,
  val consuming: Boolean?,
  val producing: Boolean?,
  val stats: WebRtcTransportStats?,
)

// data class NetworkQualityInformation(
//    val regionId: String,
//    val iceServers: List<IceServerInfo>,
//    val throughput: Long,
//    val fractionLoss: Long,
//    val RTT: Long,
//    val jitter: Long,
//    val backendRTT: Long,
//    val connectivity: Boolean
// )

@Serializable
sealed interface EventData {
  @Serializable class CallJoinData(val peerMetaData: PeerMetaData) : EventData

  @Serializable
  class PingStatsData(
    val producingTransportStats: TransportStatistics?,
    val consumingTransportStats: TransportStatistics?,
    val producerStats: List<ProducerStatistics>?,
    val consumerStats: List<ConsumerStatistics>?
  ) : EventData

  @Serializable
  class PreCallTestData(
    val connectionInfo: NetworkInformation?,
  ) : EventData

  @Serializable
  class RegionalNetworkQualityTestData(
    val networkResults: NetworkQualityInformation,
  ) : EventData

  @Serializable
  class NetworkQualityTestData(
    val regionData: List<RegionalNetworkQualityTestData>,
  ) : EventData

  @Serializable class ParticipantRoleData(val participantRole: String) : EventData

  @Serializable object EmptyData : EventData

  @Serializable class Object(val data: Map<String, @Contextual Any>) : EventData
}

// TODO: Try to find this class in org.webrtc again, if found uncomment *DevicesEntry class
// and also implement devices() and selectedDevices() function in CallStats.kt
// sealed class MediaDevicesData(deviceList: List<Media>): EventData

@Serializable
sealed class EventEntry {
  abstract val event: Event
  abstract var timestamp: Instant
  abstract val metaData: EventData

  @Serializable
  class PreCallTestBeginEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event = Event.PRECALL_TEST_BEGIN
  }

  @Serializable
  class PreCallTestCompleteEntry(
    override var timestamp: Instant,
    override val metaData: EventData.PreCallTestData
  ) : EventEntry() {
    override val event: Event = Event.PRECALL_TEST_COMPLETE
  }

  @Serializable
  class CallJoinBeginEntry(
    override var timestamp: Instant,
    override val metaData: EventData.CallJoinData
  ) : EventEntry() {
    override val event: Event = Event.CALL_JOIN_BEGIN
  }

  @Serializable
  class NetworkQualityTestBeginEntry(
    override var timestamp: Instant,
    override val metaData: EventData
  ) : EventEntry() {
    override val event: Event = Event.NET_QUALITY_TEST_BEGIN
  }

  @Serializable
  class NetworkQualityTestEndEntry(
    override var timestamp: Instant,
    override val metaData: EventData.NetworkQualityTestData
  ) : EventEntry() {
    override val event: Event = Event.NET_QUALITY_TEST_END
  }

  @Serializable
  class WebSocketConnectedEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.WEBSOCKET_CONNECTED
  }

  @Serializable
  class TransportConnectedEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.TRANSPORT_CONNECTED
  }

  @Serializable
  class AudioOffEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.AUDIO_OFF
  }

  @Serializable
  class AudioOnEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.AUDIO_ON
  }

  @Serializable
  class VideoOffEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.VIDEO_OFF
  }

  @Serializable
  class VideoOnEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.VIDEO_ON
  }

  @Serializable
  class ScreenShareStartedEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.SCREENSHARE_STARTED
  }

  @Serializable
  class ScreenShareStoppedEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.SCREENSHARE_STOPPED
  }

  @Serializable
  class ScreenShareRequestedEntry(
    override var timestamp: Instant,
    override val metaData: EventData
  ) : EventEntry() {
    override val event: Event = Event.SCREENSHARE_START_REQUESTED
  }

  @Serializable
  class DominantSpeakerEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.DOMINANT_SPEAKER
  }

  // class AudioDevicesEntry(timestamp: Instant, metaData: MediaDeviceInfo) :
  //    EventEntryBase(Event.AUDIO_DEVICES_UPDATES, timestamp, metaData)
  //
  // class VideoDevicesEntry(timestamp: Instant, metaData: MediaDeviceInfo) :
  //    EventEntryBase(Event.VIDEO_DEVICES_UPDATES, timestamp, metaData)
  //
  // class SelectedCameraEntry(timestamp: Instant, metaData: MediaDeviceInfo) :
  //    EventEntryBase(Event.SELECTED_CAMERA_UPDATE, timestamp, metaData)
  //
  // class SelectedMicrophoneEntry(timestamp: Instant, metaData: MediaDeviceInfo) :
  //    EventEntryBase(Event.SELECTED_MICROPHONE_UPDATE, timestamp, metaData)
  //
  // class SelectedSpeakerEntry(timestamp: Instant, metaData: MediaDeviceInfo) :
  //    EventEntryBase(Event.SELECTED_SPEAKER_UPDATE, timestamp, metaData)

  @Serializable
  class MediaPermissionEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.MEDIA_PERMISSION
  }

  @Serializable
  class AudioPlaybackFailureEntry(
    override var timestamp: Instant,
    override val metaData: EventData
  ) : EventEntry() {
    override val event: Event = Event.AUDIO_PLAY_FAILED
  }

  @Serializable
  class VideoPlaybackFailureEntry(
    override var timestamp: Instant,
    override val metaData: EventData
  ) : EventEntry() {
    override val event: Event = Event.VIDEO_PLAY_FAILED
  }

  @Serializable
  class AudioTrackFailureEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.AUDIO_TRACK_MUTED
  }

  @Serializable
  class VideoTrackFailureEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.VIDEO_TRACK_MUTED
  }

  // Not implementing TabChangeEntry, BrowserBackgroundedEntry, BrowserForegroundedEntry

  @Serializable
  class LegacySwitchEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.LEGACY_SWITCH
  }

  @Serializable
  class ParticipantRoleToggleEntry(
    override var timestamp: Instant,
    override val metaData: EventData.ParticipantRoleData
  ) : EventEntry() {
    override val event: Event = Event.PARTICIPANT_ROLE
  }

  @Serializable
  class PingStatsEntry(
    override var timestamp: Instant,
    override val metaData: EventData.PingStatsData
  ) : EventEntry() {
    override val event: Event = Event.PING_STAT
  }

  @Serializable
  class DisconnectEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.DISCONNECT
  }

  @Serializable
  class ReconnectAttemptEntry(
    override var timestamp: Instant,
    override val metaData: EventData,
  ) : EventEntry() {
    override val event: Event = Event.RECONNECT_ATTEMPT
  }
}

@Serializable class EventChunk(val entries: MutableList<EventEntry>)

@OptIn(ExperimentalSerializationApi::class)
@Serializer(forClass = Event::class)
object EventSerializer : KSerializer<Event> {

  override val descriptor: SerialDescriptor = PrimitiveSerialDescriptor("event", PrimitiveKind.STRING)
  override fun serialize(encoder: Encoder, value: Event) {
    encoder.encodeString(value.name.lowercase())
  }
  override fun deserialize(decoder: Decoder): Event {
    return Event.valueOf(decoder.decodeString().uppercase())
  }
}
