/*
 * Copyright © 2015 - 2021 ReSys (info@dialob.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.dialob.service.common.security;

import java.util.Arrays;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.web.access.ExceptionTranslationFilter;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.web.cors.CorsConfiguration;

import io.dialob.security.spring.tenant.DefaultTenantSupplier;
import io.dialob.security.spring.tenant.RequestParameterTenantScopeFilter;
import io.dialob.security.spring.tenant.TenantAccessEvaluator;
import io.dialob.security.tenant.ImmutableTenant;
import io.dialob.settings.DialobSettings;
import io.dialob.settings.DialobSettings.TenantSettings;
import io.dialob.settings.DialobSettings.TenantSettings.Mode;

public class SecurityDisabledConfigurer extends WebSecurityConfigurerAdapter {

  private static final Logger LOGGER = LoggerFactory.getLogger(SecurityDisabledConfigurer.class);

  private static final CorsConfiguration PERMIT_ALL_CORS;

  static {
    PERMIT_ALL_CORS = new CorsConfiguration();
    PERMIT_ALL_CORS.setAllowedMethods(Arrays.asList("GET","HEAD","POST","PUT","DELETE"));
    PERMIT_ALL_CORS.applyPermitDefaultValues();
  }

  private TenantSettings tenantSettings;
  
  public SecurityDisabledConfigurer(DialobSettings dialobSettings) {
    this.tenantSettings = dialobSettings.getTenant();
  }

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    LOGGER.warn("Security disabled!");
    http.antMatcher("/**")
        .authorizeRequests().anyRequest().permitAll()
      .and()
        .csrf().disable()
        .cors().configurationSource(request -> PERMIT_ALL_CORS)
      .and().headers()
        .frameOptions().disable();
    configureRequestParameterTenantScopeFilter(http);
  }
  
  protected HttpSecurity configureRequestParameterTenantScopeFilter(HttpSecurity http) {
    // @formatter:off
    getRequestParameterTenantScopeFilter()
      .ifPresent(requestParameterTenantScopeFilter -> http.addFilterAfter(requestParameterTenantScopeFilter, ExceptionTranslationFilter.class));
    return http;
    // @formatter:on
  }
  
  @NonNull
  protected Optional<RequestParameterTenantScopeFilter> getRequestParameterTenantScopeFilter() {
    final RequestParameterTenantScopeFilter requestParameterTenantScopeFilter = new RequestParameterTenantScopeFilter(
      getTenantPermissionEvaluator(),
      getDefaultTenantSupplier()
    );
    requestParameterTenantScopeFilter.setTenantRequiredMatcher(getTenantRequiredMatcher());
    requestParameterTenantScopeFilter.setParameterName(tenantSettings.getUrlParameter());
    return Optional.of(requestParameterTenantScopeFilter);
  }
  
  private DefaultTenantSupplier getDefaultTenantSupplier() {
    if (StringUtils.isEmpty(tenantSettings.getFixedId())) {
      return () -> Optional.empty();
    }
    return () -> Optional.of(ImmutableTenant.of(tenantSettings.getFixedId(), Optional.empty()));
  }

  @NonNull
  protected RequestMatcher getTenantRequiredMatcher() {
    return request -> tenantSettings.getMode() == Mode.URL_PARAM;
  }

  @NonNull
  protected TenantAccessEvaluator getTenantPermissionEvaluator() {
    return tenant -> true;
  }
}

