package io.adbrix.sdk.utils;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;

import org.json.JSONException;
import org.json.JSONObject;

import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.function.Function;
import io.adbrix.sdk.domain.function.ThrowingFunction;

public class CoreUtils {

    private CoreUtils() {
    }

    public static SimpleDateFormat createDateFormat(String pattern) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(pattern, Locale.US);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT+0"));
        return dateFormat;
    }

    public static String getDateViaGMTDateFormat(String format) {
        DateFormat date = new SimpleDateFormat("z", Locale.getDefault());
        String gmt = date.format(System.currentTimeMillis());
        SimpleDateFormat utcDateFormat = new SimpleDateFormat(format, Locale.US);
        utcDateFormat.setTimeZone(TimeZone.getTimeZone(gmt));
        return utcDateFormat.format(new Date());
    }

    // HMAC-SHA256 Algorithm을 이용한 결과값을 반환한다. Exception 발생 시, Empty String을 반환한다.
    public static String hmacSha256(String key, String message) {
        try {
            String algorithm = "HmacSHA256";
            Mac mac = Mac.getInstance(algorithm);
            SecretKeySpec secretKey = new SecretKeySpec(key.getBytes(), algorithm);
            mac.init(secretKey);
            byte[] hash = mac.doFinal(message.getBytes());

            StringBuffer buffer = new StringBuffer();
            for (int i = 0; i < hash.length; i++) {
                int d = hash[i];

                if (d < 0) {
                    d = d + 0x100;
                }

                if (d < 16) {
                    buffer.append("0");
                }

                buffer.append(Integer.toString(d, 0x10));
            }

            return buffer.toString();
        } catch (NoSuchAlgorithmException | InvalidKeyException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return "";
    }

    public static String md5(String s) {
        String md5 = "";

        try {

            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(s.getBytes());
            byte[] byteData = md.digest();

            StringBuffer sb = new StringBuffer();

            for (int i = 0; i < byteData.length; i++) {
                sb.append(Integer.toString((byteData[i] & 0xff) + 0x100, 16).substring(1));
            }

            md5 = sb.toString();
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            md5 = null;
        }
        return md5;
    }

    public static byte[] sha256(String msg) throws NoSuchAlgorithmException {
        MessageDigest md = MessageDigest.getInstance("SHA-256");
        md.update(msg.getBytes(Charset.forName("UTF-8")));

        return md.digest();
    }

    public static long toUInt32(byte[] bytes, int offset) {
        long result = toUnsignedLong(bytes[offset]);
        result |= toUnsignedLong(bytes[offset + 1]) << 8;
        result |= toUnsignedLong(bytes[offset + 2]) << 16;
        result |= toUnsignedLong(bytes[offset + 3]) << 24;
        return result;
    }

    public static long toUnsignedLong(byte x) {
        return ((long) x) & 0xffL;
    }

    public static int getSHA256HashToInt(String plainText, int digit) {
        if (plainText.isEmpty() || digit < 1)
            return -1;

        byte[] encoded;
        try {
            encoded = sha256(plainText);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return -1;
        }

        double hashed = toUInt32(encoded, 0) % Math.pow(10, digit);
        return (int) hashed;
    }

    public static Object convertNullableToJSONNullable(Object object) {
        return object == null ? JSONObject.NULL : object;
    }

    public static JSONObject getJSONObjectFromMap(Map<String, Object> map) {
        JSONObject jsonObject = new JSONObject();
        try {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                jsonObject.put(key, value);
            }
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return jsonObject;
    }

    public static String getValueOfJsonObject(String uriStr, String key) {
        try {
            JSONObject myObj = new JSONObject(uriStr);
            if (CommonUtils.isNullOrEmpty(key) || myObj.isNull(key)) return null;
            else return String.valueOf(myObj.get(key));
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return "JsonException error : " + e.getMessage();
        }
    }

    public static String get02XMD5(String s) {
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(s.getBytes());
            byte[] messageDigest = digest.digest();

            // Create Hex String
            StringBuffer hexString = new StringBuffer();
            for (int i = 0; i < messageDigest.length; i++) {
                hexString.append(String.format("%02X", 0xFF & messageDigest[i]));
            }

            String temp = hexString.toString();
            return temp.toLowerCase(Locale.ENGLISH);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.w("md5 error: " + e.getMessage(), true);
        }
        return s;
    }

    public static synchronized boolean isOnline(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Network network = connectivityManager.getActiveNetwork();
            if (network != null) {
                NetworkCapabilities networkCapabilities = connectivityManager.getNetworkCapabilities(network);
                if (networkCapabilities != null) {
                    return networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) || networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR);
                }
            }
            return false;
        } else {
            NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
            return activeNetworkInfo != null && activeNetworkInfo.isConnected();
        }
    }

    // Get immutable URI
    public static Uri getDeeplinkUriFromCurrentActivity(Activity activity) {
        if (activity != null && activity.getIntent() != null && activity.getIntent().getData() != null) {

            Uri iUri = activity.getIntent().getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            return iUri;
        } else return null;
    }

    public static Uri getDeeplinkUriFromCurrentIntent(Intent deeplinkIntent) {
        if (deeplinkIntent != null && deeplinkIntent.getData() != null) {

            Uri iUri = deeplinkIntent.getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            return iUri;
        } else return null;
    }

    public static void clearAdbrixTrackingParamFromCurrentActivity(Activity activity, DataRegistry dataRegistry) {
        if (activity != null && activity.getIntent() != null && activity.getIntent().getData() != null) {

            Uri iUri = activity.getIntent().getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            if (iUri.toString().indexOf('#') != -1)
                iUri = Uri.parse(iUri.toString().replaceAll("\\#", "%32"));
            String uriString = activity.getIntent().getDataString();
            uriString = Uri.decode(uriString);

            if (iUri != null && iUri.isHierarchical()) {
                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                } else {
                    AbxLog.d("abx_tid doesn't exist", true);
                }

                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                } else {
                    AbxLog.d("abx_tracker_id doesn't exist", true);
                }

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

                if (isApplink(iUri, appkey)) {
                    if (!CommonUtils.isNullOrEmpty(uriString) && iUri.getQueryParameter(CoreConstants.IS_USED_APPLINK_KEY) == null) {
                        uriString = Uri.parse(uriString).buildUpon().appendQueryParameter(CoreConstants.IS_USED_APPLINK_KEY, "true").build().toString();
                    }
                }

                if ((iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) ||
                        isApplink(iUri, appkey)) {
                    Uri newData = Uri.parse(uriString);
                    activity.getIntent().setData(newData);
                } else {
                    AbxLog.e("Check Adbrix Parameters :: There is no Deeplink information in Activity!", true);
                }
            }
        }
    }

    public static void clearAdbrixTrackingParamFromCurrentIntent(Intent deeplinkIntent, DataRegistry dataRegistry) {
        if (deeplinkIntent != null && deeplinkIntent.getData() != null) {

            Uri iUri = deeplinkIntent.getData(); // Immutable URI reference
            iUri = Uri.parse(Uri.decode(iUri.toString()));

            if (iUri.toString().indexOf('#') != -1)
                iUri = Uri.parse(iUri.toString().replaceAll("\\#", "%32"));
            String uriString = deeplinkIntent.getDataString();
            uriString = Uri.decode(uriString);

            if (iUri != null && iUri.isHierarchical()) {
                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                } else {
                    AbxLog.d("abx_tid doesn't exist", true);
                }

                if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) {
                    String paramString = CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY);
                    if (iUri.getQuery().length() == paramString.length()) {
                        paramString = "\\?" + paramString;
                    } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                        paramString = "&" + paramString;
                    } else {
                        paramString = paramString + "&";
                    }
                    if (!CommonUtils.isNullOrEmpty(uriString)) {
                        uriString = uriString.replaceFirst(paramString, "");
                    }
                } else {
                    AbxLog.d("abx_tracker_id doesn't exist", true);
                }

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

                if (isApplink(iUri, appkey)) {

                    if (!CommonUtils.isNullOrEmpty(uriString) && iUri.getQueryParameter(CoreConstants.IS_USED_APPLINK_KEY) == null) {
                        uriString = Uri.parse(uriString).buildUpon().appendQueryParameter(CoreConstants.IS_USED_APPLINK_KEY, "true").build().toString();
                    }
                }

                if ((iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) ||
                        isApplink(iUri, appkey)) {
                    Uri newData = Uri.parse(uriString);
                    deeplinkIntent.setData(newData);
                } else {
                    AbxLog.e("Check Adbrix Parameters :: There is no Deeplink information in Activity!", true);
                }
            }
        }
    }

    public static Uri getDeeplinkUriExceptAdbrixParameter(Uri deeplinkUri, DataRegistry dataRegistry) {
        Uri iUri = Uri.parse(Uri.decode(deeplinkUri.toString()));

        if (iUri.toString().indexOf('#') != -1)
            iUri = Uri.parse(iUri.toString().replaceAll("\\#", "%32"));
        String uriString = deeplinkUri.toString();
        uriString = Uri.decode(uriString);

        if (iUri != null && iUri.isHierarchical()) {
            if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) {
                String paramString = CoreConstants.DEEPLINK_ABX_TID + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID);
                if (iUri.getQuery().length() == paramString.length()) {
                    paramString = "\\?" + paramString;
                } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                    paramString = "&" + paramString;
                } else {
                    paramString = paramString + "&";
                }
                if (!CommonUtils.isNullOrEmpty(uriString)) {
                    uriString = uriString.replaceFirst(paramString, "");
                }
            } else {
                AbxLog.d("abx_tid doesn't exist", true);
            }

            if (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) {
                String paramString = CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY + "=" + iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY);
                if (iUri.getQuery().length() == paramString.length()) {
                    paramString = "\\?" + paramString;
                } else if (uriString != null && (uriString.length() - paramString.length()) == uriString.indexOf(paramString)) {
                    paramString = "&" + paramString;
                } else {
                    paramString = paramString + "&";
                }
                if (!CommonUtils.isNullOrEmpty(uriString)) {
                    uriString = uriString.replaceFirst(paramString, "");
                }
            } else {
                AbxLog.d("abx_tracker_id doesn't exist", true);
            }

            String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

            if (isApplink(iUri, appkey)) {
                if (!CommonUtils.isNullOrEmpty(uriString) && iUri.getQueryParameter(CoreConstants.IS_USED_APPLINK_KEY) == null) {
                    uriString = Uri.parse(uriString).buildUpon().appendQueryParameter(CoreConstants.IS_USED_APPLINK_KEY, "true").build().toString();
                }
            }

            if ((iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                    (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) ||
                    isApplink(iUri, appkey)) {
                return Uri.parse(uriString);
            } else {
                AbxLog.e("Check Adbrix Parameters :: There is no Deeplink information in Activity!", true);
                return Uri.parse(uriString);
            }
        }
        return Uri.parse(uriString);
    }

    public static Map<String, Object> getDeeplinkParameterFromUriString(String encodedString) {
        String uriString = Uri.decode(encodedString);
        Uri uri = Uri.parse(uriString);

        if (uri == null || !uri.isHierarchical()) {
            AbxLog.e("ERROR :: Uri is null or parameter doesn't exist in ...", true);
            return new HashMap<>();
        }

        if (uri.toString().contains("#")) {
            uri = Uri.parse(uri.toString().replaceAll("\\#", "%32"));
        }

        Map<String, Object> parameter = new HashMap<>();
        parameter.put(CoreConstants.DEEPLINK_PAYLOAD, uri.toString());

        String abxTid = uri.getQueryParameter("abx_tid");
        String abxTrackerId = uri.getQueryParameter("abx_tracker_id");

        if (!CommonUtils.isNullOrEmpty(abxTid)) {
            parameter.put(CoreConstants.DEEPLINK_ABX_ADKEY, abxTid);
        }
        if (!CommonUtils.isNullOrEmpty(abxTrackerId)) {
            parameter.put(CoreConstants.DEEPLINK_ABX_TRACKER_ID_PARAM_KEY, abxTrackerId);
        }

        return parameter;
    }

    public static boolean isActivityStartedByDeeplink(Activity activity, DataRegistry dataRegistry) {
        if (activity != null && activity.getIntent() != null && activity.getIntent().getData() != null) {
            Uri iUri = activity.getIntent().getData(); // Immutable URI reference
            if (iUri != null && iUri.isHierarchical()) {

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

                return (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) ||
                        isApplink(iUri, appkey);
            }
        }
        return false;
    }

    public static boolean isStartedByDeeplinkIntent(Intent deeplinkIntent, DataRegistry dataRegistry) {
        if (deeplinkIntent != null && deeplinkIntent.getData() != null) {
            Uri iUri = deeplinkIntent.getData(); // Immutable URI reference
            if (iUri != null && iUri.isHierarchical()) {

                String appkey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);

                return (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TID) != null) ||
                        (iUri.getQueryParameter(CoreConstants.DEEPLINK_ABX_TRACKER_ID_QUERY_KEY) != null) ||
                        isApplink(iUri, appkey);
            }
        }
        return false;
    }

    private static boolean isApplink(Uri uri, String appkey) {
        return (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_LIVE_URL_LEGACY)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_STAGE_URL_LEGACY)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_DEV_URL_LEGACY)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_LIVE_URL_TOKYO)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_STAGE_URL_TOKYO)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_DEV_URL_TOKYO)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_LIVE_URL_SEOUL)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_STAGE_URL_SEOUL)) ||
                (uri.toString().toLowerCase(Locale.ENGLISH).startsWith("https://" + appkey.toLowerCase(Locale.ENGLISH) + CoreConstants.DEEPLINK_DEV_URL_SEOUL));
    }

    public static boolean isDevState(Context context) {
        try {
            ApplicationInfo appInfo;
            appInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = appInfo.metaData;
            return bundle.getBoolean("19327da43293b1dce56da3a83829ccaf434bb68ef83ab61a2dca61d3bb15f2be");
        } catch (PackageManager.NameNotFoundException e1) {
            return false;
        } catch (Exception e2) {
            return false;
        }
    }

    public static boolean isStageState(Context context) {
        try {
            ApplicationInfo appInfo;
            appInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = appInfo.metaData;
            return bundle.getBoolean("AdBrixRmStageMode");
        } catch (PackageManager.NameNotFoundException e1) {
            return false;
        } catch (Exception e2) {
            return false;
        }
    }

    public static DRState getDRState(DataRegistry dataRegistry) {

        if (dataRegistry == null) {
            AbxLog.d("DRSTATE :: Cannot get DRState before initialized!", true);
            return DRState.NORMAL;
        }

        Boolean sdkStopped = dataRegistry.getBooleanOrNull(DataRegistryKey.BOOLEAN_IS_SDK_STOPPED);
        if (sdkStopped == null) {
            return DRState.NORMAL;
        }

        boolean stopServerSync = dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC, false);

        int zerothBit = stopServerSync ? 1 : 0;
        int firstBit = sdkStopped ? 1 : 0;

        int result = 2 * firstBit + zerothBit;

        return DRState.fromInteger(result);
    }

    public static void printDRState(DataRegistry dataRegistry) {
        switch (getDRState(dataRegistry)) {
            case INIT_RESTART_SYNCED:
                AbxLog.d("SDK STOPPED STATE :: SDK restarted by user, also  synced!", true);
                break;
            case INIT_RESTART_NOT_SYNCED:
                AbxLog.d("SDK STOPPED STATE :: SDK restarted by user, but not synced!", true);
                break;
            case DELETE_NOT_SYNCED:
                AbxLog.d("SDK STOPPED STATE :: Deleting user data requested, but not synced!", true);
                break;
            case DELETE_SYNCED:
                AbxLog.d("SDK STOPPED STATE :: Deleting user data requested, also synced!", true);
                break;
            default:
                AbxLog.d("SDK STOPPED STATE :: normal state!", true);
        }
    }

    public static boolean isAdbrixDisabled(DataRegistry dataRegistry, Runnable gdprForgetMe) {
        if (dataRegistry == null) {
            return false;
        }

        if (isGdprForgetMe(dataRegistry, gdprForgetMe) || isDeleted(dataRegistry) || isAdbrixPause(dataRegistry)) {
            AbxLog.d("Adbrix Disabled : " + true, true);
            return true;
        } else return false;
    }

    public static boolean isDeleted(DataRegistry dataRegistry) {
        if (dataRegistry == null) {
            return false;
        }

        DRState currentState = CoreUtils.getDRState(dataRegistry);

        return currentState == DRState.DELETE_NOT_SYNCED || currentState == DRState.DELETE_SYNCED;
    }

    public static boolean isGdprForgetMe(DataRegistry dataRegistry, Runnable gdprForgetMe) {
        if (dataRegistry == null) {
            return false;
        }

        boolean isGdprForgetMe = dataRegistry.safeGetLong(DataRegistryKey.LONG_GDPR_FORGETME, 0L) == 1L;

        if (isGdprForgetMe) {
            if (!isGdprForgetMeSync(dataRegistry)) {
                gdprForgetMe.run();
            }
            return true;
        } else return false;
    }

    public static boolean isGdprForgetMeSync(DataRegistry dataRegistry) {
        if (dataRegistry == null) {
            return false;
        }

        return dataRegistry.safeGetLong(DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC, 0L) == 1L;
    }

    public static boolean isAdbrixPause(DataRegistry dataRegistry) {
        if (dataRegistry == null) {
            return false;
        }

        return dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE, 0L) == 1L;
    }

    public static boolean isAdbrixAllStop(DataRegistry dataRegistry) {
        if (dataRegistry == null) {
            return false;
        }

        return dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP, 0L) == 1L;
    }

    public static <E, R> List<R> map(List<? extends E> list, Function<? super E, ? extends R> mapper) {
        List<R> mappedList = new ArrayList<>();

        for (E e : list) {
            mappedList.add(mapper.apply(e));
        }

        return mappedList;
    }

    public static <E, R, EX extends Exception> List<R> mapCatching(List<? extends E> list, ThrowingFunction<? super E, ? extends R, EX> mapper) {
        List<R> mappedList = new ArrayList<>();

        for (E e : list) {
            try {
                mappedList.add(mapper.apply(e));
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }

        return mappedList;
    }

    public static long getStartDateTimeLong(String startDateString) {
        SimpleDateFormat startAndEndDateFormat = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT, Locale.US);
        startAndEndDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        SimpleDateFormat startAndEndDateFormatNoMills = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT_NO_MILLIS, Locale.US);
        startAndEndDateFormatNoMills.setTimeZone(TimeZone.getTimeZone("UTC"));
        Date startDate;
        try {
            startDate = startAndEndDateFormat.parse(startDateString);
        } catch (ParseException e) {
            try {
                startDate = startAndEndDateFormatNoMills.parse(startDateString);
            } catch (ParseException parseException) {
                AbxLog.w("getStartDateTimeLong: parse Error", true);
                return 0;
            }
        }

        if (startDate != null) {
            return startDate.getTime();
        }

        return 0;
    }

    public static long getEndDateTimeLong(String endDateString) {
        if (endDateString == null || endDateString.equals("null"))
            return 0;

        SimpleDateFormat startAndEndDateFormat = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT, Locale.US);
        startAndEndDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        SimpleDateFormat startAndEndDateFormatNoMills = new SimpleDateFormat(CoreConstants.IAM_START_AND_END_DATE_FORMAT_NO_MILLIS, Locale.US);
        startAndEndDateFormatNoMills.setTimeZone(TimeZone.getTimeZone("UTC"));

        Date endDate;
        try {
            endDate = startAndEndDateFormat.parse(endDateString);
        } catch (ParseException e) {
            try {
                endDate = startAndEndDateFormatNoMills.parse(endDateString);
            } catch (ParseException parseException) {
                AbxLog.w("getEndDateTimeLong: parse Error", true);
                return -1;
            }
        }

        if (endDate != null) {
            return endDate.getTime();
        }

        return 0;
    }

    public static List<String> getColumnNameList(SQLiteDatabase db, String table) {
        try (Cursor cursor = db.rawQuery("PRAGMA table_info(" + table + ");", null)) {
            List<String> columns = new ArrayList<>();
            while (cursor != null && cursor.moveToNext()) {
                columns.add(cursor.getString(cursor.getColumnIndex("name")));
            }
            return columns;
        } catch (Exception ignored) {
            return Collections.emptyList();
        }
    }

    public static boolean isColumnExists(SQLiteDatabase db, String table, String columnName) {
        return getColumnNameList(db, table).contains(columnName);
    }
}
