package io.adbrix.sdk.domain.model;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.Map;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

/**
 * 이벤트 데이터
 */
public final class EventModel implements IDataModel {

    public final String lastFirstopenId;
    public final String lastDeeplinkId;
    public final String lastOpenId;
    public final String prevId;
    public final String eventDatetime;
    public final String group;
    public final String eventName;
    @Nullable
    public final Map<String, Object> param;
    public final long sessionInterval;
    public final long sessionLength;
    public final String logId;
    public final String sessionId;
    public final Location location;
    public final String userSnapshotId;
    public final JSONObject userProperties;
    public final JSONObject ci;

    public EventModel(
            String lastFirstopenId,
            String lastDeeplinkId,
            String lastOpenId,
            String prevId,
            String eventDatetime,
            String group,
            String eventName,
            @Nullable
            Map<String, Object> param,
            long sessionInterval,
            long sessionLength,
            String logId,
            String sessionId,
            Location location,
            String userSnapshotId,
            JSONObject userProperties,
            JSONObject ci
    ) {
        this.lastFirstopenId = lastFirstopenId;
        this.lastDeeplinkId = lastDeeplinkId;
        this.lastOpenId = lastOpenId;
        this.prevId = prevId;
        this.eventDatetime = eventDatetime;
        this.group = group;
        this.eventName = eventName;
        this.param = param;
        this.sessionInterval = sessionInterval;
        this.sessionLength = sessionLength;
        this.logId = logId;
        this.sessionId = sessionId;
        this.location = location;
        this.userSnapshotId = userSnapshotId;
        this.userProperties = userProperties;
        this.ci = ci;
    }

    public static EventModel fromJson(JSONObject jsonObject) throws JSONException {

        Location location = null;
        Map<String, Object> param = null;

        try {
            if (jsonObject.get("location") instanceof JSONObject) {
                location = new Location(
                        CommonUtils.convertNullStringToNull(
                                jsonObject.getJSONObject("location").getString("lwid")),
                        jsonObject.getJSONObject("location").getDouble("lat"),
                        jsonObject.getJSONObject("location").getDouble("lng")
                );
            }
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        try {
            if (jsonObject.get("param") instanceof JSONObject) {
                param = CommonUtils.getMapFromJSONObject(jsonObject.getJSONObject("param"));
            }
        } catch (JSONException e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
        }

        return new EventModel(
                CommonUtils.convertNullStringToNull(jsonObject.optString("last_firstopen_id")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("last_deeplink_id")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("last_open_id")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("prev_id")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("event_datetime")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("group")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("event_name")),
                param,
                jsonObject.optLong("session_interval"),
                jsonObject.optLong("session_length"),
                CommonUtils.convertNullStringToNull(jsonObject.optString("log_id")),
                CommonUtils.convertNullStringToNull(jsonObject.optString("session_id")),
                location,
                CommonUtils.convertNullStringToNull(jsonObject.optString("user_snapshot_id")),
                jsonObject.optJSONObject("user_properties"),
                jsonObject.optJSONObject("ci")
        );
    }

    /**
     * "evt": {  //고객이 입력한 이벤트
     * "event_datetime": "20151106094112",    //이벤트의 시간을 yyyyMMddHHmmss 형태의 문자열로 변환
     * "event_name": "open_menu",              //고객이 입력한 이벤트명
     * "location": {                      //optional (위치 추적을 할 수 없다면 null)
     * "lat": 37.7673973149,            //위도
     * "lng": -122.4293583478           //경도
     * },
     * "param":                           //고객이 입력한 파라미터 key:value 페어로 이루어져있다.
     * {
     * "menu_name" : "menu1",
     * "menu_id" : 30
     * },
     * "user_properties": {               //고객이 입력한 프로퍼티 key:value 페어로 이루어져 있다.
     * "birthyear": 1982,
     * "gender": "m",
     * "level": 37,
     * "character_class": "knight",
     * "gold": 300
     * }
     * }
     *
     * @return
     */
    public JSONObject getJson() throws JSONException {
        JSONObject root = new JSONObject();

        root.put("last_firstopen_id", this.lastFirstopenId);
        root.put("last_deeplink_id", CoreUtils.convertNullableToJSONNullable(this.lastDeeplinkId));
        root.put("last_open_id", this.lastOpenId);
        root.put("prev_id", CoreUtils.convertNullableToJSONNullable(this.prevId));
        //"event_datetime": "20151106094112"
        root.put("event_datetime", this.eventDatetime);
        root.put("group", this.group);
        //"event_name": "open_menu"
        root.put("event_name", this.eventName);

            /*
            "param":                           //고객이 입력한 파라미터 key:value 페어로 이루어져있다.
             {
                "menu_name" : "menu1",
                "menu_id" : 30
             }
            */
        if (this.param != null) {
            JSONObject paramJson = new JSONObject();

            for (String key : this.param.keySet()) {
                paramJson.put(key, this.param.get(key));
            }

            root.put("param", paramJson);
        } else {
            root.put("param", JSONObject.NULL);
        }

        root.put("session_interval", this.sessionInterval);
        root.put("session_length", this.sessionLength);
        root.put("log_id", this.logId);
        root.put("session_id", CoreUtils.convertNullableToJSONNullable(this.sessionId));

            /*
            "location": {                      //optional (위치 추적을 할 수 없다면 null)
                "lat": 37.7673973149,            //위도
                "lng": -122.4293583478           //경도
            }
            */
        if (this.location != null) {
            JSONObject locationJson = new JSONObject();

            if (this.location.lwid == null) {
                root.put("location", JSONObject.NULL);
            } else {
                locationJson.put("lwid", this.location.lwid);
                locationJson.put("lat", this.location.lat);
                locationJson.put("lng", this.location.lng);

                root.put("location", locationJson);
            }
        } else {
            root.put("location", JSONObject.NULL);
        }

        root.put("user_snapshot_id", CoreUtils.convertNullableToJSONNullable(this.userSnapshotId));

            /*
            "user_properties": {               //고객이 입력한 프로퍼티 key:value 페어로 이루어져 있다.
                "birthyear": 1982,
                "gender": "m",
                "level": 37,
                "character_class": "knight",
                "gold": 300
            }
            */

        if (userProperties != null)
            root.put("user_properties", userProperties);
        else
            root.put("user_properties", new JSONObject());

        if (ci != null)
            root.put("ci", ci);
        else
            root.put("ci", new JSONObject());

        return root;
    }

    public static class Location {
        public final String lwid;
        public final Double lat;
        public final Double lng;

        public Location(
                String lwid,
                Double lat,
                Double lng) {
            this.lwid = lwid;
            this.lat = lat;
            this.lng = lng;
        }
    }
}
