package io.adbrix.sdk.data.net;

import android.content.Context;
import android.net.TrafficStats;
import android.os.Build;

import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;
import com.google.android.gms.security.ProviderInstaller;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.IApiModel;
import io.adbrix.sdk.utils.Base64;
import io.adbrix.sdk.utils.CoreUtils;

public class ApiConnection {

    private DataRegistry dataRegistry;
    private Context androidContext;
    private int responseCode;
    private StringBuilder responseStringBuilder;
    private String dynamicHashKey;
    private HostnameVerifier trustedVerifier;
    private SSLSocketFactory trustedFactory;
    private JSONObject body;
    private String urlString;
    private RequestMethod requestMethod;
    private Map<String, String> header = new HashMap<>();

    public ApiConnection(DataRegistry dataRegistry, Context androidContext) {
        this.dataRegistry = dataRegistry;
        this.androidContext = androidContext;
    }

    public int getResponseCode() {
        return responseCode;
    }

    public String getResponseString() {
        return responseStringBuilder.toString();
    }

    public ApiConnection header(Map<String, String> header) {
        this.header = header;
        return this;
    }

    public ApiConnection get(String urlString) {
        this.urlString = urlString;
        this.requestMethod = RequestMethod.GET;
        return this;
    }

    public ApiConnection get(IApiModel apiModel) {
        this.body = TryOptional.of(apiModel::getJson).orElseGet(JSONObject::new);
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.GET;
        return this;
    }

    public ApiConnection post(IApiModel apiModel) {
        this.body = TryOptional.of(apiModel::getJson).orElseGet(JSONObject::new);
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.POST;
        return this;
    }

    public ApiConnection delete(IApiModel apiModel) {
        this.body = TryOptional.of(apiModel::getJson).orElseGet(JSONObject::new);
        this.urlString = apiModel.getUrlString();
        this.requestMethod = RequestMethod.DELETE;
        return this;
    }

    public String request() throws Exception {
        validate();

        TrafficStats.setThreadStatsTag(Math.abs((int) Thread.currentThread().getId()));

        HttpURLConnection connection = null;
        try {
            connection = this.createConnection(urlString);

            setExtraHeader(connection);

            if (requestMethod == RequestMethod.POST || requestMethod == RequestMethod.DELETE) {
                setHeader(connection);
                writeBody(connection);
            }

            response(connection);

        } catch (IOException e) {
            AbxLog.w(Arrays.toString(e.getStackTrace()), true);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }

        TrafficStats.clearThreadStatsTag();

        if (responseStringBuilder != null) {
            return responseStringBuilder.toString();
        } else return null;
    }


    private void validate() throws Exception {
        StringBuilder errorMessageBuilder = new StringBuilder();
        if (urlString == null) {
            errorMessageBuilder.append("urlString must not be null.\n");
        }
        if (requestMethod == null) {
            errorMessageBuilder.append("requestMethod must not be null.\n");
        }
        if (requestMethod == RequestMethod.POST && body == null) {
            errorMessageBuilder.append("body must not be null with POST method.\n");
        }
        if (header == null)
            header = new HashMap<>();

        if (errorMessageBuilder.length() > 0)
            throw new Exception(errorMessageBuilder.toString());
    }

    private void setExtraHeader(HttpURLConnection connection) {
        for (Map.Entry<String, String> entry : header.entrySet()) {
            connection.setRequestProperty(entry.getKey(), entry.getValue());
        }
    }

    private void setHeader(HttpURLConnection connection) {
        String userAgent = String.format("IGAWorks/%s (Android %s; U; %s Build/%s)",
                SdkVersion.SDK_VERSION,
                dataRegistry.safeGetString(DataRegistryKey.STRING_OS, null),
                dataRegistry.safeGetString(DataRegistryKey.STRING_MODEL, null),
                dataRegistry.safeGetString(DataRegistryKey.STRING_BUILD_ID, null)
        );

        connection.setRequestProperty("User-Agent", userAgent);

        if (urlString.contains("opengdpr_requests") || urlString.contains("inappmessage") || urlString.toLowerCase(Locale.ENGLISH).contains("actionhistory")) {
            connection.setRequestProperty("Content-Type", "application/json; charset=utf-8");
        } else {
            connection.setRequestProperty("Content-Type", "application/octet-stream; charset=utf-8");
        }

        connection.setRequestProperty("abx-auth-time", getAbxAuthTime(body));
        connection.setRequestProperty("abx-auth-cs", getAbxAuthCs(body));
        connection.setDoInput(true);
        connection.setDoOutput(true);
        connection.setRequestProperty("Accept-Charset", "UTF-8");
    }

    private void writeBody(HttpURLConnection connection) throws IOException {
        BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(connection.getOutputStream(),
                "UTF-8"));

        if (urlString.contains("opengdpr_requests") || urlString.contains("inappmessage") || urlString.toLowerCase(Locale.ENGLISH).contains("actionhistory")) {
            bufferedWriter.write(body.toString());
        } else {
            bufferedWriter.write(Base64.encode(body.toString()));
        }
        bufferedWriter.flush();
        bufferedWriter.close();
    }

    private void response(HttpURLConnection connection) throws IOException {
        InputStream inputStream;
        BufferedReader bufferedReader;
        responseStringBuilder = new StringBuilder();
        responseCode = connection.getResponseCode();

        if (200 <= responseCode && responseCode <= 299) {
            inputStream = connection.getInputStream();
        } else {
            inputStream = connection.getErrorStream();
        }

        bufferedReader = new BufferedReader(new InputStreamReader(inputStream));

        String currentLine;

        while ((currentLine = bufferedReader.readLine()) != null)
            responseStringBuilder.append(currentLine);

        bufferedReader.close();
    }

    public boolean isHttpOK() {
        return HttpsURLConnection.HTTP_OK <= responseCode && responseCode < 300;
    }

    public boolean isWrongAppkey() {
        return responseCode == 404 || (responseCode > 500 && responseCode < 600);
    }

    public boolean isInvalidAppkey() {
        return responseCode == 400;
    }

    private HttpURLConnection createConnection(String urlString) throws IOException {
        URL url = new URL(urlString);

        final HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setReadTimeout(CoreConstants.NETWORK_TIMEOUT);
        connection.setConnectTimeout(CoreConstants.NETWORK_TIMEOUT);

        if (urlString.startsWith("https")) {
            configureHttpsConnection((HttpsURLConnection) connection);
        }

        connection.setRequestMethod(requestMethod.getMethodString());
        connection.setInstanceFollowRedirects(false);

        return connection;
    }

    private void configureHttpsConnection(HttpsURLConnection connection) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            try {
                SSLContext.getInstance("TLSv1.2");
            } catch (NoSuchAlgorithmException e) {
                AbxLog.w(Arrays.toString(e.getStackTrace()), true);
            }

            try {
                ProviderInstaller.installIfNeeded(androidContext.getApplicationContext());
            } catch (GooglePlayServicesRepairableException | GooglePlayServicesNotAvailableException e) {
                AbxLog.w(Arrays.toString(e.getStackTrace()), true);
            }
        }

        /*
        2020-09-01
        리마스터는 모두 https를 사용하고 있으므로 우회하는 코드를 사용하 않는다.
        google xtrust509 issue report를 해결하기 위한 주석처리.
         */
//        connection.setHostnameVerifier(getTrustedVerifier());
//        connection.setSSLSocketFactory(getTrustedFactory());
    }

    /**
     * Configure HTTPS connection to trust all hosts using a custom
     * {@link HostnameVerifier} that always returns <code>true</code> for each
     * host verified
     * <p>
     * This method does nothing if the current request is not a HTTPS request
     *
     * @return this request
     */
//    private HostnameVerifier getTrustedVerifier() {
//        if (trustedVerifier == null) {
//            trustedVerifier = new HostnameVerifier() {
//                @Override
//                public boolean verify(String hostname, SSLSession session) {
//                    return true;
//                }
//            };
//        }
//
//        return trustedVerifier;
//    }

    /**
     * Configure HTTPS connection to trust all certificates
     * <p>
     * This method does nothing if the current request is not a HTTPS request
     *
     * @return this request
     * @throws 'HttpRequestException
     */
//    private SSLSocketFactory getTrustedFactory() {
//        if (trustedFactory == null) {
//            final TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager() {
//
//                public X509Certificate[] getAcceptedIssuers() {
//                    return new X509Certificate[0];
//                }
//
//                public void checkClientTrusted(X509Certificate[] chain, String authType) {
//                    // Intentionally left blank
//                }
//
//                public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
//                    try {
//                        chain[0].checkValidity();
//                    } catch (Exception e) {
//                        throw new CertificateException("Certificate not valid or trusted.");
//                    }
//                }
//            } };
//
//            try {
//                SSLContext context = SSLContext.getInstance("TLS");
//                context.init(null, trustAllCerts, new SecureRandom());
//                trustedFactory = context.getSocketFactory();
//            } catch (GeneralSecurityException e) {
//                AbxLog.w(Arrays.toString(e.getStackTrace()), true);
//            }
//        }
//
//        return trustedFactory;
//    }

    private String getAbxAuthTime(JSONObject jsonObject) {
        Date currentTime = null;
        SimpleDateFormat sdfKST = CoreUtils.createDateFormat(CoreConstants.DB_DATE_FORMAT);

        try {
            if (jsonObject.has("common")) {
                JSONObject common = jsonObject.getJSONObject("common");
                if (common.has("request_datetime")) {
                    Object requestDatetime = common.get("request_datetime");
                    if (requestDatetime instanceof String) {
                        currentTime = sdfKST.parse((String) requestDatetime);
                    }
                }
            }
        } catch (JSONException | ParseException e) {
            AbxLog.w(Arrays.toString(e.getStackTrace()), true);
        } finally {
            if (currentTime == null) {
                currentTime = new Date();
            }
        }

        setDynamicHashKey(currentTime);
        return sdfKST.format(currentTime);
    }

    private void setDynamicHashKey(Date currentTime) {
        DateFormat h = CoreUtils.createDateFormat("HH");
        String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, "");
        char[] epK = {
                (char) 99, (char) 50, (char) 100, (char) 108,
                (char) 97, (char) 51, (char) 74, (char) 122,
                (char) 98, (char) 88, (char) 82, (char) 105,
                (char) 89, (char) 81, (char) 61, (char) 61
        };
        int index = (Integer.parseInt(h.format(currentTime)) * 49 + 6909 * 6) % 32;
        String fullHashKey = secretKey + Base64.decode(String.valueOf(epK));
        dynamicHashKey = fullHashKey.substring(index) + fullHashKey.substring(0, index);
    }

    private String getAbxAuthCs(JSONObject jsonObject) {
        // first encrypt by hashkey, second encrypt by dsk
        return CoreUtils.hmacSha256(dynamicHashKey, getDynamicHashKeyMD5(Base64.encode(jsonObject.toString())));
    }

    private String getDynamicHashKeyMD5(String string) {
        try {
            // Create MD5 Hash
            MessageDigest digest = java.security.MessageDigest.getInstance("MD5");
            digest.update(string.getBytes());
            byte[] messageDigest = digest.digest();

            // Create Hex String
            StringBuffer hexString = new StringBuffer();
            for (int i = 0; i < messageDigest.length; i++) {
                hexString.append(String.format("%02X", 0xFF & messageDigest[i]));
            }

            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            AbxLog.w(Arrays.toString(e.getStackTrace()), true);
        }
        return string;
    }

    private enum RequestMethod {
        GET("GET"),
        POST("POST"),
        DELETE("DELETE"),
        NONE("");

        private final String methodString;

        RequestMethod(String methodString) {
            this.methodString = methodString;
        }

        public String getMethodString() {
            return methodString;
        }
    }
}
