package io.adbrix.sdk.data.inAppMessage;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BlendMode;
import android.graphics.BlendModeColorFilter;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.os.CountDownTimer;
import android.view.Display;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.configuration.DefaultABXContextController;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.IAMConstants;
import io.adbrix.sdk.domain.model.DfnGrowthEventParam;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class FullScreenMessageViewFactory implements IInAppMessageViewFactory {
    CountDownTimer imageCountDownTimer;
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private InAppMessageDAO inAppMessageDAO;

    public FullScreenMessageViewFactory(InAppMessageDAO inAppMessageDAO) {
        add(DefaultABXContextController.getInstance());
        this.inAppMessageDAO = inAppMessageDAO;
    }

    @Override
    public void createInAppMessage(Activity activity, DfnInAppMessage inAppMessage, IObserver<AbxDialog> abxDialogIObserver) {
        LayoutInflater inflater = activity.getLayoutInflater();
        View view;

        if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.IMAGE.getValue()))
            view = inflater.inflate(R.layout.layout_full_message_image, null);
        else if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue()))
            view = inflater.inflate(R.layout.layout_full_message_scrollable_image, null);
        else
            view = inflater.inflate(R.layout.layout_full_message_image_with_text, null);

        activity.runOnUiThread(() -> {
            if (ABXBooleanState.getInstance().isInAppMessagePresented()) {
                AbxLog.d("InAppMessage already shown", true);
                return;
            }

            try {
                AbxDialog dialog = new AbxDialog(activity, android.R.style.Theme_Light);

                dialog.requestWindowFeature(Window.FEATURE_NO_TITLE);
                dialog.setContentView(view);

                dialog.setOnCancelListener(v -> {
                    setDialogTerminateState(abxDialogIObserver);
                });

                dialog.setOnDismissListener(v -> {
                    setDialogTerminateState(abxDialogIObserver);
                });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;

                setImage(view, isPortrait, inAppMessage, dialog, activity);

                switch (inAppMessage.getNumberOfButtons()) {
                    case 0:
                        setNoButton(view);
                        break;
                    case 1:
                        setSingleButton(view, dialog, inAppMessage, activity);
                        break;
                    case 2:
                        setDualButton(view, dialog, inAppMessage, activity);
                        break;
                    default:
                        setNoButton(view);
                        AbxLog.e("createInAppMessage: Invalid button number!", true);
                }

                if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue()))
                    setText(view, inAppMessage);

                setCancelButton(view, dialog, inAppMessage, abxDialogIObserver);

                open(dialog, inAppMessage, isPortrait);

                abxDialogIObserver.update(dialog);
            } catch (Exception exception) {
                AbxLog.e("createInAppMessage: " + Arrays.toString(exception.getStackTrace()) + "\n" + exception, true);
            }
        });
    }

    private void open(AbxDialog dialog, DfnInAppMessage inAppMessage, boolean isPortrait) {
        dialog.show();

        if (!dialog.isShowing()) {
            return;
        }

        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                .mapCatching(DfnGrowthEventParam::getJson)
                .map(CommonUtils::getMapFromJSONObject)
                .orElseGet(HashMap::new);

        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
        }
        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.IMPRESSION.getValue());

        if (isPortrait) {
            if (inAppMessage.getPortraitImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                }
            }
        } else {
            if (inAppMessage.getLandScapeImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                }
            }
        }

        notifyObserver(new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_IN_APP_MESSAGE,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0
        ));
    }

    private void setCancelButton(View view, AbxDialog dialog, DfnInAppMessage inAppMessage, IObserver<AbxDialog> abxDialogIObserver) {
        ImageButton cancelButton = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_close_button);
        cancelButton.setOnClickListener(button -> {
            if (dialog != null)
                dialog.cancel();
            setDialogTerminateState(abxDialogIObserver);
        });

        int closedButtonColor = Color.parseColor(inAppMessage.getDefaultCloseButtonColor());
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButton.setColorFilter(closedButtonColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButton.setColorFilter(new BlendModeColorFilter(closedButtonColor, BlendMode.SRC_IN));
        }

        int closedButtonBgColor = Color.parseColor(inAppMessage.getDefaultCloseButtonBgColor());
        Drawable cancelButtonDrawable = view.getResources().getDrawable(R.drawable.ic_com_adbrix_inappmessage_cancel_button_border);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButtonDrawable.setColorFilter(closedButtonBgColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButtonDrawable.setColorFilter(new BlendModeColorFilter(closedButtonBgColor, BlendMode.SRC_IN));
        }

        cancelButton.setBackgroundDrawable(cancelButtonDrawable);
    }

    private void setImage(View view, boolean isPortrait, DfnInAppMessage inAppMessage, AbxDialog dialog, Activity activity) {
        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_imageview);
        ProgressBar progressBar = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_progressbar);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_progressbar_layout);
        ImageView defaultImageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_default_imageview);
        LinearLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_default_imageview_layout);

        Bitmap bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();

        if (bitmapImage == null) {
            progressBar.setVisibility(View.VISIBLE);
            progressBarLayout.setVisibility(View.VISIBLE);
            imageView.setVisibility(GONE);
            defaultImageViewLayout.setVisibility(GONE);

            imageCountDownTimer = new CountDownTimer(9_000, 3_000) {
                @Override
                public void onTick(long millisUntilFinished) {
                    if (dialog != null && dialog.isShowing()) {
                        inAppMessageDAO.imageDownloadExecutorService.submit(() -> {
                            if (imageCountDownTimer == null)
                                return;

                            HashMap<String, Object> imageHashMap = inAppMessageDAO.getAndSaveImageInfo(inAppMessage.getCampaignId(), isPortrait ? IAMConstants.PORTRAIT : IAMConstants.LANDSCAPE, true);

                            if (imageHashMap != null) {
                                if (isPortrait) {
                                    inAppMessage.setPortraitImage((Bitmap) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()));
                                    inAppMessage.setPortraitImageURL((String) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()));
                                    inAppMessage.setIsPortraitImageDownloaded((Boolean) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()));
                                } else {
                                    inAppMessage.setLandscapeImage((Bitmap) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()));
                                    inAppMessage.setLandscapeImageURL((String) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()));
                                    inAppMessage.setIsLandscapeImageDownloaded((Boolean) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()));
                                }
                                inAppMessage.setImageClickAction((String) imageHashMap.get(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue()));

                                activity.runOnUiThread(() -> setImage(view, isPortrait, inAppMessage, dialog, activity));

                                if (imageCountDownTimer != null) {
                                    imageCountDownTimer.cancel();
                                    imageCountDownTimer = null;
                                }
                            }
                        });
                    }
                }

                @Override
                public void onFinish() {
                    if (dialog != null && dialog.isShowing()) {
                        activity.runOnUiThread(() -> {
                            defaultImageViewLayout.setVisibility(VISIBLE);
                            progressBarLayout.setVisibility(GONE);
                            progressBar.setVisibility(GONE);
                            imageView.setVisibility(GONE);
                        });

                        defaultImageView.setOnClickListener(v -> activity.runOnUiThread(() ->
                                setImage(view, isPortrait, inAppMessage, dialog, activity)
                        ));
                    }
                    this.cancel();
                }
            }.start();

            return;
        } else {
            if (imageCountDownTimer != null) {
                imageCountDownTimer.cancel();
                imageCountDownTimer = null;
            }
            progressBar.setVisibility(GONE);
            progressBarLayout.setVisibility(GONE);
            imageView.setVisibility(View.VISIBLE);
            defaultImageViewLayout.setVisibility(GONE);
        }

        setImageScaleType(activity, view, inAppMessage, imageView, bitmapImage);

        if (inAppMessage.getImageClickActionId() != null) {
            imageView.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                    inAppMessage.getCampaignId(),
                    inAppMessage.getImageClickActionId(),
                    inAppMessage.getImageClickActionType(),
                    inAppMessage.getImageClickActionArg(),
                    dialog,
                    () -> {
                        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                                .mapCatching(DfnGrowthEventParam::getJson)
                                .map(CommonUtils::getMapFromJSONObject)
                                .orElseGet(HashMap::new);

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }

                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getImageClickActionId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());

                        notifyObserver(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));
                        if (
                                IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                        ) {
                            if (imageCountDownTimer != null) {
                                imageCountDownTimer.cancel();
                                imageCountDownTimer = null;
                            }
                        }
                    },
                    inAppMessageDAO
            ));
        }
    }

    private void setImageScaleType(Activity activity, View view, DfnInAppMessage inAppMessage, ImageView imageView, Bitmap bitmapImage) {
        if (!IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue().equals(inAppMessage.getLayout())) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        WindowManager windowManager = (WindowManager) activity.getSystemService(Context.WINDOW_SERVICE);
        if (windowManager == null) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        double deviceRatio = (double) point.y / point.x;
        double imageRatio = (double) bitmapImage.getHeight() / bitmapImage.getWidth();

        if (imageRatio > deviceRatio) {
            imageView.setImageBitmap(bitmapImage);
            setImageViewGravity(imageView, Gravity.TOP);
            return;
        }

        if (IAMEnums.ScrollableImageOptionAlign.CENTER.getValue().equals(inAppMessage.getScrollableImageOptionAlign())) {
            setImageViewGravity(imageView, Gravity.CENTER);
        } else {
            setImageViewGravity(imageView, Gravity.TOP);
        }

        imageView.setImageBitmap(bitmapImage);
        ScrollView scrollView = (ScrollView) view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_scrollable_image_scrollview);
        scrollView.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setImageViewGravity(ImageView imageView, int gravity) {
        FrameLayout.LayoutParams layoutParams = (FrameLayout.LayoutParams) imageView.getLayoutParams();
        layoutParams.gravity = gravity;
        imageView.setLayoutParams(layoutParams);
    }

    private void setImageScaleType2(Activity activity, View view, DfnInAppMessage inAppMessage, ImageView imageView, Bitmap bitmapImage) {
        WindowManager windowManager = (WindowManager) activity.getSystemService(Context.WINDOW_SERVICE);
        if (windowManager == null) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        double deviceRatio = (double) point.y / point.x;
        double imageRatio = (double) bitmapImage.getHeight() / bitmapImage.getWidth();

        if (imageRatio > deviceRatio) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.FIT_START);
            return;
        }

        if (!inAppMessage.getLayout().equals(IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue())) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

//        ImageView notScrollableImageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_not_scrollable_imageview);
//        ScrollView scrollView = (ScrollView) view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_scrollable_image_scrollview);
//
//        notScrollableImageView.setImageBitmap(bitmapImage);
//        notScrollableImageView.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
//
//        if (IAMEnums.ScrollableImageOptionAlign.CENTER.getValue().equals(inAppMessage.getScrollableImageOptionAlign())) {
//            notScrollableImageView.setScaleType(ImageView.ScaleType.FIT_CENTER);
//        } else {
//            notScrollableImageView.setScaleType(ImageView.ScaleType.FIT_START);
//        }
//
//        scrollView.setVisibility(GONE);
//        notScrollableImageView.setVisibility(VISIBLE);
    }

    private void setText(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout textLayout = view.findViewById(R.id.com_adbrix_inappmessage_text_layout);

        TextView titleText = view.findViewById(R.id.com_adbrix_inappmessage_title);
        titleText.setText(inAppMessage.getTitleText());
        titleText.setTextColor(Color.parseColor(inAppMessage.getTitleTextColor()));

        TextView bodyText = view.findViewById(R.id.com_adbrix_inappmessage_body);
        bodyText.setText(inAppMessage.getBodyText());
        bodyText.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (IAMEnums.TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.CENTER);
            bodyText.setGravity(Gravity.CENTER);
        } else if (IAMEnums.TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.RIGHT);
            bodyText.setGravity(Gravity.RIGHT);
        } else if (IAMEnums.TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.LEFT);
            bodyText.setGravity(Gravity.LEFT);
        }

        textLayout.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setNoButton(View view) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(GONE);
    }

    public void setSingleButton(View view, AbxDialog dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout doubleButtonsLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        doubleButtonsLayout.setVisibility(View.GONE);
        singleButtonLayout.setVisibility(View.VISIBLE);

        Button singleButton = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_single);

        setButton(dialog, singleButton, inAppMessage, 0, activity);
    }

    public void setDualButton(View view, AbxDialog dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout doubleButtonsLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        doubleButtonsLayout.setVisibility(View.VISIBLE);
        singleButtonLayout.setVisibility(View.GONE);

        Button buttonFirst = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_dual_first);
        Button buttonSecond = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_dual_second);

        setButton(dialog, buttonFirst, inAppMessage, 0, activity);
        setButton(dialog, buttonSecond, inAppMessage, 1, activity);
    }

    public void setButton(AbxDialog dialog, Button button, DfnInAppMessage inAppMessage, int index, Activity activity) {
        //text
        button.setText(inAppMessage.getButtonText(index));
        button.setTextColor(Color.parseColor(inAppMessage.getButtonTextColor(index)));

        //background and border
        GradientDrawable singleButtonBackground = new GradientDrawable();
        singleButtonBackground.setShape(GradientDrawable.RECTANGLE);
        singleButtonBackground.setStroke(5, Color.parseColor(inAppMessage.getButtonBoarderColor(index)));
        singleButtonBackground.setColor(Color.parseColor(inAppMessage.getButtonBackgroundColor(index)));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            button.setBackground(singleButtonBackground);
        } else {
            button.setBackgroundDrawable(singleButtonBackground);
        }

        button.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                inAppMessage.getCampaignId(),
                inAppMessage.getButtonActionID(index),
                inAppMessage.getButtonActionType(index),
                inAppMessage.getButtonActionArgument(index),
                dialog,
                () -> {
                    Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                            .mapCatching(DfnGrowthEventParam::getJson)
                            .map(CommonUtils::getMapFromJSONObject)
                            .orElseGet(HashMap::new);

                    if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                    }
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getButtonActionID(index));

                    notifyObserver(new LogEventParameter(
                            CoreConstants.GROUP_ABX,
                            CoreConstants.EVENT_IN_APP_MESSAGE,
                            CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                            0,
                            0
                    ));
                    if (
                            IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                    ) {
                        if (imageCountDownTimer != null) {
                            imageCountDownTimer.cancel();
                            imageCountDownTimer = null;
                        }
                    }
                },
                inAppMessageDAO
        ));
    }

    private void setDialogTerminateState(IObserver<AbxDialog> abxDialogIObserver) {
        ABXBooleanState.getInstance().isInAppMessagePresented.getAndSet(false);
        if (imageCountDownTimer != null) {
            imageCountDownTimer.cancel();
            imageCountDownTimer = null;
        }
        if (abxDialogIObserver != null)
            abxDialogIObserver.update(null);
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }
}
