package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import org.json.JSONObject;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.UserPropertyManager;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.interactor.FlushAllNowUseCase;
import io.adbrix.sdk.domain.interactor.GdprForgetMeUseCase;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.DfnInAppMessageFetchMode;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CoreUtils;

public class DefaultABXContextController implements IABXContextController, IObserver<LogEventParameter> {
    private IABXComponentsFactory componentsFactory;
    private ILogger logger;
    private IABXContext abxContext;
    private DataRegistry dataRegistry;
    private ExecutorService executorService;
    private boolean isInitSubmitted = false;
    private boolean isOnResumeSubmitted = false;
    private Queue<MessageInvokeRunnable> onResumeQueue;
    private Queue<MessageInvokeRunnable> etcQueue;
    private Repository repository;
    private boolean isInitRestartProcessingNow = false;
    private boolean isDeleteProcessingNow = false;

    private DefaultABXContextController() {
    }

    public static DefaultABXContextController getInstance() {
        return Singleton.INSTANCE;
    }

    public void startController(IABXComponentsFactory componentsFactory) {
        this.componentsFactory = componentsFactory;

        this.logger = this.componentsFactory.createOrGetLogger();
        executorService = Executors.newSingleThreadExecutor();
        onResumeQueue = new ConcurrentLinkedQueue<>();
        etcQueue = new ConcurrentLinkedQueue<>();

        try {
            this.dataRegistry = this.componentsFactory.createOrGetDataRegistry();
            registerDataUnitEventListener();

            this.repository = componentsFactory.createOrGetRepository();
            this.abxContext = new InitializingABXContext(this.componentsFactory, repository);
            this.componentsFactory.setABXContext(abxContext);

        } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) { //이런일이 발생해서는 안된다.
            this.logger.error(e.toString());
            disableSDK("Cannot start controller.");
        }
    }

    private void registerDataUnitEventListener() {
        this.dataRegistry.setListenOnChange(true);

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE, (oldValue, newValue) -> {
            try {
                if (newValue.getLong() == 1) {
                    new FlushAllNowUseCase(repository).execute();
                    disableSDK("Adbrix Pause");
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP, (oldValue, newValue) -> {
            try {
                if (newValue.getLong() == 1) {
                    disableSDK("Adbrix All Stop");
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        });

        List<DataRegistryKey> attributionIdKeys = Arrays.asList(
                DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                DataRegistryKey.STRING_LAST_OPEN_ID
        );

        this.dataRegistry.registerDataUnitEventListener(attributionIdKeys, (oldValue, newValue) -> {
            dataRegistry.saveRegistry();
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.STRING_SDK_VERSION, (oldValue, newValue) -> {
            try {
                if (oldValue == null) {
                    return;
                }

                if (SdkVersion.compare(oldValue.getString(), "2.3.0.0") >= 0) {
                    return;
                }

                AbxLog.d("SDKVersion changed, oldValue : " + oldValue.getString() + ", newValue : " + newValue.getString(), true);

                dataRegistry.putDataRegistry(new DataUnit(
                        DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                        0,
                        5,
                        this.getClass().getName(),
                        true
                ));

                dataRegistry.putDataRegistry(new DataUnit(
                        DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                        null,
                        5,
                        this.getClass().getName(),
                        true
                ));
            } catch (Exception e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.INT_IN_APP_MESSAGE_FETCH_MODE_VALUE, (oldValue, newValue) -> {
            try {
                if (oldValue == null)
                    return;

                DfnInAppMessageFetchMode oldMode = DfnInAppMessageFetchMode.fromInteger(oldValue.getInt());
                DfnInAppMessageFetchMode newMode = DfnInAppMessageFetchMode.fromInteger(newValue.getInt());

                if (oldMode != newMode) {
                    deleteAllInAppMessageDBContents();
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        });
    }

    public void stopController() {
        if (executorService != null)
            executorService.shutdown();

        if (repository != null)
            repository.shutDownInAppMessageExecutor();
    }

    @Override
    public IABXComponentsFactory getComponentsFactory() {
        return this.componentsFactory;
    }

    @Override
    public void changeABXContext(IABXContext newContext) {
        if (abxContext == null || componentsFactory == null) {
            AbxLog.d("controller.changeABXContext is called before controller.startcontroller", true);
            return;
        }

        if (abxContext.getClass() == newContext.getClass()) {
            AbxLog.d("Same ABXContext! : " + abxContext.getClass().getName() + ", Cannot change ABXContext!", true);
            return;
        }
        abxContext = newContext;
        componentsFactory.setABXContext(newContext);
    }

    public void initialize(Context context, String appkey, String secretkey) {
        submit(MessageType.INITIALIZE, context, appkey, secretkey);
    }

    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY, userPropertyCommand);
    }

    public void saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY_WITHOUT_EVENT, userPropertyCommand);
    }

    public void logEvent(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }

    public void logSameEventWithPaging(String eventName, List<JSONObject> eventParamList) {
        submit(MessageType.LOG_SAME_EVENT_WITH_PAGING, eventName, eventParamList);
    }

    public void onResume(Activity activity) {
        ABXBooleanState.getInstance().inForeground.getAndSet(true);
        submit(MessageType.ON_RESUME, activity);
        if (repository != null) {
            repository.updateCurrentActivity(activity);
        }
    }

    public void onPause() {
        ABXBooleanState.getInstance().inForeground.getAndSet(false);
        submit(MessageType.ON_PAUSE);
    }

    public void onDestroy(Activity activity) {
        if (repository != null) {
            repository.dismissInAppMessageDialog(activity);
            repository.deleteCurrentActivity(activity);
        }
    }

    public void deeplink(Activity deeplinkActivity) {
        submit(MessageType.DEEPLINK, deeplinkActivity);
    }

    public void deeplinkWithIntent(Intent deeplinkIntent) {
        submit(MessageType.DEEPLINK_WITH_INTENT, deeplinkIntent);
    }

    public void gdprForgetMe() {
        submit(MessageType.GDPR_FORGET_ME);
    }

    public void putDataRegistry(DataUnit dataUnit) {
        submit(MessageType.PUT_DATA_REGISTRY, dataUnit);
    }

    public void runInBackGroundInOrder(Runnable runnable) {
        submit(MessageType.RUN_IN_BACKGROUND_IN_ORDER, runnable);
    }

    public void runInBackGroundWithoutOrder(Runnable runnable) {
        submit(MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER, runnable);
    }

    public void deleteUserDataAndStopSDK(String userId, Runnable onSuccess, Runnable onFail) {
        submit(MessageType.DELETE_USER_DATA_AND_STOP_SDK, userId, onSuccess, onFail);
    }

    public void restartSDK(String userId, Runnable onSuccess, Runnable onFail) {
        submit(MessageType.RESTART_SDK, userId, onSuccess, onFail);
    }

    public DataRegistry getDataRegistry() throws NullPointerException {
        if (dataRegistry == null) {
            throw new NullPointerException();
        }

        return dataRegistry;
    }

    public UserPropertyModel getCurrentUserPropertyModel() {
        try {
            if (componentsFactory == null) {
                AbxLog.d("controller.getCurrentUserPropertyModel is called before controller.startcontroller", true);
                return new UserPropertyModel(null, new HashMap<>());
            }

            UserPropertyManager userPropertyManager = componentsFactory.createOrGetUserPropertyManager();
            return userPropertyManager.getCurrentUserPropertyModel();
        } catch (Exception e) {
            AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            return new UserPropertyModel(null, new HashMap<>());
        }
    }

    public void triggerInAppMessage(String eventGroup, String eventName) {
        if (repository != null)
            repository.openInAppMessage(eventGroup + ":" + eventName);
    }

    public void fetchActionHistoryFromServer(String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.FETCH_ACTION_HISTORY_FROM_SERVER, token, idType, actionType, completion);
    }

    public void insertPushData(String pushDataString) {
        submit(MessageType.INSERT_PUSH_DATA, pushDataString);
    }

    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.GET_ACTION_HISTORY, skip, limit, actionType, completion);
    }

    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.GET_ALL_ACTION_HISTORY, actionType, completion);
    }

    public void deleteActionHistory(
            @Nullable String token,
            String historyId,
            long timestamp,
            Completion<Result<Empty>> completion
    ) {
       submit(MessageType.DELETE_ACTION_HISTORY, token, historyId, timestamp, completion);
    }

    public void deleteAllActionHistory(@Nullable String token, ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        submit(MessageType.DELETE_ALL_ACTION_HISTORY, token, idType, completion);
    }

    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        submit(MessageType.CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB, completion);
    }

    public void clearAllActionHistoryInLocalDB() {
        submit(MessageType.CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB);
    }

    public void fetchInAppMessage(Completion<Result<Empty>> completion) {
        submit(MessageType.FETCH_IN_APP_MESSAGE, completion);
    }

    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        submit(MessageType.GET_ALL_IN_APP_MESSAGE, completion);
    }

    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        abxContext.openInAppMessage(campaignId, completion);
    }

    public void deleteAllInAppMessageDBContents() {
        submit(MessageType.DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS);
    }

    private void submit(MessageType messageType, Object... args) {
        if (executorService == null || etcQueue == null || onResumeQueue == null) {
            AbxLog.d("controller.submit is called before controller.startcontroller", true);
            return;
        }

        if (executorService.isShutdown())
            executorService = Executors.newSingleThreadExecutor();

        if (isInitSubmitted && messageType == MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.RESTART_SDK) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.DELETE_USER_DATA_AND_STOP_SDK) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.INSERT_PUSH_DATA) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isOnResumeSubmitted) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (messageType == MessageType.INITIALIZE) {
            executorService.submit(new MessageInvokeRunnable(messageType, args));
            isInitSubmitted = true;
        } else if (messageType == MessageType.ON_RESUME)
            onResumeQueue.offer(new MessageInvokeRunnable(messageType, args));
        else etcQueue.offer(new MessageInvokeRunnable(messageType, args));

        if (isInitSubmitted && !onResumeQueue.isEmpty() && !isOnResumeSubmitted) {
            executorService.submit(onResumeQueue.poll());
            isOnResumeSubmitted = true;
        }

        if (isOnResumeSubmitted) {
            while (!etcQueue.isEmpty())
                executorService.submit(etcQueue.poll());
        }
    }

    private void onRestartCompleted() {
        isInitRestartProcessingNow = false;

        if (abxContext == null || componentsFactory == null || dataRegistry == null) {
            AbxLog.d("controller.onRestartCompleted is called before controller.startcontroller", true);
            return;
        }

        if (abxContext instanceof DisabledABXContext) {
            changeABXContext(
                    new InitializingABXContext(componentsFactory, repository));

            Context context = null;
            try {
                context = componentsFactory.getAndroidContext();
            } catch (IABXComponentsFactory.ComponentsCanNotCreateException e) {
                AbxLog.d(Arrays.toString(e.getStackTrace()), true);
            }
            String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
            String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

            this.initialize(context, appKey, secretKey);

            //session 정보 초기화
            ABXBooleanState.getInstance().isSessionStarted.getAndSet(false);
            //이때 abxContext는 무조건 Active 이다. 이벤트 처리 전 바로 start_session 이벤트를 날리기 위함.
            this.onResume(null);
        }
    }

    public void disableSDK(String disabledReason) {
        if (componentsFactory == null || repository == null) {
            AbxLog.d("controller.disableSDK is called before controller.startcontroller", true);
            return;
        }

        changeABXContext(new DisabledABXContext(disabledReason, repository));
        componentsFactory.deactivate();
    }

    @Override
    public void update(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }

    private enum MessageType {
        CHANGE_ABX_CONTEXT,
        INITIALIZE,
        SAVE_USER_PROPERTY,
        SAVE_USER_PROPERTY_WITHOUT_EVENT,
        LOG_EVENT,
        LOG_SAME_EVENT_WITH_PAGING,
        ON_RESUME,
        ON_PAUSE,
        DEEPLINK,
        DEEPLINK_WITH_INTENT,
        GDPR_FORGET_ME,
        PUT_DATA_REGISTRY,
        RUN_IN_BACKGROUND_IN_ORDER,
        RUN_IN_BACKGROUND_WITHOUT_ORDER,
        DELETE_USER_DATA_AND_STOP_SDK,
        RESTART_SDK,
        FETCH_ACTION_HISTORY_FROM_SERVER,
        INSERT_PUSH_DATA,
        GET_ACTION_HISTORY,
        GET_ALL_ACTION_HISTORY,
        DELETE_ACTION_HISTORY,
        DELETE_ALL_ACTION_HISTORY,
        CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB,
        CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB,
        FETCH_IN_APP_MESSAGE,
        GET_ALL_IN_APP_MESSAGE,
        OPEN_IN_APP_MESSAGE,
        DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS
    }

    private static class Singleton {
        private static final DefaultABXContextController INSTANCE = new DefaultABXContextController();
    }

    private static class Message {
        private MessageType messageType;
        private Object[] args;

        Message(MessageType messageType, Object... args) {
            this.messageType = messageType;
            this.args = args;
        }
    }

    private class MessageInvokeRunnable implements Runnable {
        private Message message;

        MessageInvokeRunnable(MessageType messageType, Object... args) {
            this.message = new Message(messageType, args);
        }

        @Override
        public void run() {
            //메시지 처리
            try {
                if (message == null) {
                    idleTime();
                } else {
                    try {
                        processMessage(message);
                    } catch (BeforeInitializeException be) {
                        AbxLog.w(message.messageType + "\n" +
                                String.format(
                                        LogMessageFormat.DEFAULT_ABX_CONTEXT_BEFORE_INITIALIZE_EXCEPTION,
                                        be.toString()
                                ), true
                        );
                    }
                }
            } catch (Exception e) {
                AbxLog.e(
                        String.format(
                                LogMessageFormat.DEFAULT_ABX_CONTEXT_CONTROLLER_QUEUE_EXCEPTION,
                                message.messageType.toString(),
                                e.toString()
                        ), true
                );
                AbxLog.e(Arrays.toString(e.getStackTrace()), true);
            }
        }

        /**
         * message가 없을때 백그라운드 작업을 수행한다.
         */
        private void idleTime() {
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }

        /**
         * message를 처리한다.
         *
         * @param message
         */
        private void processMessage(Message message) {
            switch (message.messageType) {
                case CHANGE_ABX_CONTEXT:
                    IABXContext newContext = (IABXContext) message.args[0];
                    DefaultABXContextController.this.abxContext = newContext;
                    break;

                case INITIALIZE:
                    Context context = (Context) message.args[0];
                    String appkey = (String) message.args[1];
                    String secretkey = (String) message.args[2];

                    DefaultABXContextController.this.abxContext.initialize(context, appkey, secretkey)
                            .onSuccess(successfulResult -> {
                                changeABXContext(new ActiveABXContext(componentsFactory, repository));
                                AbxLog.d(successfulResult, true);
                            })
                            .onFailure(error -> {
                                disableSDK(error.getMessage());
                                AbxLog.w(error.getMessage(), true);
                            });
                    break;

                case SAVE_USER_PROPERTY:
                    UserPropertyCommand userPropertyCommand1 = (UserPropertyCommand) message.args[0];
                    DefaultABXContextController.this.abxContext.saveUserProperty(userPropertyCommand1);
                    break;

                case SAVE_USER_PROPERTY_WITHOUT_EVENT:
                    UserPropertyCommand userPropertyCommand2 = (UserPropertyCommand) message.args[0];
                    DefaultABXContextController.this.abxContext.saveUserPropertyWithoutEvent(userPropertyCommand2);
                    break;

                case LOG_EVENT:
                    LogEventParameter logEventParameter = (LogEventParameter) message.args[0];

                    DefaultABXContextController.this.abxContext.logEvent(logEventParameter);
                    break;

                case LOG_SAME_EVENT_WITH_PAGING:
                    String eventName = (String) message.args[0];

                    @SuppressWarnings("unchecked")
                    List<JSONObject> eventParamList = (List<JSONObject>) message.args[1];

                    DefaultABXContextController.this.abxContext.logSameEventWithPaging(eventName, eventParamList);
                    break;

                case ON_RESUME:
                    Activity activity = (Activity) message.args[0];
                    DefaultABXContextController.this.abxContext.onResume(activity);
                    break;

                case ON_PAUSE:
                    DefaultABXContextController.this.abxContext.onPause();
                    break;

                case DEEPLINK:
                    Activity deeplinkActivity = (Activity) message.args[0];
                    DefaultABXContextController.this.abxContext.deeplink(deeplinkActivity);
                    break;

                case DEEPLINK_WITH_INTENT:
                    Intent deeplinkIntent = (Intent) message.args[0];
                    DefaultABXContextController.this.abxContext.deeplinkWithIntent(deeplinkIntent);
                    break;

                case GDPR_FORGET_ME:
                    if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) {
                        AbxLog.i("Set GDPR Sync set:: " + CoreUtils.isGdprForgetMeSync(dataRegistry), true);
                        if (CoreUtils.isGdprForgetMeSync(dataRegistry)) {
                            AbxLog.i("Set GDPR is already set the 'true'", true);
                            return;
                        }
                    }

                    new GdprForgetMeUseCase(repository).execute();

                    disableSDK("GDPR FORGET ME");
                    break;

                case PUT_DATA_REGISTRY:
                    DataUnit dataUnit = (DataUnit) message.args[0];
                    DefaultABXContextController.this.abxContext.putDataRegistry(dataUnit);
                    break;

                case RUN_IN_BACKGROUND_IN_ORDER:
                case RUN_IN_BACKGROUND_WITHOUT_ORDER:
                    Runnable runnable = (Runnable) message.args[0];
                    runnable.run();
                    break;

                case DELETE_USER_DATA_AND_STOP_SDK:

                    String deleteUserId = (String) message.args[0];
                    Runnable onDeleteSuccess = (Runnable) message.args[1];
                    Runnable onDeleteFail = (Runnable) message.args[2];

                    if (CoreUtils.isAdbrixPause(dataRegistry) ||
                            CoreUtils.isGdprForgetMe(dataRegistry, DefaultABXContextController.this::gdprForgetMe)) {
                        if (onDeleteFail != null)
                            onDeleteFail.run();

                        AbxLog.d("deleteUserDataAndStopSDK is failed due to (gdpr || pause || stop)", false);
                        return;
                    }

                    CoreUtils.printDRState(dataRegistry);
                    if (CoreUtils.getDRState(dataRegistry) == DRState.DELETE_SYNCED) {
                        AbxLog.d("Delete user data already synced!", true);
                        if (onDeleteSuccess != null)
                            onDeleteSuccess.run();
                        return;
                    }

                    if (isDeleteProcessingNow) {
                        AbxLog.d("Delete API is already processing!", true);
                        return;
                    }

                    isDeleteProcessingNow = true;

                    DefaultABXContextController.this.abxContext.deleteUserData(deleteUserId)
                            .onSuccess(empty -> {
                                if (onDeleteSuccess != null)
                                    onDeleteSuccess.run();
                            })
                            .onFailure(error -> {
                                if (onDeleteFail != null)
                                    onDeleteFail.run();
                            })
                            .onComplete(() -> {
                                //Delete 종료
                                isDeleteProcessingNow = false;
                                //stop sdk.
                                disableSDK("SDK STOPPED");
                            });
                    break;

                case RESTART_SDK:

                    String restartUserId = (String) message.args[0];
                    Runnable onRestartSuccess = (Runnable) message.args[1];
                    Runnable onRestartFail = (Runnable) message.args[2];

                    if (CoreUtils.isAdbrixPause(dataRegistry) ||
                            CoreUtils.isGdprForgetMe(dataRegistry, DefaultABXContextController.this::gdprForgetMe)) {
                        if (onRestartFail != null)
                            onRestartFail.run();

                        AbxLog.d("restartSDK is failed due to (gdpr || pause || stop)", false);
                        return;
                    }

                    CoreUtils.printDRState(dataRegistry);
                    if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_SYNCED) {
                        AbxLog.d("Restart already synced!", true);
                        if (onRestartSuccess != null)
                            onRestartSuccess.run();
                        return;
                    }

                    if (isInitRestartProcessingNow) {
                        AbxLog.d("InitRestart API is already processing!", true);
                        return;
                    }

                    isInitRestartProcessingNow = true;

                    DefaultABXContextController.this.abxContext.restartSDK(restartUserId)
                            .onSuccess(empty -> {
                                if (onRestartSuccess != null)
                                    onRestartSuccess.run();
                            })
                            .onFailure(error -> {
                                if (onRestartFail != null)
                                    onRestartFail.run();
                            })
                            .onComplete(DefaultABXContextController.this::onRestartCompleted);
                    break;

                case FETCH_ACTION_HISTORY_FROM_SERVER: {
                    String token = (String) message.args[0];
                    ActionHistoryIdType idType = (ActionHistoryIdType) message.args[1];
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[3];

                    DefaultABXContextController.this.abxContext.fetchActionHistoryFromServer(token, idType, actionType, completion);
                    break;
                }

                case INSERT_PUSH_DATA:
                    String pushDataString = (String) message.args[0];

                    DefaultABXContextController.this.abxContext.insertPushData(pushDataString);
                    break;

                case GET_ACTION_HISTORY: {
                    int skip = (int) message.args[0];
                    int limit = (int) message.args[1];
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[3];

                    DefaultABXContextController.this.abxContext.getActionHistory(skip, limit, actionType, completion);
                    break;
                }

                case GET_ALL_ACTION_HISTORY: {
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[0];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[1];

                    DefaultABXContextController.this.abxContext.getAllActionHistory(actionType, completion);
                    break;
                }

                case DELETE_ACTION_HISTORY: {
                    String token = (String) message.args[0];
                    String historyId = (String) message.args[1];
                    long timestamp = (long) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[3];

                    DefaultABXContextController.this.abxContext.deleteActionHistory(token, historyId, timestamp, completion);
                    break;
                }

                case DELETE_ALL_ACTION_HISTORY: {
                    String token = (String) message.args[0];
                    ActionHistoryIdType idType = (ActionHistoryIdType) message.args[1];
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[2];

                    DefaultABXContextController.this.abxContext.deleteAllActionHistory(token, idType, completion);
                    break;
                }

                case CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[0];

                    DefaultABXContextController.this.abxContext.clearSyncedActionHistoryInLocalDB(completion);
                    break;
                }

                case CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB:

                    DefaultABXContextController.this.abxContext.clearAllActionHistoryInLocalDB();
                    break;

                case FETCH_IN_APP_MESSAGE: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion =
                            (Completion<Result<Empty>>) message.args[0];

                    DefaultABXContextController.this.abxContext.fetchInAppMessage(completion);
                    break;
                }


                case GET_ALL_IN_APP_MESSAGE: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<DfnInAppMessage>>> completion =
                            (Completion<Result<List<DfnInAppMessage>>>) message.args[0];

                    DefaultABXContextController.this.abxContext.getAllInAppMessage(completion);
                    break;
                }

                case OPEN_IN_APP_MESSAGE: {
                    String campaignId = (String) message.args[0];
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion =
                            (Completion<Result<Empty>>) message.args[1];

                    DefaultABXContextController.this.abxContext.openInAppMessage(campaignId, completion);
                    break;
                }

                case DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS:
                    DefaultABXContextController.this.abxContext.deleteAllInAppMessageDBContents();
                    break;

                default:
                    throw new IllegalStateException("Unexpected value: " + message.messageType);
            }

            //메시지 처리 이후 백그라운드 작업을 수행한다.
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }
    }
}
