package io.adbrix.sdk.component.autoEvent;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DeeplinkParameterSet;
import io.adbrix.sdk.component.DeeplinkPostingObservable;
import io.adbrix.sdk.component.IObservable;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.InAppMessageAutoFetchObservable;
import io.adbrix.sdk.component.PushOpenMonitor;
import io.adbrix.sdk.component.V1MigrationManager;
import io.adbrix.sdk.data.ABXBooleanState;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class SessionAutoEventGenerator implements IObservable<LogEventParameter> {

    private static final String BASIC_OPEN = "basic_open";
    private static final String FIRST_OPEN = "first_open";
    private static final String DEEPLINK_OPEN = "deeplink_open";
    private static final String PUSH_OPEN = "push_open";

    private DataRegistry dataRegistry;
    private FirstOpenAutoEventGenerator firstOpenAutoEventGenerator;

    private String lastCreatedSessionId;
    private String lastEndSessionId;
    private long lastSessionCreatedTime;
    private long lastSessionEndTime;
    private boolean isEndSessionSentLast;
    private Repository repository;
    private DeeplinkParameterSet deeplinkParameterSet;
    private V1MigrationManager v1MigrationManager;
    private List<IObserver<LogEventParameter>> observers;

    private Timer timer;

    public SessionAutoEventGenerator(
            DataRegistry dataRegistry,
            FirstOpenAutoEventGenerator firstOpenAutoEventGenerator,
            Repository repository,
            DeeplinkParameterSet deeplinkParameterSet,
            V1MigrationManager v1MigrationManager
    ) {

        this.dataRegistry = dataRegistry;
        this.firstOpenAutoEventGenerator = firstOpenAutoEventGenerator;
        this.repository = repository;
        this.deeplinkParameterSet = deeplinkParameterSet;
        this.v1MigrationManager = v1MigrationManager;
        observers = new ArrayList<>();

        lastCreatedSessionId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_CREATED_SESSION_ID, null);
        lastEndSessionId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_END_SESSION_ID, null);
        lastSessionCreatedTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME, 0);
        lastSessionEndTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_LAST_SESSION_END_TIME, 0);
        isEndSessionSentLast = dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST, false);

        timer = null;
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }

    public void onResume(final Activity activity) {
        if (timer != null) {
            // Cancel the app background event from being sent,
            // it hasn't been enough time between previous Session end and this start
            cancelEndSessionSchedule();
            return;
        }

        if (ABXBooleanState.getInstance().isSessionStarted())
            return;

        if (!CommonUtils.isNullOrEmpty(lastCreatedSessionId) && // First StartSession
                lastCreatedSessionId != lastEndSessionId) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_END_SESSION,
                    null,
                    0,
                    getSessionLength()
            );

            processEndSession(eventParameter);
        }

        if (firstOpenAutoEventGenerator.isTrigger()) {

            if (v1MigrationManager.isMigrationCompleted()) {
                processStartSession(BASIC_OPEN, null);
                return;
            }

            Map<String, Object> referrerMap = firstOpenAutoEventGenerator.getGooglePlayReferrerPropertiesOrNull();

            AbxLog.d("FirstOpen start", true);

            processStartSession(FIRST_OPEN, referrerMap);

            AbxLog.d("FirstOpen finished successfully", true);
            firstOpenAutoEventGenerator.runDeferredDeeplinkProcess();

            CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry);

        } else if (CoreUtils.isActivityStartedByDeeplink(activity, dataRegistry)) {
            Uri uri = CoreUtils.getDeeplinkUriFromCurrentActivity(activity);

            if (uri == null) {
                AbxLog.d("deeplink uri is null", true);
                processStartSession(BASIC_OPEN, null);
                return;
            }

            String deeplinkUrl = uri.toString();

            Map<String, Object> deeplinkEventParam = CoreUtils.getDeeplinkParameterFromUriString(deeplinkUrl);

            if (deeplinkParameterSet.isAlreadyUsedDeeplinkParameter(deeplinkEventParam)) {
                processStartSession(BASIC_OPEN, null);
                return;
            }

            processStartSession(DEEPLINK_OPEN, deeplinkEventParam);

            DeeplinkPostingObservable.getInstance().notifyObserver(deeplinkUrl);

            CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry);
        } else if (PushOpenMonitor.getInstance().isOpenedByPush()) {
            Map<String, Object> pushOpenEventParam = PushOpenMonitor.getInstance().getPushOpenEventParam();

            processStartSession(PUSH_OPEN, pushOpenEventParam);

            PushOpenMonitor.getInstance().refresh();
        } else {
            processStartSession(BASIC_OPEN, null);
        }
    }

    // Schedule timer to send an end_session event after SESSION_TIMEOUT ms have passed
    public void onPause() {
        cancelEndSessionSchedule();

        long currentTimeMillis = System.currentTimeMillis();
        updateLastSessionEndTime(currentTimeMillis);

        String eventId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);
        String eventDateTime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);

        LogEventParameter eventParameter = new LogEventParameter(
                eventId,
                null,
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_END_SESSION,
                null,
                0,
                getSessionLength(),
                eventDateTime
        );

        timer = new Timer();
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                // Null out the timer so we know it's finished running
                if (timer != null) {
                    timer = null;
                    processEndSession(eventParameter);
                    dataRegistry.saveRegistry();
                }
            }
        }, CoreConstants.SESSION_TIMEOUT);
    }

    public void ping() {
        // updateElapsedTimeInForeground(SystemClock.elapsedRealtime());
    }

    public boolean isStartSessionByDeeplink(Activity deeplinkActivity) {
        return CoreUtils.isActivityStartedByDeeplink(deeplinkActivity, dataRegistry) && !ABXBooleanState.getInstance().isSessionStarted();
    }

    public boolean isStartSessionByDeeplinkIntent(Intent deeplinkIntent) {
        return CoreUtils.isStartedByDeeplinkIntent(deeplinkIntent, dataRegistry) && !ABXBooleanState.getInstance().isSessionStarted();
    }

    //Used for Start Session Event only
    private long getSessionInterval() {
        return lastSessionEndTime == 0 ? 0 : System.currentTimeMillis() - lastSessionEndTime;
    }

    //Used for EndSession Event only
    private long getSessionLength() {
        return lastSessionEndTime - lastSessionCreatedTime;
    }

    private void processStartSession(String openType, Map<String, Object> eventParam) {

        if (ABXBooleanState.getInstance().isSessionStarted()) {
            return;
        }

        updateLastCreatedSessionId(CommonUtils.randomUUIDWithCurrentTime());
        updateLastSessionCreatedTime(System.currentTimeMillis());

        //InAppMessage
        repository.resetInAppMessageFrequencySession();

        int minutesToExpiry = dataRegistry.safeGetInt(DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY, -1);
        long inAppMessageLastResponseTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME, -1);

        if (minutesToExpiry == -1 || inAppMessageLastResponseTime == -1 //first inAppMessage request
                || inAppMessageLastResponseTime + minutesToExpiry * 60 * 1000 < System.currentTimeMillis()//cache expired
        ) {
            repository.fetchInAppMessage(result -> {
                AbxLog.d("SessionAutoEventGenerator fetchInAppMessage completed: " + result, true);
                InAppMessageAutoFetchObservable.getInstance().notifyObserver(result);
            });
        }
        //InAppMessage end

        if (eventParam == null) {
            eventParam = new HashMap<>();
        }
        eventParam.put("abx:open_type", openType);

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_START_SESSION,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                getSessionInterval(),
                0
        );

        if (openType.equals(FIRST_OPEN)) {
            //DataRegistry에 last_first_open_event_id 설정
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            //DataRegistry에 last_open_id 설정
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            getClass().getName(),
                            true
                    ));
        } else if (openType.equals(DEEPLINK_OPEN)) {
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));
        }

        //start_session 시 session_order_no 초기화
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_SESSION_ORDER_NO,
                        0L,
                        5,
                        this.getClass().getName(),
                        true
                ));

        notifyObserver(eventParameter);
        ABXBooleanState.getInstance().isSessionStarted.getAndSet(true);

        isEndSessionSentLast = false;

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                false,
                5,
                this.getClass().getName(),
                true
        ));
    }

    private void processEndSession(LogEventParameter eventParameter) {
        if (isEndSessionSentLast)
            return;

        updateLastEndSessionId(lastCreatedSessionId);

        notifyObserver(eventParameter);
        ABXBooleanState.getInstance().isSessionStarted.getAndSet(false);

        isEndSessionSentLast = true;

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                true,
                5,
                this.getClass().getName(),
                true
        ));
    }

    public void cancelEndSessionSchedule() {
        if (timer != null) {
            timer.cancel();
            timer = null;
        }
    }

    private void updateLastCreatedSessionId(String id) {
        lastCreatedSessionId = id;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_CREATED_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastEndSessionId(String id) {
        lastEndSessionId = id;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_END_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionCreatedTime(long time) {

        lastSessionCreatedTime = time;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionEndTime(long time) {
        lastSessionEndTime = time;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_END_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }
}
