package io.adbrix.sdk.component;

import java.util.ArrayList;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventPackage;

public class EventUploadIntervalManager implements IObservable<Void>, ITimerManager {
    private int countInterval = 30;
    private int timeInterval = 60_000;
    private Timer eventBusTimer;
    private ArrayList<IObserver<Void>> observerArrayList = new ArrayList<>();

    private EventUploadIntervalManager() {
        TimerManager.getInstance().add(this);
    }

    public static EventUploadIntervalManager getInstance() {
        return Singleton.INSTANCE;
    }

    public void setCountInterval(int countInterval) {
        this.countInterval = countInterval;
    }

    public void setTimeInterval(int timeInterval) {
        this.timeInterval = timeInterval;
        setTimerToNull();
        scheduleEventSendingJob();
    }

    public void manageEventUploadInterval(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        AbxLog.d("EventQueue size : " + packageQueue.size(), true);
        scheduleEventSendingJob();
        countEventSendingJob(eventSendingTask, packageQueue);
    }

    //multi thread에서 timer에 동시 접근하는 것을 막기 위해 synchronized 처리
    //https://iga-dev.atlassian.net/wiki/spaces/DFNDEV/pages/2729738241/2022-1-21+Android+SDK+2.1.2.5+QA+Guide
    private synchronized void scheduleEventSendingJob() {
        if (eventBusTimer == null) {
            eventBusTimer = new Timer();

            eventBusTimer.schedule(new TimerTask() {
                @Override
                public void run() {
                    AbxLog.d("Trying to upload events every " + timeInterval / 1000 + " seconds.", true);
                    notifyObserver(null);
                }
            }, this.timeInterval, this.timeInterval);
        }
    }

    private void countEventSendingJob(Runnable eventSendingTask, Queue<EventPackage> packageQueue) {
        if (packageQueue.peek() == null)
            return;

        if (packageQueue.size() >= (countInterval / CoreConstants.EVENT_BUFFER_COUNT)) {
            AbxLog.d("Trying to upload " + countInterval + " events.", true);
            eventSendingTask.run();
        }
    }

    @Override
    public void add(IObserver<Void> observer) {
        observerArrayList.add(observer);
    }

    @Override
    public void delete(IObserver<Void> observer) {
        observerArrayList.remove(observer);
    }

    @Override
    public void notifyObserver(Void v) {
        for (IObserver<Void> observer : observerArrayList) {
            observer.update(v);
        }
    }

    private static class Singleton {
        private static final EventUploadIntervalManager INSTANCE = new EventUploadIntervalManager();
    }

    //multi thread에서 timer에 동시 접근하는 것을 막기 위해 synchronized 처리
    //https://iga-dev.atlassian.net/wiki/spaces/DFNDEV/pages/2729738241/2022-1-21+Android+SDK+2.1.2.5+QA+Guide
    @Override
    public synchronized void setTimerToNull() {
        if (eventBusTimer != null) {
            eventBusTimer.cancel();
            eventBusTimer = null;
            observerArrayList.clear();
        }
    }
}
