package io.adbrix.sdk.component;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.data.dataprovider.INoWaitEventNameProvider;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;


public class EventBuffer implements IObserver<Void> {

    private int bufferCapacity = CoreConstants.EVENT_BUFFER_COUNT;
    private Queue<EventModel> currentBuffer;
    private EventPackageContainer eventPackageContainer;
    private INoWaitEventNameProvider noWaitEventNameProvider;

    public EventBuffer(
            EventPackageContainer eventPackageContainer,
            INoWaitEventNameProvider noWaitEventNameProvider
    ) {
        this.eventPackageContainer = eventPackageContainer;
        this.noWaitEventNameProvider = noWaitEventNameProvider;
        this.currentBuffer = new ConcurrentLinkedQueue<>();
        EventUploadIntervalManager.getInstance().add(this);
    }

    public synchronized void addEventModel(EventModel model) {
        Queue<EventModel> buffer = getCurrentBuffer();
        buffer.add(model);

        if (noWaitEventNameProvider.isNowaitEvent(model.eventName))
            clearBufferAndFlushContainer();
        else
            addEventPackageToContainerIfBufferFull();
    }

    public void flushAllNow() {
        clearBufferAndFlushContainer();
    }

    public void flushContainerAtIntervals() {
        this.eventPackageContainer.flushAtIntervals();
    }

    @Override
    public void update(Void v) {
        flushAllNow();
    }

    public void saveUnsentEvents() {
        if (this.currentBuffer == null) {
            this.currentBuffer = new ConcurrentLinkedQueue<>();
        }

        if (currentBuffer.size() > 0) {
            eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
            this.currentBuffer = new ConcurrentLinkedQueue<>();
        } else {
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
        }
    }

    private Queue<EventModel> getCurrentBuffer() {
        if (this.currentBuffer == null) {
            this.currentBuffer = new ConcurrentLinkedQueue<>();
            return currentBuffer;
        }

        addEventPackageToContainerIfBufferFull();

        return this.currentBuffer;
    }

    private synchronized void clearBufferAndFlushContainer() {
        if (currentBuffer == null)
            return;

        synchronized (currentBuffer) {
            //onNoWaitEvent
            if (currentBuffer.size() > 0) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                eventPackageContainer.flushEventsNotSent();
                this.currentBuffer = new ConcurrentLinkedQueue<>();
            } else {
                eventPackageContainer.flushEventsNotSent();
            }
        }
    }

    private void addEventPackageToContainerIfBufferFull() {
        if (currentBuffer == null)
            return;

        synchronized (currentBuffer) {
            //onBufferFull
            if (currentBuffer.size() >= bufferCapacity) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                this.currentBuffer = new ConcurrentLinkedQueue<>();
            }
        }
    }

    public static class BufferCapacityLimitsException extends Exception {
        BufferCapacityLimitsException(String message) {
            super(message);
        }
    }
}
